// META: title=test WebNN API reshape operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-reshape-method
// Alter the shape of a tensor to a new shape.
//
// MLOperand reshape(
//     MLOperand input, sequence<[EnforceRange] unsigned long> newShape);

const reshapeTests = [
  {
    'name': 'reshape float32 tensor to a new shape (reorder all dimensions)',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 2, 3]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'reshape float32 constant tensor to a new shape (reorder all dimensions)',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 2, 3]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape float32 tensor to a new shape (reduce dimensions)',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 1, 1, 1, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 1, 1, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 1, 1, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape float32 tensor to a new shape (extend dimensions)',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [2, 2, 2, 3, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 2, 2, 3, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape float32 tensor to a new shape (4D to 4D)',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [3, 2, 2, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 2, 3, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 2, 3, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape float32 tensor to 1D tensor',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [3, 2, 2, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float32 2D tensor by eliminating one dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float32 3D tensor by eliminating one dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 1, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float32 3D tensor by eliminating two dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 24, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float32 4D tensor by eliminating two dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 4, 1, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float32 4D tensor by eliminating all dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [-33.82555389404297],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': []}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [-33.82555389404297],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'reshape (squeeze) float32 5D tensor by eliminating four dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 1, 1, 24, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float32 2D tensor by eliminating 1st dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'reshape (squeeze) float32 3D tensor by eliminating 2nd and 3rd dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24, 1, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'reshape (squeeze) float32 4D tensor by eliminating 1st and 4th dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 4, 6, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'reshape (squeeze) float32 5D tensor by eliminating 2nd and 3rd dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 1, 1, 12, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [2, 12, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 12, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'reshape (squeeze) float32 5D tensor by eliminating 1st, 2nd and 5th dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 1, 1, 24, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float32 0D tensor to 4D',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [-33.82555389404297],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 1, 1, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [-33.82555389404297],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float32 1D tensor by adding one dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float32 1D tensor by adding two dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 24, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 24, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float32 1D tensor to 5D',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments':
            [{'input': 'reshapeInput'}, {'newShape': [1, 1, 1, 24, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 1, 1, 24, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'reshape (unsqueeze) float32 1D tensor by adding 2nd and 3rd dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24, 1, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float32 2D tensor by adding one dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 1, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 1, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float32 2D tensor by adding two dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 4, 1, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 4, 1, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float32 2D tensor by adding 1st dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'reshape (unsqueeze) float32 2D tensor by adding 1st and 4th dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 4, 6, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 4, 6, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'reshape (unsqueeze) float32 3D tensor by adding 2nd and 3rd dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 12, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments':
            [{'input': 'reshapeInput'}, {'newShape': [2, 1, 1, 12, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 1, 1, 12, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float32 4D tensor by adding 2nd dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [2, 1, 2, 2, 3]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 1, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float32 5D tensor by adding 4th dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 1, 4, 3, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments':
            [{'input': 'reshapeInput'}, {'newShape': [2, 1, 4, 1, 3, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 1, 4, 1, 3, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (flatten) float32 3D tensor to 2D',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [2, 12]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 12], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (flatten) float32 4D to 2D',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (flatten) float32 4D to 2D exclusive 1st dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [1, 24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'reshape (flatten) float32 4D to 2D exclusive 4th dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0561466217041,  99.56941986083984,   88.04620361328125,
            -91.87507629394531, -23.7972354888916,   -91.28665161132812,
            -63.15204620361328, 12.0669527053833,    -96.1172866821289,
            -44.77365493774414, -80.08650970458984,  -64.43756866455078,
            27.64195442199707,  -96.86306762695312,  83.6834716796875,
            50.599483489990234, -20.18765640258789,  -1.3904608488082886,
            -96.93603515625,    65.34143829345703,   34.835994720458984,
            62.01485824584961,  -2.8698415756225586, 27.903749465942383
          ],
          'descriptor': {shape: [24, 1], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'reshape float16 tensor to a new shape (reorder all dimensions)',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 2, 3]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'reshape float16 constant tensor to a new shape (reorder all dimensions)',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 2, 3]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape float16 tensor to a new shape (reduce dimensions)',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 1, 1, 1, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 1, 1, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 1, 1, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape float16 tensor to a new shape (extend dimensions)',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [2, 2, 2, 3, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 2, 2, 3, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape float16 tensor to a new shape (4D to 4D)',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [3, 2, 2, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 2, 3, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 2, 3, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape float16 tensor to 1D tensor',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [3, 2, 2, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float16 2D tensor by eliminating one dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float16 3D tensor by eliminating one dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 1, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float16 3D tensor by eliminating two dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 24, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float16 4D tensor by eliminating two dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 4, 1, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float16 4D tensor by eliminating all dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [-33.8125],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': []}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput':
            {'data': [-33.8125], 'descriptor': {shape: [], dataType: 'float16'}}
      }
    }
  },
  {
    'name':
        'reshape (squeeze) float16 5D tensor by eliminating four dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 1, 1, 24, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (squeeze) float16 2D tensor by eliminating 1st dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'reshape (squeeze) float16 3D tensor by eliminating 2nd and 3rd dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24, 1, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'reshape (squeeze) float16 4D tensor by eliminating 1st and 4th dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 4, 6, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'reshape (squeeze) float16 5D tensor by eliminating 2nd and 3rd dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 1, 1, 12, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [2, 12, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 12, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'reshape (squeeze) float16 5D tensor by eliminating 1st, 2nd and 5th dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 1, 1, 24, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float16 0D tensor to 4D',
    'graph': {
      'inputs': {
        'reshapeInput':
            {'data': [-33.8125], 'descriptor': {shape: [], dataType: 'float16'}}
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 1, 1, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [-33.8125],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float16 1D tensor by adding one dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float16 1D tensor by adding two dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 24, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 24, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float16 1D tensor to 5D',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments':
            [{'input': 'reshapeInput'}, {'newShape': [1, 1, 1, 24, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 1, 1, 24, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'reshape (unsqueeze) float16 1D tensor by adding 2nd and 3rd dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24, 1, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float16 2D tensor by adding one dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 1, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 1, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float16 2D tensor by adding two dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 4, 1, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 4, 1, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float16 2D tensor by adding 1st dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'reshape (unsqueeze) float16 2D tensor by adding 1st and 4th dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 4, 6, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 4, 6, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'reshape (unsqueeze) float16 3D tensor by adding 2nd and 3rd dimensions',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 12, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments':
            [{'input': 'reshapeInput'}, {'newShape': [2, 1, 1, 12, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 1, 1, 12, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float16 4D tensor by adding 2nd dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [2, 1, 2, 2, 3]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 1, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (unsqueeze) float16 5D tensor by adding 4th dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 1, 4, 3, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments':
            [{'input': 'reshapeInput'}, {'newShape': [2, 1, 4, 1, 3, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 1, 4, 1, 3, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (flatten) float16 3D tensor to 2D',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [2, 12]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 12], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (flatten) float16 4D to 2D',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [4, 6]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (flatten) float16 4D to 2D exclusive 1st dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [1, 24]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [1, 24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'reshape (flatten) float16 4D to 2D exclusive 4th dimension',
    'graph': {
      'inputs': {
        'reshapeInput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'reshape',
        'arguments': [{'input': 'reshapeInput'}, {'newShape': [24, 1]}],
        'outputs': 'reshapeOutput'
      }],
      'expectedOutputs': {
        'reshapeOutput': {
          'data': [
            -30.0625,  99.5625,    88.0625,  -91.875,   -23.796875,   -91.3125,
            -63.15625, 12.0703125, -96.125,  -44.78125, -80.0625,     -64.4375,
            27.640625, -96.875,    83.6875,  50.59375,  -20.1875,     -1.390625,
            -96.9375,  65.3125,    34.84375, 62,        -2.869140625, 27.90625
          ],
          'descriptor': {shape: [24, 1], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(reshapeTests, buildAndExecuteGraph, getZeroULPTolerance);
