// META: title=test WebNN API prelu operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-prelu
// Calculate the parametric version of rectified linear function (Parametric
// ReLU) on the input tensor element-wise. The calculation follows the
// expression max(0, x) + slope * min(0, x).
//
// MLOperand prelu(MLOperand input, MLOperand slope);

const preluTests = [
  {
    'name': 'prelu float32 0D scalar',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [-4.794857501983643],
          'descriptor': {shape: [], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [1.1202747821807861],
          'descriptor': {shape: [], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [-5.371557712554932],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 1D constant tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        },
        'preluSlope': {
          'data': [
            9.343092918395996,  0.2800687253475189,  -4.617084980010986,
            1.1202747821807861, -1.4334710836410522, -3.157594919204712,
            -6.28995418548584,  -5.0107879638671875, -6.899077415466309,
            3.5725347995758057, 6.861966609954834,   -1.961531400680542,
            4.5832037925720215, 2.6643502712249756,  9.192955017089844,
            -9.554699897766113, -5.505102157592773,  -2.3927369117736816,
            3.58212947845459,   -2.3224003314971924, -1.9816573858261108,
            4.155889987945557,  -1.799522042274475,  9.295849800109863
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.817113876342773, -1.342889666557312,  8.413617134094238,
            6.108623504638672,   12.173455238342285,  3.3143365383148193,
            1.1687211990356445,  0.7103435397148132,  46.32490539550781,
            5.787421703338623,   -25.7709903717041,   9.608142852783203,
            7.3295159339904785,  -10.535453796386719, 7.067296981811523,
            9.439736366271973,   14.083043098449707,  20.718313217163086,
            8.47507381439209,    4.551425457000732,   18.365745544433594,
            -1.0895805358886719, 1.3258955478668213,  -68.95950317382812
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 1D tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [
            9.343092918395996,  0.2800687253475189,  -4.617084980010986,
            1.1202747821807861, -1.4334710836410522, -3.157594919204712,
            -6.28995418548584,  -5.0107879638671875, -6.899077415466309,
            3.5725347995758057, 6.861966609954834,   -1.961531400680542,
            4.5832037925720215, 2.6643502712249756,  9.192955017089844,
            -9.554699897766113, -5.505102157592773,  -2.3927369117736816,
            3.58212947845459,   -2.3224003314971924, -1.9816573858261108,
            4.155889987945557,  -1.799522042274475,  9.295849800109863
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.817113876342773, -1.342889666557312,  8.413617134094238,
            6.108623504638672,   12.173455238342285,  3.3143365383148193,
            1.1687211990356445,  0.7103435397148132,  46.32490539550781,
            5.787421703338623,   -25.7709903717041,   9.608142852783203,
            7.3295159339904785,  -10.535453796386719, 7.067296981811523,
            9.439736366271973,   14.083043098449707,  20.718313217163086,
            8.47507381439209,    4.551425457000732,   18.365745544433594,
            -1.0895805358886719, 1.3258955478668213,  -68.95950317382812
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 1D non-constant slope',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [
            9.343092918395996,  0.2800687253475189,  -4.617084980010986,
            1.1202747821807861, -1.4334710836410522, -3.157594919204712,
            -6.28995418548584,  -5.0107879638671875, -6.899077415466309,
            3.5725347995758057, 6.861966609954834,   -1.961531400680542,
            4.5832037925720215, 2.6643502712249756,  9.192955017089844,
            -9.554699897766113, -5.505102157592773,  -2.3927369117736816,
            3.58212947845459,   -2.3224003314971924, -1.9816573858261108,
            4.155889987945557,  -1.799522042274475,  9.295849800109863
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.817113876342773, -1.342889666557312,  8.413617134094238,
            6.108623504638672,   12.173455238342285,  3.3143365383148193,
            1.1687211990356445,  0.7103435397148132,  46.32490539550781,
            5.787421703338623,   -25.7709903717041,   9.608142852783203,
            7.3295159339904785,  -10.535453796386719, 7.067296981811523,
            9.439736366271973,   14.083043098449707,  20.718313217163086,
            8.47507381439209,    4.551425457000732,   18.365745544433594,
            -1.0895805358886719, 1.3258955478668213,  -68.95950317382812
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 2D tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [
            9.343092918395996,  0.2800687253475189,  -4.617084980010986,
            1.1202747821807861, -1.4334710836410522, -3.157594919204712,
            -6.28995418548584,  -5.0107879638671875, -6.899077415466309,
            3.5725347995758057, 6.861966609954834,   -1.961531400680542,
            4.5832037925720215, 2.6643502712249756,  9.192955017089844,
            -9.554699897766113, -5.505102157592773,  -2.3927369117736816,
            3.58212947845459,   -2.3224003314971924, -1.9816573858261108,
            4.155889987945557,  -1.799522042274475,  9.295849800109863
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.817113876342773, -1.342889666557312,  8.413617134094238,
            6.108623504638672,   12.173455238342285,  3.3143365383148193,
            1.1687211990356445,  0.7103435397148132,  46.32490539550781,
            5.787421703338623,   -25.7709903717041,   9.608142852783203,
            7.3295159339904785,  -10.535453796386719, 7.067296981811523,
            9.439736366271973,   14.083043098449707,  20.718313217163086,
            8.47507381439209,    4.551425457000732,   18.365745544433594,
            -1.0895805358886719, 1.3258955478668213,  -68.95950317382812
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 3D tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [
            9.343092918395996,  0.2800687253475189,  -4.617084980010986,
            1.1202747821807861, -1.4334710836410522, -3.157594919204712,
            -6.28995418548584,  -5.0107879638671875, -6.899077415466309,
            3.5725347995758057, 6.861966609954834,   -1.961531400680542,
            4.5832037925720215, 2.6643502712249756,  9.192955017089844,
            -9.554699897766113, -5.505102157592773,  -2.3927369117736816,
            3.58212947845459,   -2.3224003314971924, -1.9816573858261108,
            4.155889987945557,  -1.799522042274475,  9.295849800109863
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.817113876342773, -1.342889666557312,  8.413617134094238,
            6.108623504638672,   12.173455238342285,  3.3143365383148193,
            1.1687211990356445,  0.7103435397148132,  46.32490539550781,
            5.787421703338623,   -25.7709903717041,   9.608142852783203,
            7.3295159339904785,  -10.535453796386719, 7.067296981811523,
            9.439736366271973,   14.083043098449707,  20.718313217163086,
            8.47507381439209,    4.551425457000732,   18.365745544433594,
            -1.0895805358886719, 1.3258955478668213,  -68.95950317382812
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 4D tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [
            9.343092918395996,  0.2800687253475189,  -4.617084980010986,
            1.1202747821807861, -1.4334710836410522, -3.157594919204712,
            -6.28995418548584,  -5.0107879638671875, -6.899077415466309,
            3.5725347995758057, 6.861966609954834,   -1.961531400680542,
            4.5832037925720215, 2.6643502712249756,  9.192955017089844,
            -9.554699897766113, -5.505102157592773,  -2.3927369117736816,
            3.58212947845459,   -2.3224003314971924, -1.9816573858261108,
            4.155889987945557,  -1.799522042274475,  9.295849800109863
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.817113876342773, -1.342889666557312,  8.413617134094238,
            6.108623504638672,   12.173455238342285,  3.3143365383148193,
            1.1687211990356445,  0.7103435397148132,  46.32490539550781,
            5.787421703338623,   -25.7709903717041,   9.608142852783203,
            7.3295159339904785,  -10.535453796386719, 7.067296981811523,
            9.439736366271973,   14.083043098449707,  20.718313217163086,
            8.47507381439209,    4.551425457000732,   18.365745544433594,
            -1.0895805358886719, 1.3258955478668213,  -68.95950317382812
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 5D tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [
            9.343092918395996,  0.2800687253475189,  -4.617084980010986,
            1.1202747821807861, -1.4334710836410522, -3.157594919204712,
            -6.28995418548584,  -5.0107879638671875, -6.899077415466309,
            3.5725347995758057, 6.861966609954834,   -1.961531400680542,
            4.5832037925720215, 2.6643502712249756,  9.192955017089844,
            -9.554699897766113, -5.505102157592773,  -2.3927369117736816,
            3.58212947845459,   -2.3224003314971924, -1.9816573858261108,
            4.155889987945557,  -1.799522042274475,  9.295849800109863
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.817113876342773, -1.342889666557312,  8.413617134094238,
            6.108623504638672,   12.173455238342285,  3.3143365383148193,
            1.1687211990356445,  0.7103435397148132,  46.32490539550781,
            5.787421703338623,   -25.7709903717041,   9.608142852783203,
            7.3295159339904785,  -10.535453796386719, 7.067296981811523,
            9.439736366271973,   14.083043098449707,  20.718313217163086,
            8.47507381439209,    4.551425457000732,   18.365745544433594,
            -1.0895805358886719, 1.3258955478668213,  -68.95950317382812
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 broadcast 4D x 1D slope',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [5.073923110961914, 0.480774462223053, -7.091750144958496],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -12.934283256530762, -2.3052449226379395,  8.413617134094238,
            6.108623504638672,   -4.082877159118652,   3.3143365383148193,
            1.1687211990356445,  -0.06815595179796219, 47.61863327026367,
            5.787421703338623,   -1.8056097030639648,  34.737422943115234,
            7.3295159339904785,  -1.901092767715454,   7.067296981811523,
            9.439736366271973,   -1.2299076318740845,  61.40629196166992,
            8.47507381439209,    4.551425457000732,    65.72542572021484,
            -1.330268144607544,  1.3258955478668213,   52.60881042480469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 broadcast 4D x 1D slope of shape [1]',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [5.0114545822143555],
          'descriptor': {shape: [1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -12.775040626525879, -24.029211044311523, 8.413617134094238,
            6.108623504638672,   -42.558738708496094, 3.3143365383148193,
            1.1687211990356445,  -0.7104380130767822, -33.65017318725586,
            5.787421703338623,   -18.821155548095703, -24.54753875732422,
            7.3295159339904785,  -19.816442489624023, 7.067296981811523,
            9.439736366271973,   -12.82020378112793,  -43.39335632324219,
            8.47507381439209,    4.551425457000732,   -46.44551467895508,
            -1.3138903379440308, 1.3258955478668213,  -37.17652893066406
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 broadcast 4D x 2D slope',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [
            4.874276161193848, -8.501633644104004, 1.1819270849227905,
            -9.985190391540527, -4.424202919006348, -6.654683589935303
          ],
          'descriptor': {shape: [2, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -12.425349235534668, 40.764122009277344, 8.413617134094238,
            6.108623504638672,   37.571624755859375, 3.3143365383148193,
            1.1687211990356445,  1.2052156925201416, -7.936229228973389,
            5.787421703338623,   16.615657806396484, 32.5965461730957,
            7.3295159339904785,  33.61741256713867,  7.067296981811523,
            9.439736366271973,   11.31790828704834,  57.621803283691406,
            8.47507381439209,    4.551425457000732,  -10.953948020935059,
            2.617891550064087,   1.3258955478668213, 49.366512298583984
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 broadcast 4D x 3D slope',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [5.073923110961914, 0.480774462223053, -7.091750144958496],
          'descriptor': {shape: [1, 1, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -12.934283256530762, -2.3052449226379395,  8.413617134094238,
            6.108623504638672,   -4.082877159118652,   3.3143365383148193,
            1.1687211990356445,  -0.06815595179796219, 47.61863327026367,
            5.787421703338623,   -1.8056097030639648,  34.737422943115234,
            7.3295159339904785,  -1.901092767715454,   7.067296981811523,
            9.439736366271973,   -1.2299076318740845,  61.40629196166992,
            8.47507381439209,    4.551425457000732,    65.72542572021484,
            -1.330268144607544,  1.3258955478668213,   52.60881042480469
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'prelu float32 broadcast 4D x 4D slope',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.549168109893799, -4.794857501983643,  8.413617134094238,
            6.108623504638672,  -8.492292404174805,  3.3143365383148193,
            1.1687211990356445, -0.141762837767601,  -6.714652061462402,
            5.787421703338623,  -3.755627393722534,  -4.89828634262085,
            7.3295159339904785, -3.9542298316955566, 7.067296981811523,
            9.439736366271973,  -2.558180093765259,  -8.658834457397461,
            8.47507381439209,   4.551425457000732,   -9.267870903015137,
            -0.262177437543869, 1.3258955478668213,  -7.41831111907959
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'preluSlope': {
          'data': [5.0114545822143555, 5.0114545822143555],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -12.775040626525879, -24.029211044311523, 8.413617134094238,
            6.108623504638672,   -42.558738708496094, 3.3143365383148193,
            1.1687211990356445,  -0.7104380130767822, -33.65017318725586,
            5.787421703338623,   -18.821155548095703, -24.54753875732422,
            7.3295159339904785,  -19.816442489624023, 7.067296981811523,
            9.439736366271973,   -12.82020378112793,  -43.39335632324219,
            8.47507381439209,    4.551425457000732,   -46.44551467895508,
            -1.3138903379440308, 1.3258955478668213,  -37.17652893066406
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'prelu float16 0D scalar',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [-4.79296875],
          'descriptor': {shape: [], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [1.1201171875],
          'descriptor': {shape: [], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [-5.3671875],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 1D constant tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [
            9.34375,       0.280029296875, -4.6171875,    1.1201171875,
            -1.43359375,   -3.158203125,   -6.2890625,    -5.01171875,
            -6.8984375,    3.572265625,    6.86328125,    -1.9619140625,
            4.58203125,    2.6640625,      9.1953125,     -9.5546875,
            -5.50390625,   -2.392578125,   3.58203125,    -2.322265625,
            -1.9814453125, 4.15625,        -1.7998046875, 9.296875
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.8125,    -1.341796875, 8.4140625,     6.109375,    12.171875,
            3.314453125, 1.1689453125, 0.71044921875, 46.3125,     5.7890625,
            -25.78125,   9.609375,     7.328125,      -10.5390625, 7.06640625,
            9.4375,      14.0859375,   20.703125,     8.4765625,   4.55078125,
            18.359375,   -1.08984375,  1.326171875,   -68.9375
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 1D tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [
            9.34375,       0.280029296875, -4.6171875,    1.1201171875,
            -1.43359375,   -3.158203125,   -6.2890625,    -5.01171875,
            -6.8984375,    3.572265625,    6.86328125,    -1.9619140625,
            4.58203125,    2.6640625,      9.1953125,     -9.5546875,
            -5.50390625,   -2.392578125,   3.58203125,    -2.322265625,
            -1.9814453125, 4.15625,        -1.7998046875, 9.296875
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.8125,    -1.341796875, 8.4140625,     6.109375,    12.171875,
            3.314453125, 1.1689453125, 0.71044921875, 46.3125,     5.7890625,
            -25.78125,   9.609375,     7.328125,      -10.5390625, 7.06640625,
            9.4375,      14.0859375,   20.703125,     8.4765625,   4.55078125,
            18.359375,   -1.08984375,  1.326171875,   -68.9375
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 1D non-constant slope',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [
            9.34375,       0.280029296875, -4.6171875,    1.1201171875,
            -1.43359375,   -3.158203125,   -6.2890625,    -5.01171875,
            -6.8984375,    3.572265625,    6.86328125,    -1.9619140625,
            4.58203125,    2.6640625,      9.1953125,     -9.5546875,
            -5.50390625,   -2.392578125,   3.58203125,    -2.322265625,
            -1.9814453125, 4.15625,        -1.7998046875, 9.296875
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.8125,    -1.341796875, 8.4140625,     6.109375,    12.171875,
            3.314453125, 1.1689453125, 0.71044921875, 46.3125,     5.7890625,
            -25.78125,   9.609375,     7.328125,      -10.5390625, 7.06640625,
            9.4375,      14.0859375,   20.703125,     8.4765625,   4.55078125,
            18.359375,   -1.08984375,  1.326171875,   -68.9375
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 2D tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [
            9.34375,       0.280029296875, -4.6171875,    1.1201171875,
            -1.43359375,   -3.158203125,   -6.2890625,    -5.01171875,
            -6.8984375,    3.572265625,    6.86328125,    -1.9619140625,
            4.58203125,    2.6640625,      9.1953125,     -9.5546875,
            -5.50390625,   -2.392578125,   3.58203125,    -2.322265625,
            -1.9814453125, 4.15625,        -1.7998046875, 9.296875
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.8125,    -1.341796875, 8.4140625,     6.109375,    12.171875,
            3.314453125, 1.1689453125, 0.71044921875, 46.3125,     5.7890625,
            -25.78125,   9.609375,     7.328125,      -10.5390625, 7.06640625,
            9.4375,      14.0859375,   20.703125,     8.4765625,   4.55078125,
            18.359375,   -1.08984375,  1.326171875,   -68.9375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 3D tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [
            9.34375,       0.280029296875, -4.6171875,    1.1201171875,
            -1.43359375,   -3.158203125,   -6.2890625,    -5.01171875,
            -6.8984375,    3.572265625,    6.86328125,    -1.9619140625,
            4.58203125,    2.6640625,      9.1953125,     -9.5546875,
            -5.50390625,   -2.392578125,   3.58203125,    -2.322265625,
            -1.9814453125, 4.15625,        -1.7998046875, 9.296875
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.8125,    -1.341796875, 8.4140625,     6.109375,    12.171875,
            3.314453125, 1.1689453125, 0.71044921875, 46.3125,     5.7890625,
            -25.78125,   9.609375,     7.328125,      -10.5390625, 7.06640625,
            9.4375,      14.0859375,   20.703125,     8.4765625,   4.55078125,
            18.359375,   -1.08984375,  1.326171875,   -68.9375
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 4D tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [
            9.34375,       0.280029296875, -4.6171875,    1.1201171875,
            -1.43359375,   -3.158203125,   -6.2890625,    -5.01171875,
            -6.8984375,    3.572265625,    6.86328125,    -1.9619140625,
            4.58203125,    2.6640625,      9.1953125,     -9.5546875,
            -5.50390625,   -2.392578125,   3.58203125,    -2.322265625,
            -1.9814453125, 4.15625,        -1.7998046875, 9.296875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.8125,    -1.341796875, 8.4140625,     6.109375,    12.171875,
            3.314453125, 1.1689453125, 0.71044921875, 46.3125,     5.7890625,
            -25.78125,   9.609375,     7.328125,      -10.5390625, 7.06640625,
            9.4375,      14.0859375,   20.703125,     8.4765625,   4.55078125,
            18.359375,   -1.08984375,  1.326171875,   -68.9375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 5D tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [
            9.34375,       0.280029296875, -4.6171875,    1.1201171875,
            -1.43359375,   -3.158203125,   -6.2890625,    -5.01171875,
            -6.8984375,    3.572265625,    6.86328125,    -1.9619140625,
            4.58203125,    2.6640625,      9.1953125,     -9.5546875,
            -5.50390625,   -2.392578125,   3.58203125,    -2.322265625,
            -1.9814453125, 4.15625,        -1.7998046875, 9.296875
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -23.8125,    -1.341796875, 8.4140625,     6.109375,    12.171875,
            3.314453125, 1.1689453125, 0.71044921875, 46.3125,     5.7890625,
            -25.78125,   9.609375,     7.328125,      -10.5390625, 7.06640625,
            9.4375,      14.0859375,   20.703125,     8.4765625,   4.55078125,
            18.359375,   -1.08984375,  1.326171875,   -68.9375
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 broadcast 4D x 1D slope',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [5.07421875, 0.480712890625, -7.08984375],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -12.9296875,   -2.3046875,    8.4140625,     6.109375,
            -4.08203125,   3.314453125,   1.1689453125,  -0.068115234375,
            47.59375,      5.7890625,     -1.8056640625, 34.71875,
            7.328125,      -1.9013671875, 7.06640625,    9.4375,
            -1.2294921875, 61.375,        8.4765625,     4.55078125,
            65.6875,       -1.330078125,  1.326171875,   52.59375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 broadcast 4D x 1D slope of shape [1]',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [5.01171875],
          'descriptor': {shape: [1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -12.7734375, -24.015625,   8.4140625,      6.109375,   -42.5625,
            3.314453125, 1.1689453125, -0.71044921875, -33.65625,  5.7890625,
            -18.828125,  -24.546875,   7.328125,       -19.828125, 7.06640625,
            9.4375,      -12.8203125,  -43.375,        8.4765625,  4.55078125,
            -46.4375,    -1.314453125, 1.326171875,    -37.1875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 broadcast 4D x 2D slope',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [4.875, -8.5, 1.181640625, -9.984375, -4.42578125, -6.65625],
          'descriptor': {shape: [2, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -12.421875,  40.75,        8.4140625,   6.109375,    37.59375,
            3.314453125, 1.1689453125, 1.205078125, -7.93359375, 5.7890625,
            16.625,      32.59375,     7.328125,    33.625,      7.06640625,
            9.4375,      11.3203125,   57.625,      8.4765625,   4.55078125,
            -10.9453125, 2.6171875,    1.326171875, 49.375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 broadcast 4D x 3D slope',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [5.07421875, 0.480712890625, -7.08984375],
          'descriptor': {shape: [1, 1, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -12.9296875,   -2.3046875,    8.4140625,     6.109375,
            -4.08203125,   3.314453125,   1.1689453125,  -0.068115234375,
            47.59375,      5.7890625,     -1.8056640625, 34.71875,
            7.328125,      -1.9013671875, 7.06640625,    9.4375,
            -1.2294921875, 61.375,        8.4765625,     4.55078125,
            65.6875,       -1.330078125,  1.326171875,   52.59375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'prelu float16 broadcast 4D x 4D slope',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [
            -2.548828125, -4.79296875,    8.4140625,    6.109375,
            -8.4921875,   3.314453125,    1.1689453125, -0.1417236328125,
            -6.71484375,  5.7890625,      -3.755859375, -4.8984375,
            7.328125,     -3.955078125,   7.06640625,   9.4375,
            -2.55859375,  -8.65625,       8.4765625,    4.55078125,
            -9.265625,    -0.26220703125, 1.326171875,  -7.41796875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'preluSlope': {
          'data': [5.01171875],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [
            -12.7734375, -24.015625,   8.4140625,      6.109375,   -42.5625,
            3.314453125, 1.1689453125, -0.71044921875, -33.65625,  5.7890625,
            -18.828125,  -24.546875,   7.328125,       -19.828125, 7.06640625,
            9.4375,      -12.8203125,  -43.375,        8.4765625,  4.55078125,
            -46.4375,    -1.314453125, 1.326171875,    -37.1875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },

  // int64 tests
  {
    'name': 'prelu int64 2D constant tensors',
    'graph': {
      'inputs': {
        'preluInput': {
          'data': [-4, -2, -1, 0, 0, 0, 1, 2, 4],
          'descriptor': {shape: [3, 3], dataType: 'int64'},
          'constant': true
        },
        'preluSlope': {
          'data': [-5, 0, 5, -5, 0, 5, -5, 0, 5],
          'descriptor': {shape: [3, 3], dataType: 'int64'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'prelu',
        'arguments': [{'input': 'preluInput'}, {'slope': 'preluSlope'}],
        'outputs': 'preluOutput'
      }],
      'expectedOutputs': {
        'preluOutput': {
          'data': [20, 0, -5, 0, 0, 0, 1, 2, 4],
          'descriptor': {shape: [3, 3], dataType: 'int64'}
        }
      }
    }
  }
];

webnn_conformance_test(preluTests, buildAndExecuteGraph, getPrecisionTolerance);
