// META: title=test WebNN API element-wise pow operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-binary
// Compute the element-wise binary power of the two input tensors.
// MLOperand pow(MLOperand a, MLOperand b);


const getPowPrecisionTolerance = (graphResources) => {
  const toleranceValueDict = {float32: 32, float16: 2};
  const expectedDataType =
      getExpectedDataTypeOfSingleOutput(graphResources.expectedOutputs);
  return {metricType: 'ULP', value: toleranceValueDict[expectedDataType]};
};

const powTests = [
  {
    'name':
        'pow float32 constant 1D base tensor and 1D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.846010208129883,  -0.0631069764494896, -9.868203163146973,
            11.17772102355957,   -17.346275329589844, 11.862250328063965,
            -16.832275390625,    2.6574816703796387,  -2.783346652984619,
            -13.756400108337402, 13.131382942199707,  -0.4376337230205536,
            -15.678689002990723, 10.283306121826172,  14.893174171447754,
            -4.941208362579346,  -14.231812477111816, 3.2646026611328125,
            17.229148864746094,  -2.885918140411377,  -1.4400150775909424,
            -5.757015705108643,  17.41126823425293,   17.41521453857422
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        },
        'inputB': {
          'data': [
            1,  6,  -7, 7,  -2, 1, 4,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 6, -6, 7,   -5, -5, 7,  -6
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.846010208129883,      6.316321332633379e-8,
            -1.0973203501407625e-7,  21800822,
            0.0033234376460313797,   11.862250328063965,
            80273.3359375,           0.00005692423656000756,
            0.12908191978931427,     -0.0000020299064544815337,
            0.005799346603453159,    3880.540283203125,
            2.7385585465999895e-10,  8.223764069725803e-8,
            0.06714485585689545,     -0.008288968354463577,
            -4.1750155416186985e-11, 1210.5478515625,
            3.8231124932508465e-8,   -1667.201416015625,
            -0.16149713099002838,    -0.00015812950732652098,
            485079424,               3.584487018315485e-8
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'pow float32 1D base tensor and 1D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.846010208129883,  -0.0631069764494896, -9.868203163146973,
            11.17772102355957,   -17.346275329589844, 11.862250328063965,
            -16.832275390625,    2.6574816703796387,  -2.783346652984619,
            -13.756400108337402, 13.131382942199707,  -0.4376337230205536,
            -15.678689002990723, 10.283306121826172,  14.893174171447754,
            -4.941208362579346,  -14.231812477111816, 3.2646026611328125,
            17.229148864746094,  -2.885918140411377,  -1.4400150775909424,
            -5.757015705108643,  17.41126823425293,   17.41521453857422
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            1,  6,  -7, 7,  -2, 1, 4,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 6, -6, 7,   -5, -5, 7,  -6
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.846010208129883,      6.316321332633379e-8,
            -1.0973203501407625e-7,  21800822,
            0.0033234376460313797,   11.862250328063965,
            80273.3359375,           0.00005692423656000756,
            0.12908191978931427,     -0.0000020299064544815337,
            0.005799346603453159,    3880.540283203125,
            2.7385585465999895e-10,  8.223764069725803e-8,
            0.06714485585689545,     -0.008288968354463577,
            -4.1750155416186985e-11, 1210.5478515625,
            3.8231124932508465e-8,   -1667.201416015625,
            -0.16149713099002838,    -0.00015812950732652098,
            485079424,               3.584487018315485e-8
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'pow float32 2D base tensor and 2D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.846010208129883,  -0.0631069764494896, -9.868203163146973,
            11.17772102355957,   -17.346275329589844, 11.862250328063965,
            -16.832275390625,    2.6574816703796387,  -2.783346652984619,
            -13.756400108337402, 13.131382942199707,  -0.4376337230205536,
            -15.678689002990723, 10.283306121826172,  14.893174171447754,
            -4.941208362579346,  -14.231812477111816, 3.2646026611328125,
            17.229148864746094,  -2.885918140411377,  -1.4400150775909424,
            -5.757015705108643,  17.41126823425293,   17.41521453857422
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            1,  6,  -7, 7,  -2, 1, 4,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 6, -6, 7,   -5, -5, 7,  -6
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.846010208129883,      6.316321332633379e-8,
            -1.0973203501407625e-7,  21800822,
            0.0033234376460313797,   11.862250328063965,
            80273.3359375,           0.00005692423656000756,
            0.12908191978931427,     -0.0000020299064544815337,
            0.005799346603453159,    3880.540283203125,
            2.7385585465999895e-10,  8.223764069725803e-8,
            0.06714485585689545,     -0.008288968354463577,
            -4.1750155416186985e-11, 1210.5478515625,
            3.8231124932508465e-8,   -1667.201416015625,
            -0.16149713099002838,    -0.00015812950732652098,
            485079424,               3.584487018315485e-8
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'pow float32 3D base tensor and 3D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.846010208129883,  -0.0631069764494896, -9.868203163146973,
            11.17772102355957,   -17.346275329589844, 11.862250328063965,
            -16.832275390625,    2.6574816703796387,  -2.783346652984619,
            -13.756400108337402, 13.131382942199707,  -0.4376337230205536,
            -15.678689002990723, 10.283306121826172,  14.893174171447754,
            -4.941208362579346,  -14.231812477111816, 3.2646026611328125,
            17.229148864746094,  -2.885918140411377,  -1.4400150775909424,
            -5.757015705108643,  17.41126823425293,   17.41521453857422
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            1,  6,  -7, 7,  -2, 1, 4,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 6, -6, 7,   -5, -5, 7,  -6
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.846010208129883,      6.316321332633379e-8,
            -1.0973203501407625e-7,  21800822,
            0.0033234376460313797,   11.862250328063965,
            80273.3359375,           0.00005692423656000756,
            0.12908191978931427,     -0.0000020299064544815337,
            0.005799346603453159,    3880.540283203125,
            2.7385585465999895e-10,  8.223764069725803e-8,
            0.06714485585689545,     -0.008288968354463577,
            -4.1750155416186985e-11, 1210.5478515625,
            3.8231124932508465e-8,   -1667.201416015625,
            -0.16149713099002838,    -0.00015812950732652098,
            485079424,               3.584487018315485e-8
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'pow float32 4D base tensor and 4D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.846010208129883,  -0.0631069764494896, -9.868203163146973,
            11.17772102355957,   -17.346275329589844, 11.862250328063965,
            -16.832275390625,    2.6574816703796387,  -2.783346652984619,
            -13.756400108337402, 13.131382942199707,  -0.4376337230205536,
            -15.678689002990723, 10.283306121826172,  14.893174171447754,
            -4.941208362579346,  -14.231812477111816, 3.2646026611328125,
            17.229148864746094,  -2.885918140411377,  -1.4400150775909424,
            -5.757015705108643,  17.41126823425293,   17.41521453857422
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            1,  6,  -7, 7,  -2, 1, 4,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 6, -6, 7,   -5, -5, 7,  -6
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.846010208129883,      6.316321332633379e-8,
            -1.0973203501407625e-7,  21800822,
            0.0033234376460313797,   11.862250328063965,
            80273.3359375,           0.00005692423656000756,
            0.12908191978931427,     -0.0000020299064544815337,
            0.005799346603453159,    3880.540283203125,
            2.7385585465999895e-10,  8.223764069725803e-8,
            0.06714485585689545,     -0.008288968354463577,
            -4.1750155416186985e-11, 1210.5478515625,
            3.8231124932508465e-8,   -1667.201416015625,
            -0.16149713099002838,    -0.00015812950732652098,
            485079424,               3.584487018315485e-8
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'pow float32 5D base tensor and 5D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.846010208129883,  -0.0631069764494896, -9.868203163146973,
            11.17772102355957,   -17.346275329589844, 11.862250328063965,
            -16.832275390625,    2.6574816703796387,  -2.783346652984619,
            -13.756400108337402, 13.131382942199707,  -0.4376337230205536,
            -15.678689002990723, 10.283306121826172,  14.893174171447754,
            -4.941208362579346,  -14.231812477111816, 3.2646026611328125,
            17.229148864746094,  -2.885918140411377,  -1.4400150775909424,
            -5.757015705108643,  17.41126823425293,   17.41521453857422
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            1,  6,  -7, 7,  -2, 1, 4,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 6, -6, 7,   -5, -5, 7,  -6
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.846010208129883,      6.316321332633379e-8,
            -1.0973203501407625e-7,  21800822,
            0.0033234376460313797,   11.862250328063965,
            80273.3359375,           0.00005692423656000756,
            0.12908191978931427,     -0.0000020299064544815337,
            0.005799346603453159,    3880.540283203125,
            2.7385585465999895e-10,  8.223764069725803e-8,
            0.06714485585689545,     -0.008288968354463577,
            -4.1750155416186985e-11, 1210.5478515625,
            3.8231124932508465e-8,   -1667.201416015625,
            -0.16149713099002838,    -0.00015812950732652098,
            485079424,               3.584487018315485e-8
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'pow (square) float32 4D base tensor and broadcastable 0D integer exponent scalar',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.846010208129883,  -0.0631069764494896, -9.868203163146973,
            11.17772102355957,   -17.346275329589844, 11.862250328063965,
            -16.832275390625,    2.6574816703796387,  -2.783346652984619,
            -13.756400108337402, 13.131382942199707,  -0.4376337230205536,
            -15.678689002990723, 10.283306121826172,  14.893174171447754,
            -4.941208362579346,  -14.231812477111816, 3.2646026611328125,
            17.229148864746094,  -2.885918140411377,  -1.4400150775909424,
            -5.757015705108643,  17.41126823425293,   17.41521453857422
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'},
          'constant': true
        },
        'inputB': {'data': [2], 'descriptor': {shape: [], dataType: 'float32'}}
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            318.4800720214844,  0.00398249039426446, 97.38143157958984,
            124.94144439697266, 300.8932800292969,   140.71298217773438,
            283.32550048828125, 7.062208652496338,   7.747018814086914,
            189.23854064941406, 172.43321228027344,  0.19152326881885529,
            245.8212890625,     105.74638366699219,  221.806640625,
            24.41554069519043,  202.5444793701172,   10.657630920410156,
            296.84356689453125, 8.328523635864258,   2.073643445968628,
            33.14323043823242,  303.1522521972656,   303.2897033691406
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'pow (sqrt) float32 4D positive base tensor and broadcastable 0D integer exponent scalar',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            1.418652057647705,  19.384845733642578,  12.983916282653809,
            2.4603159427642822, 7.818154811859131,   6.94444465637207,
            12.183951377868652, 17.912473678588867,  11.356568336486816,
            8.924248695373535,  17.636823654174805,  11.49622917175293,
            18.516279220581055, 2.2580490112304688,  2.231948137283325,
            13.629855155944824, 17.54841423034668,   0.5390734076499939,
            5.891367435455322,  0.12803149223327637, 19.654495239257812,
            3.4122724533081055, 4.945034980773926,   4.437101364135742
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'},
          'constant': true
        },
        'inputB':
            {'data': [0.5], 'descriptor': {shape: [], dataType: 'float32'}}
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1.19107186794281,   4.402822494506836,   3.6033201217651367,
            1.5685393810272217, 2.7960963249206543,  2.6352314949035645,
            3.490551710128784,  4.23231315612793,    3.369950771331787,
            2.9873480796813965, 4.199621677398682,   3.3906090259552,
            4.3030548095703125, 1.5026806592941284,  1.4939706325531006,
            3.6918632984161377, 4.189082622528076,   0.7342162132263184,
            2.4272139072418213, 0.35781487822532654, 4.4333391189575195,
            1.847233772277832,  2.223743438720703,   2.106442928314209
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'pow float32 4D base tensor and broadcastable 2D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.846010208129883,  -0.0631069764494896, -9.868203163146973,
            11.17772102355957,   -17.346275329589844, 11.862250328063965,
            -16.832275390625,    2.6574816703796387,  -2.783346652984619,
            -13.756400108337402, 13.131382942199707,  -0.4376337230205536,
            -15.678689002990723, 10.283306121826172,  14.893174171447754,
            -4.941208362579346,  -14.231812477111816, 3.2646026611328125,
            17.229148864746094,  -2.885918140411377,  -1.4400150775909424,
            -5.757015705108643,  17.41126823425293,   17.41521453857422
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [5, -10, -10, 7, -7, -9],
          'descriptor': {shape: [2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1810113,        998220038144,           1.1418765932802444e-10,
            21800822,       -2.11619832768406e-9,   2.1502860603206386e-10,
            -1351182.875,   0.00005692423656000756, 0.000035836616007145494,
            -93225256,      1.4853429597394552e-8,  -1698.2552490234375,
            -947433.5,      7.562621362477984e-11,  1.8626330946375225e-12,
            -71917.1015625, -8.45626324519344e-9,   0.00002374253199377563,
            1518165.5,      0.00002495513399480842, 0.026081321761012077,
            -209595.46875,  2.0615180673644318e-9,  6.786416914539295e-12
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'pow float32 4D base tensor and broadcastable 3D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.846010208129883,  -0.0631069764494896, -9.868203163146973,
            11.17772102355957,   -17.346275329589844, 11.862250328063965,
            -16.832275390625,    2.6574816703796387,  -2.783346652984619,
            -13.756400108337402, 13.131382942199707,  -0.4376337230205536,
            -15.678689002990723, 10.283306121826172,  14.893174171447754,
            -4.941208362579346,  -14.231812477111816, 3.2646026611328125,
            17.229148864746094,  -2.885918140411377,  -1.4400150775909424,
            -5.757015705108643,  17.41126823425293,   17.41521453857422
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [-5, -10, 9, -6],
          'descriptor': {shape: [2, 2, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            5.524516950572433e-7,      -999109.625,
            -0.000010685862434911542,  3.284485530774539e-11,
            4.0545030440680696e-13,    1.81271334748212e-11,
            -108463955968,             6610.47265625,
            -10025.4921875,            1.4756086841316574e-7,
            1.9504606996179064e-7,     142.34274291992188,
            -0.0000010554830396358739, 0.000008696333679836243,
            0.0000013647832020069472,  1.1525726506533829e-7,
            2.9335795945217846e-12,    0.000007272717084561009,
            133774827520,              -13885.326171875,
            -26.625843048095703,       0.000027467271138448268,
            3.5893645389251105e-8,     3.584487018315485e-8
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name':
        'pow float16 constant 1D base tensor and 1D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.84375,      -0.0631103515625, -9.8671875, 11.1796875,
            -17.34375,     11.859375,        -16.828125, 2.658203125,
            -2.783203125,  -13.7578125,      13.1328125, -0.437744140625,
            -15.6796875,   10.28125,         14.890625,  -4.94140625,
            -14.234375,    3.263671875,      17.234375,  -2.88671875,
            -1.4404296875, -5.7578125,       17.40625,   17.421875
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        },
        'inputB': {
          'data': [
            1,  3,  -7, 2,  -2, 1, 2,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 2, -6, 3,   -5, -5, 2,  -6
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.84375,               -0.0002512931823730469,
            -1.1920928955078125e-7, 125,
            0.0033245086669921875,  11.859375,
            283.25,                 0.000056743621826171875,
            0.129150390625,         -0.0000020265579223632812,
            0.00579833984375,       3870,
            0,                      5.960464477539063e-8,
            0.067138671875,         -0.0082855224609375,
            0,                      10.6484375,
            5.960464477539063e-8,   -24.0625,
            -0.1612548828125,       -0.00015807151794433594,
            303,                    5.960464477539063e-8
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'pow float16 1D base tensor and 1D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.84375,      -0.0631103515625, -9.8671875, 11.1796875,
            -17.34375,     11.859375,        -16.828125, 2.658203125,
            -2.783203125,  -13.7578125,      13.1328125, -0.437744140625,
            -15.6796875,   10.28125,         14.890625,  -4.94140625,
            -14.234375,    3.263671875,      17.234375,  -2.88671875,
            -1.4404296875, -5.7578125,       17.40625,   17.421875
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            1,  3,  -7, 2,  -2, 1, 2,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 2, -6, 3,   -5, -5, 2,  -6
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.84375,               -0.0002512931823730469,
            -1.1920928955078125e-7, 125,
            0.0033245086669921875,  11.859375,
            283.25,                 0.000056743621826171875,
            0.129150390625,         -0.0000020265579223632812,
            0.00579833984375,       3870,
            0,                      5.960464477539063e-8,
            0.067138671875,         -0.0082855224609375,
            0,                      10.6484375,
            5.960464477539063e-8,   -24.0625,
            -0.1612548828125,       -0.00015807151794433594,
            303,                    5.960464477539063e-8
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'pow float16 2D base tensor and 2D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.84375,      -0.0631103515625, -9.8671875, 11.1796875,
            -17.34375,     11.859375,        -16.828125, 2.658203125,
            -2.783203125,  -13.7578125,      13.1328125, -0.437744140625,
            -15.6796875,   10.28125,         14.890625,  -4.94140625,
            -14.234375,    3.263671875,      17.234375,  -2.88671875,
            -1.4404296875, -5.7578125,       17.40625,   17.421875
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            1,  3,  -7, 2,  -2, 1, 2,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 2, -6, 3,   -5, -5, 2,  -6
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.84375,               -0.0002512931823730469,
            -1.1920928955078125e-7, 125,
            0.0033245086669921875,  11.859375,
            283.25,                 0.000056743621826171875,
            0.129150390625,         -0.0000020265579223632812,
            0.00579833984375,       3870,
            0,                      5.960464477539063e-8,
            0.067138671875,         -0.0082855224609375,
            0,                      10.6484375,
            5.960464477539063e-8,   -24.0625,
            -0.1612548828125,       -0.00015807151794433594,
            303,                    5.960464477539063e-8
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'pow float16 3D base tensor and 3D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.84375,      -0.0631103515625, -9.8671875, 11.1796875,
            -17.34375,     11.859375,        -16.828125, 2.658203125,
            -2.783203125,  -13.7578125,      13.1328125, -0.437744140625,
            -15.6796875,   10.28125,         14.890625,  -4.94140625,
            -14.234375,    3.263671875,      17.234375,  -2.88671875,
            -1.4404296875, -5.7578125,       17.40625,   17.421875
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            1,  3,  -7, 2,  -2, 1, 2,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 2, -6, 3,   -5, -5, 2,  -6
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.84375,               -0.0002512931823730469,
            -1.1920928955078125e-7, 125,
            0.0033245086669921875,  11.859375,
            283.25,                 0.000056743621826171875,
            0.129150390625,         -0.0000020265579223632812,
            0.00579833984375,       3870,
            0,                      5.960464477539063e-8,
            0.067138671875,         -0.0082855224609375,
            0,                      10.6484375,
            5.960464477539063e-8,   -24.0625,
            -0.1612548828125,       -0.00015807151794433594,
            303,                    5.960464477539063e-8
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'pow float16 4D base tensor and 4D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.84375,      -0.0631103515625, -9.8671875, 11.1796875,
            -17.34375,     11.859375,        -16.828125, 2.658203125,
            -2.783203125,  -13.7578125,      13.1328125, -0.437744140625,
            -15.6796875,   10.28125,         14.890625,  -4.94140625,
            -14.234375,    3.263671875,      17.234375,  -2.88671875,
            -1.4404296875, -5.7578125,       17.40625,   17.421875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            1,  3,  -7, 2,  -2, 1, 2,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 2, -6, 3,   -5, -5, 2,  -6
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.84375,               -0.0002512931823730469,
            -1.1920928955078125e-7, 125,
            0.0033245086669921875,  11.859375,
            283.25,                 0.000056743621826171875,
            0.129150390625,         -0.0000020265579223632812,
            0.00579833984375,       3870,
            0,                      5.960464477539063e-8,
            0.067138671875,         -0.0082855224609375,
            0,                      10.6484375,
            5.960464477539063e-8,   -24.0625,
            -0.1612548828125,       -0.00015807151794433594,
            303,                    5.960464477539063e-8
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'pow float16 5D base tensor and 5D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.84375,      -0.0631103515625, -9.8671875, 11.1796875,
            -17.34375,     11.859375,        -16.828125, 2.658203125,
            -2.783203125,  -13.7578125,      13.1328125, -0.437744140625,
            -15.6796875,   10.28125,         14.890625,  -4.94140625,
            -14.234375,    3.263671875,      17.234375,  -2.88671875,
            -1.4404296875, -5.7578125,       17.40625,   17.421875
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            1,  3,  -7, 2,  -2, 1, 2,  -10, -2, -5, -2, -10,
            -8, -7, -1, -3, -9, 2, -6, 3,   -5, -5, 2,  -6
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            17.84375,               -0.0002512931823730469,
            -1.1920928955078125e-7, 125,
            0.0033245086669921875,  11.859375,
            283.25,                 0.000056743621826171875,
            0.129150390625,         -0.0000020265579223632812,
            0.00579833984375,       3870,
            0,                      5.960464477539063e-8,
            0.067138671875,         -0.0082855224609375,
            0,                      10.6484375,
            5.960464477539063e-8,   -24.0625,
            -0.1612548828125,       -0.00015807151794433594,
            303,                    5.960464477539063e-8
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'pow (square) float16 4D base tensor and broadcastable 0D integer exponent scalar',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.84375,      -0.0631103515625, -9.8671875, 11.1796875,
            -17.34375,     11.859375,        -16.828125, 2.658203125,
            -2.783203125,  -13.7578125,      13.1328125, -0.437744140625,
            -15.6796875,   10.28125,         14.890625,  -4.94140625,
            -14.234375,    3.263671875,      17.234375,  -2.88671875,
            -1.4404296875, -5.7578125,       17.40625,   17.421875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'},
          'constant': true
        },
        'inputB': {'data': [2], 'descriptor': {shape: [], dataType: 'float16'}}
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            318.5,      0.0039825439453125,
            97.375,     125,
            300.75,     140.625,
            283.25,     7.06640625,
            7.74609375, 189.25,
            172.5,      0.191650390625,
            245.875,    105.6875,
            221.75,     24.421875,
            202.625,    10.6484375,
            297,        8.3359375,
            2.07421875, 33.15625,
            303,        303.5
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'pow (sqrt) float16 4D positive base tensor and broadcastable 0D integer exponent scalar',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            1.4189453125, 19.390625,   12.984375, 2.4609375, 7.81640625,
            6.9453125,    12.1875,     17.90625,  11.359375, 8.921875,
            17.640625,    11.5,        18.515625, 2.2578125, 2.232421875,
            13.6328125,   17.546875,   0.5390625, 5.890625,  0.1280517578125,
            19.65625,     3.412109375, 4.9453125, 4.4375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'},
          'constant': true
        },
        'inputB':
            {'data': [0.5], 'descriptor': {shape: [], dataType: 'float16'}}
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1.19140625,  4.40234375,  3.603515625, 1.568359375,  2.794921875,
            2.634765625, 3.490234375, 4.23046875,  3.37109375,   2.986328125,
            4.19921875,  3.390625,    4.3046875,   1.5029296875, 1.494140625,
            3.69140625,  4.1875,      0.734375,    2.427734375,  0.35791015625,
            4.43359375,  1.84765625,  2.224609375, 2.107421875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'pow float16 4D base tensor and broadcastable 2D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.84375,      -0.0631103515625, -9.8671875, 11.1796875,
            -17.34375,     11.859375,        -16.828125, 2.658203125,
            -2.783203125,  -13.7578125,      13.1328125, -0.437744140625,
            -15.6796875,   10.28125,         14.890625,  -4.94140625,
            -14.234375,    3.263671875,      17.234375,  -2.88671875,
            -1.4404296875, -5.7578125,       17.40625,   17.421875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [3, -2, -1, 2, -7, -5],
          'descriptor': {shape: [2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            5680,      251.125,          -0.101318359375,
            125,       0,                0.000004291534423828125,
            -4764,     0.1414794921875,  -0.359375,
            189.25,    0,                -62.21875,
            -3854,     0.00946044921875, 0.067138671875,
            24.421875, 0,                0.0027008056640625,
            5120,      0.1199951171875,  -0.6943359375,
            33.15625,  0,                5.960464477539062e-7
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'pow float16 4D base tensor and broadcastable 3D integer exponent tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            17.84375,      -0.0631103515625, -9.8671875, 11.1796875,
            -17.34375,     11.859375,        -16.828125, 2.658203125,
            -2.783203125,  -13.7578125,      13.1328125, -0.437744140625,
            -15.6796875,   10.28125,         14.890625,  -4.94140625,
            -14.234375,    3.263671875,      17.234375,  -2.88671875,
            -1.4404296875, -5.7578125,       17.40625,   17.421875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [-1, 1, 3, -2],
          'descriptor': {shape: [2, 2, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'pow',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            0.0560302734375,      -15.84375,            -0.101318359375,
            11.1796875,           -17.34375,            11.859375,
            -4764,                18.78125,             -21.5625,
            0.005283355712890625, 0.00579833984375,     5.21875,
            -0.06378173828125,    0.0972900390625,      0.067138671875,
            -4.94140625,          -14.234375,           3.263671875,
            5120,                 -24.0625,             -2.98828125,
            0.0301666259765625,   0.003299713134765625, 0.0032939910888671875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(
    powTests, buildAndExecuteGraph, getPowPrecisionTolerance);
