// META: title=test WebNN API maxPool2d operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-pool2d
// Compute a pooling operation across all the elements within the moving window
// over the input tensor.
//
// enum MLRoundingType {
//   "floor",
//   "ceil"
// };
//
// dictionary MLPool2dOptions {
//   sequence<[EnforceRange] unsigned long> windowDimensions;
//   sequence<[EnforceRange] unsigned long> padding;
//   sequence<[EnforceRange] unsigned long> strides;
//   sequence<[EnforceRange] unsigned long> dilations;
//   MLInputOperandLayout layout = "nchw";
//   MLRoundingType roundingType = "floor";
//   sequence<[EnforceRange] unsigned long> outputSizes;
// };
//
// MLOperand maxPool2d(
//     MLOperand input, optional MLPool2dOptions options = {});

const maxPool2dTests = [
  // float32 tests
  {
    'name': 'maxPool2d float32 4D constant tensor default options',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [{'input': 'maxPool2dInput'}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.28312683105469, 81.73119354248047],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor default options',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [{'input': 'maxPool2dInput'}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.28312683105469, 81.73119354248047],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.windowDimensions',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {'options': {'windowDimensions': [3, 3]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 99.28312683105469, 66.09259033203125,
            99.28312683105469, 99.28312683105469, 72.1085205078125,
            97.90348052978516, 72.1085205078125, 72.1085205078125,
            81.73119354248047, 72.44898986816406, 72.44898986816406,
            81.73119354248047, 72.8883056640625, 72.44898986816406,
            80.30484008789062, 72.8883056640625, 72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.padding',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {'options': {'padding': [1, 0, 0, 1]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 99.28312683105469, 99.28312683105469,
            99.28312683105469, 81.73119354248047, 72.8883056640625,
            81.73119354248047, 72.8883056640625
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.strides',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'strides': [2, 2]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 66.09259033203125, 97.90348052978516,
            72.1085205078125, 81.73119354248047, 72.44898986816406,
            80.30484008789062, 72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.dilations',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'dilations': [2, 2]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [89.00830078125, 72.33577728271484],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.layout=nchw',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments':
            [{'input': 'maxPool2dInput'}, {'options': {'layout': 'nchw'}}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.28312683105469, 81.73119354248047],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.layout=nhwc',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -76.72171020507812,  -45.72039031982422,
            50.217063903808594,  -61.306129455566406, -52.895477294921875,
            -4.014514446258545,  -44.642333984375,    -94.54893493652344,
            -97.86752319335938,  46.28090286254883,   81.73119354248047,
            99.28312683105469,   5.428491115570068,   -10.057873725891113,
            -29.22772789001465,  9.742474555969238,   72.44898986816406,
            -39.03501892089844,  -59.34124755859375,  75.08192443847656,
            39.19960021972656,   12.819415092468262,  -65.99439239501953,
            -33.01505661010742,  -4.204323768615723,  38.691341400146484,
            -60.54586410522461,  66.09259033203125,   55.890525817871094,
            97.90348052978516,   80.30484008789062,   -8.737770080566406,
            72.8883056640625,    -53.42162322998047,  -46.59611129760742,
            72.1085205078125,    20.50387954711914,   -40.423091888427734,
            -31.126462936401367, -35.68864440917969,  -57.294559478759766,
            -87.64779663085938,  -26.623577117919922, 38.874244689941406,
            15.935754776000977,  39.383602142333984,  -78.77953338623047,
            7.429088115692139,   72.33577728271484
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments':
            [{'input': 'maxPool2dInput'}, {'options': {'layout': 'nhwc'}}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.28312683105469, 81.73119354248047],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.roundingType=floor',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 9.742474555969238, 99.28312683105469,
            72.1085205078125, 81.73119354248047, 72.44898986816406,
            81.73119354248047, 72.44898986816406
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.roundingType=ceil',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 9.742474555969238, -39.03501892089844,
            99.28312683105469, 72.1085205078125, 66.09259033203125,
            97.90348052978516, 72.1085205078125, 7.429088115692139,
            81.73119354248047, 72.44898986816406, -59.34124755859375,
            81.73119354248047, 72.44898986816406, 55.890525817871094,
            80.30484008789062, 72.33577728271484, 72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'maxPool2d float32 4D tensor options.outputSizes ignores options.roundingType=floor',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor',
              'outputSizes': [3, 3]
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 9.742474555969238, -39.03501892089844,
            99.28312683105469, 72.1085205078125, 66.09259033203125,
            97.90348052978516, 72.1085205078125, 7.429088115692139,
            81.73119354248047, 72.44898986816406, -59.34124755859375,
            81.73119354248047, 72.44898986816406, 55.890525817871094,
            80.30484008789062, 72.33577728271484, 72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'maxPool2d float32 4D tensor options.outputSizes ignores options.roundingType=ceil',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil',
              'outputSizes': [2, 2]
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 9.742474555969238, 99.28312683105469,
            72.1085205078125, 81.73119354248047, 72.44898986816406,
            81.73119354248047, 72.44898986816406
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'maxPool2d float32 4D tensor options.dilations with options.strides',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            34.69258117675781,   -24.706249237060547, -60.428070068359375,
            35.93883514404297,   60.896095275878906,  79.42220306396484,
            -77.85906219482422,  54.188209533691406,  -21.576934814453125,
            -49.10390853881836,  78.55176544189453,   74.28213500976562,
            -87.92769622802734,  79.82047271728516,   11.680922508239746,
            -12.154505729675293, -22.33293914794922,  33.084861755371094,
            8.640676498413086,   47.040645599365234,  95.7823486328125,
            -88.01998138427734,  -98.53630065917969,  16.158977508544922,
            21.396089553833008,  95.1323471069336,    -40.80724334716797,
            -88.70922088623047,  -40.772769927978516, 67.89842987060547,
            -50.337467193603516, -96.56610870361328,  12.508098602294922,
            -6.6358113288879395, -44.80198287963867,  80.27474212646484,
            -65.68411254882812,  -14.884790420532227, -66.54857635498047,
            20.072338104248047,  -27.954269409179688, -56.62217330932617,
            82.7479476928711,    93.30175018310547,   -34.073394775390625,
            -22.87164306640625,  73.25525665283203,   41.14021682739258,
            -19.75514793395996,  80.07701110839844,   -69.89276885986328,
            14.013250350952148,  -61.36640167236328,  51.53046417236328,
            43.53886413574219,   -89.5888671875,      51.45121765136719,
            73.9239730834961,    -80.25264739990234,  94.72747802734375,
            95.25411224365234,   58.12575149536133,   19.885723114013672,
            -70.0301284790039,   63.419517517089844,  -54.11785125732422,
            -97.22807312011719,  -60.65826416015625,  -31.04998016357422,
            -14.646553039550781, -63.73688888549805,  47.34630584716797,
            85.56405639648438,   -53.389251708984375, -70.84739685058594,
            47.355045318603516,  83.38397979736328,   7.361695289611816,
            46.85823440551758,   98.13465881347656,   -43.9396858215332,
            50.33780288696289,   37.45563507080078,   -54.52760696411133,
            -6.212307929992676,  34.41734313964844,   11.8167724609375,
            72.44517517089844,   86.3460922241211,    4.14656925201416,
            88.33040618896484,   98.29994201660156,   -66.72379302978516,
            58.0643424987793,    -51.168277740478516, -17.768583297729492,
            9.961172103881836,   -52.73843002319336
          ],
          'descriptor': {shape: [1, 7, 7, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'dilations': [1, 1],
              'layout': 'nhwc'
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            60.896095275878906, 79.42220306396484, 95.7823486328125,
            79.42220306396484, 78.55176544189453, 95.1323471069336,
            82.7479476928711, 93.30175018310547, 95.7823486328125,
            80.27474212646484, 43.53886413574219, 95.1323471069336,
            95.25411224365234, 94.72747802734375, 95.25411224365234,
            98.13465881347656, 63.419517517089844, 98.13465881347656
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'maxPool2d float16 4D constant tensor default options',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [{'input': 'maxPool2dInput'}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.3125, 81.75],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float16 4D tensor default options',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [{'input': 'maxPool2dInput'}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.3125, 81.75],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float16 4D tensor options.windowDimensions',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {'options': {'windowDimensions': [3, 3]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.3125, 99.3125, 66.0625, 99.3125, 99.3125, 72.125, 97.875, 72.125,
            72.125, 81.75, 72.4375, 72.4375, 81.75, 72.875, 72.4375, 80.3125,
            72.875, 72.3125
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float16 4D tensor options.padding',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {'options': {'padding': [1, 0, 0, 1]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.3125, 99.3125, 99.3125, 99.3125, 81.75, 72.875, 81.75, 72.875
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float16 4D tensor options.strides',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'strides': [2, 2]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.3125, 66.0625, 97.875, 72.125, 81.75, 72.4375, 80.3125, 72.3125
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float16 4D tensor options.dilations',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'dilations': [2, 2]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [89, 72.3125],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float16 4D tensor options.layout=nchw',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments':
            [{'input': 'maxPool2dInput'}, {'options': {'layout': 'nchw'}}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.3125, 81.75],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float16 4D tensor options.layout=nhwc',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,         -76.75,    -45.71875,  50.21875,  -61.3125,
            -52.90625,  -4.015625, -44.65625,  -94.5625,  -97.875,
            46.28125,   81.75,     99.3125,    5.4296875, -10.0546875,
            -29.234375, 9.7421875, 72.4375,    -39.03125, -59.34375,
            75.0625,    39.1875,   12.8203125, -66,       -33,
            -4.203125,  38.6875,   -60.53125,  66.0625,   55.875,
            97.875,     80.3125,   -8.734375,  72.875,    -53.40625,
            -46.59375,  72.125,    20.5,       -40.4375,  -31.125,
            -35.6875,   -57.28125, -87.625,    -26.625,   38.875,
            15.9375,    39.375,    -78.75,     7.4296875, 72.3125
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments':
            [{'input': 'maxPool2dInput'}, {'options': {'layout': 'nhwc'}}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.3125, 81.75],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float16 4D tensor options.roundingType=floor',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.3125, 9.7421875, 99.3125, 72.125, 81.75, 72.4375, 81.75, 72.4375
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float16 4D tensor options.roundingType=ceil',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.3125, 9.7421875, -39.03125, 99.3125, 72.125, 66.0625, 97.875,
            72.125, 7.4296875, 81.75, 72.4375, -59.34375, 81.75, 72.4375,
            55.875, 80.3125, 72.3125, 72.3125
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'maxPool2d float16 4D tensor options.outputSizes ignores options.roundingType=floor',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor',
              'outputSizes': [3, 3]
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.3125, 9.7421875, -39.03125, 99.3125, 72.125, 66.0625, 97.875,
            72.125, 7.4296875, 81.75, 72.4375, -59.34375, 81.75, 72.4375,
            55.875, 80.3125, 72.3125, 72.3125
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'maxPool2d float16 4D tensor options.outputSizes ignores options.roundingType=ceil',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89,        -45.71875,  -61.3125,    -4.015625, -94.5625,
            46.28125,  99.3125,    -10.0546875, 9.7421875, -39.03125,
            75.0625,   12.8203125, -33,         38.6875,   66.0625,
            97.875,    -8.734375,  -53.40625,   72.125,    -40.4375,
            -35.6875,  -87.625,    38.875,      39.375,    7.4296875,
            -76.75,    50.21875,   -52.90625,   -44.65625, -97.875,
            81.75,     5.4296875,  -29.234375,  72.4375,   -59.34375,
            39.1875,   -66,        -4.203125,   -60.53125, 55.875,
            80.3125,   72.875,     -46.59375,   20.5,      -31.125,
            -57.28125, -26.625,    15.9375,     -78.75,    72.3125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil',
              'outputSizes': [2, 2]
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.3125, 9.7421875, 99.3125, 72.125, 81.75, 72.4375, 81.75, 72.4375
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'maxPool2d float16 4D tensor options.dilations with options.strides',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            34.6875,    -24.703125, -60.4375,    35.9375,     60.90625,
            79.4375,    -77.875,    54.1875,     -21.578125,  -49.09375,
            78.5625,    74.3125,    -87.9375,    79.8125,     11.6796875,
            -12.15625,  -22.328125, 33.09375,    8.640625,    47.03125,
            95.8125,    -88,        -98.5625,    16.15625,    21.390625,
            95.125,     -40.8125,   -88.6875,    -40.78125,   67.875,
            -50.34375,  -96.5625,   12.5078125,  -6.63671875, -44.8125,
            80.25,      -65.6875,   -14.8828125, -66.5625,    20.078125,
            -27.953125, -56.625,    82.75,       93.3125,     -34.0625,
            -22.875,    73.25,      41.125,      -19.75,      80.0625,
            -69.875,    14.015625,  -61.375,     51.53125,    43.53125,
            -89.5625,   51.4375,    73.9375,     -80.25,      94.75,
            95.25,      58.125,     19.890625,   -70,         63.40625,
            -54.125,    -97.25,     -60.65625,   -31.046875,  -14.6484375,
            -63.75,     47.34375,   85.5625,     -53.375,     -70.875,
            47.34375,   83.375,     7.36328125,  46.84375,    98.125,
            -43.9375,   50.34375,   37.46875,    -54.53125,   -6.2109375,
            34.40625,   11.8203125, 72.4375,     86.375,      4.1484375,
            88.3125,    98.3125,    -66.75,      58.0625,     -51.15625,
            -17.765625, 9.9609375,  -52.75
          ],
          'descriptor': {shape: [1, 7, 7, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'dilations': [1, 1],
              'layout': 'nhwc'
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            60.90625, 79.4375, 95.8125, 79.4375, 78.5625, 95.125, 82.75,
            93.3125, 95.8125, 80.25, 43.53125, 95.125, 95.25, 94.75, 95.25,
            98.125, 63.40625, 98.125
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(
    maxPool2dTests, buildAndExecuteGraph, getPrecisionTolerance);
