// META: title=test WebNN API matmul operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-matmul
// Compute the matrix product of two input tensors.
// MLOperand matmul(MLOperand a, MLOperand b);


const matmulTests = [
  {
    'name': 'matmul float32 2D and 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            10.43066120147705, 24.467519760131836, 55.887577056884766,
            3.2410826683044434, 87.58891296386719, 22.40154457092285,
            79.85144805908203, 99.64449310302734, 24.740541458129883,
            65.9624252319336, 38.136077880859375, 87.11140441894531
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            88.1700439453125,   78.4012680053711,   14.819003105163574,
            3.6923038959503174, 45.906429290771484, 43.083919525146484,
            47.199466705322266, 60.92521667480469,  8.162760734558105,
            20.333263397216797, 20.438398361206055, 27.0194091796875,
            15.601424217224121, 87.46969604492188,  65.79554748535156,
            69.31697082519531,  31.984439849853516, 12.291812896728516,
            13.304834365844727, 85.26705169677734
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            3340.7431640625, 3586.344482421875, 2557.025634765625,
            5169.8271484375, 4929.85009765625, 17226.955078125, 13269.03515625,
            5133.4072265625, 8816.5986328125, 18226.65234375, 11841.033203125,
            8869.705078125, 6051.1396484375, 5124.5390625, 12413.8984375
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'matmul float32 3D and 3D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            56.46701431274414,  99.86045837402344,  71.054931640625,
            32.454383850097656, 17.310747146606445, 2.586275100708008,
            92.31499481201172,  96.9758529663086,   26.472131729125977,
            77.67031860351562,  29.278789520263672, 82.12142181396484,
            89.89308166503906,  82.49795532226562,  64.36865997314453,
            23.75928497314453,  6.670266628265381,  81.55583190917969,
            16.142963409423828, 57.45134735107422,  26.826417922973633,
            85.02970123291016,  36.1988639831543,   89.60960388183594
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            70.38780212402344,  25.48917579650879,  28.254196166992188,
            84.5148696899414,   58.873416900634766, 46.252838134765625,
            24.897335052490234, 44.0944938659668,   51.86564636230469,
            1.123237133026123,  4.187554836273193,  71.24649810791016,
            16.034526824951172, 23.677297592163086, 61.27727508544922,
            65.15946197509766,  58.39249801635742,  70.12741088867188,
            9.11972713470459,   24.179977416992188, 84.4263687133789,
            78.55551147460938,  38.48297119140625,  80.6541519165039,
            57.152122497558594, 46.109710693359375, 28.41227912902832,
            90.92566680908203,  94.66068267822266,  61.77287673950195,
            60.324859619140625, 70.97433471679688,  10.631051063537598,
            55.61628341674805,  35.052310943603516, 22.12839126586914,
            38.19757843017578,  78.17564392089844,  62.57684326171875,
            88.35256958007812
          ],
          'descriptor': {shape: [2, 4, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            11005.6748046875, 10883.064453125,  9414.0009765625,
            11929.9931640625, 8575.3720703125,  8043.556640625,
            12745.41015625,   8884.0341796875,  4667.3173828125,
            9023.7333984375,  10929.3818359375, 9489.8232421875,
            10401.216796875,  7707.87744140625, 5425.5654296875,
            15802.0888671875, 14881.6220703125, 13502.23828125,
            20126.2890625,    14589.1806640625, 6568.7900390625,
            6181.41162109375, 12335.123046875,  12751.05078125,
            11060.98046875,   10352.16015625,   10515.310546875,
            16153.86328125,   17833.36328125,   15971.80859375
          ],
          'descriptor': {shape: [2, 3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'matmul float32 4D and 4D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            56.46701431274414,  99.86045837402344,  71.054931640625,
            32.454383850097656, 17.310747146606445, 2.586275100708008,
            92.31499481201172,  96.9758529663086,   26.472131729125977,
            77.67031860351562,  29.278789520263672, 82.12142181396484,
            89.89308166503906,  82.49795532226562,  64.36865997314453,
            23.75928497314453,  6.670266628265381,  81.55583190917969,
            16.142963409423828, 57.45134735107422,  26.826417922973633,
            85.02970123291016,  36.1988639831543,   89.60960388183594
          ],
          'descriptor': {shape: [2, 1, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            70.38780212402344,  25.48917579650879,  28.254196166992188,
            84.5148696899414,   58.873416900634766, 46.252838134765625,
            24.897335052490234, 44.0944938659668,   51.86564636230469,
            1.123237133026123,  4.187554836273193,  71.24649810791016,
            16.034526824951172, 23.677297592163086, 61.27727508544922,
            65.15946197509766,  58.39249801635742,  70.12741088867188,
            9.11972713470459,   24.179977416992188, 84.4263687133789,
            78.55551147460938,  38.48297119140625,  80.6541519165039,
            57.152122497558594, 46.109710693359375, 28.41227912902832,
            90.92566680908203,  94.66068267822266,  61.77287673950195,
            60.324859619140625, 70.97433471679688,  10.631051063537598,
            55.61628341674805,  35.052310943603516, 22.12839126586914,
            38.19757843017578,  78.17564392089844,  62.57684326171875,
            88.35256958007812
          ],
          'descriptor': {shape: [2, 1, 4, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            11005.6748046875, 10883.064453125,  9414.0009765625,
            11929.9931640625, 8575.3720703125,  8043.556640625,
            12745.41015625,   8884.0341796875,  4667.3173828125,
            9023.7333984375,  10929.3818359375, 9489.8232421875,
            10401.216796875,  7707.87744140625, 5425.5654296875,
            15802.0888671875, 14881.6220703125, 13502.23828125,
            20126.2890625,    14589.1806640625, 6568.7900390625,
            6181.41162109375, 12335.123046875,  12751.05078125,
            11060.98046875,   10352.16015625,   10515.310546875,
            16153.86328125,   17833.36328125,   15971.80859375
          ],
          'descriptor': {shape: [2, 1, 3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'matmul float32 3D and 3D (broadcast) tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            56.46701431274414,  99.86045837402344,  71.054931640625,
            32.454383850097656, 17.310747146606445, 2.586275100708008,
            92.31499481201172,  96.9758529663086,   26.472131729125977,
            77.67031860351562,  29.278789520263672, 82.12142181396484,
            89.89308166503906,  82.49795532226562,  64.36865997314453,
            23.75928497314453,  6.670266628265381,  81.55583190917969,
            16.142963409423828, 57.45134735107422,  26.826417922973633,
            85.02970123291016,  36.1988639831543,   89.60960388183594
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            27.829805374145508, 83.1454849243164, 34.41289520263672,
            83.20379638671875
          ],
          'descriptor': {shape: [1, 4, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            15019.9462890625, 11942.376953125, 15035.0322265625,
            13553.013671875, 12302.328125, 16517.9765625
          ],
          'descriptor': {shape: [2, 3, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'matmul float32 3D and 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            56.46701431274414,  99.86045837402344,  71.054931640625,
            32.454383850097656, 17.310747146606445, 2.586275100708008,
            92.31499481201172,  96.9758529663086,   26.472131729125977,
            77.67031860351562,  29.278789520263672, 82.12142181396484,
            89.89308166503906,  82.49795532226562,  64.36865997314453,
            23.75928497314453,  6.670266628265381,  81.55583190917969,
            16.142963409423828, 57.45134735107422,  26.826417922973633,
            85.02970123291016,  36.1988639831543,   89.60960388183594
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            27.829805374145508, 83.1454849243164, 34.41289520263672,
            83.20379638671875
          ],
          'descriptor': {shape: [4, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            15019.9462890625, 11942.376953125, 15035.0322265625,
            13553.013671875, 12302.328125, 16517.9765625
          ],
          'descriptor': {shape: [2, 3, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'matmul float32 4D and 4D (broadcast) tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            33.75957107543945,  97.24552917480469,  83.7085189819336,
            64.53984069824219,  29.57938003540039,  17.19923973083496,
            67.94749450683594,  97.45838165283203,  54.449283599853516,
            29.552200317382812, 51.99970245361328,  36.03101348876953,
            9.701058387756348,  27.04842185974121,  6.020919322967529,
            22.940902709960938, 53.1243896484375,   15.292234420776367,
            48.21302795410156,  87.40799713134766,  51.34442138671875,
            21.1557559967041,   27.589487075805664, 58.412384033203125,
            5.963276386260986,  84.74938201904297,  55.45738220214844,
            50.858699798583984, 23.763574600219727, 62.330928802490234,
            35.774959564208984, 17.340242385864258, 29.16901397705078,
            23.191360473632812, 27.060928344726562, 1.2828527688980103,
            8.720425605773926,  48.45281219482422,  99.0130386352539,
            65.86412048339844,  92.69683074951172,  85.43540954589844,
            37.49127960205078,  51.397132873535156, 53.19015121459961,
            38.33119201660156,  75.20586395263672,  3.8537938594818115
          ],
          'descriptor': {shape: [2, 2, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            88.1700439453125,   78.4012680053711,   14.819003105163574,
            3.6923038959503174, 45.906429290771484, 43.083919525146484,
            47.199466705322266, 60.92521667480469,  8.162760734558105,
            20.333263397216797, 20.438398361206055, 27.0194091796875,
            15.601424217224121, 87.46969604492188,  65.79554748535156,
            69.31697082519531,  31.984439849853516, 12.291812896728516,
            13.304834365844727, 85.26705169677734
          ],
          'descriptor': {shape: [1, 1, 4, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            13350.8759765625,  11562.755859375,   8524.271484375,
            9099.0927734375,   14537.8701171875,  11493.283203125,
            8083.90869140625,  3744.22216796875,  7489.62353515625,
            14488.2314453125,  9634.3720703125,   8221.173828125,
            3861.51416015625,  5470.0556640625,   9594.072265625,
            3733.946533203125, 2933.679931640625, 2167.611083984375,
            1088.48193359375,  3347.576416015625, 12387.083984375,
            8985.1884765625,   3545.52783203125,  5701.10595703125,
            13374.9169921875,  10051.3671875,     7637.7470703125,
            3198.221435546875, 3552.6796875,      9583.1220703125,
            8835.94921875,     7592.7666015625,   6742.10400390625,
            6241.31396484375,  9982.404296875,    6713.85205078125,
            6326.3173828125,   4920.9609375,      3956.46875,
            6190.67626953125,  4213.013671875,    4153.708984375,
            2283.152099609375, 2681.085693359375, 3700.47509765625,
            9445.5869140625,   7752.5400390625,   5435.56005859375,
            9964.6591796875,   13516.18359375,    16182.931640625,
            13956.9560546875,  7795.52685546875,  5002.8349609375,
            12841.802734375,   8145.45654296875,  8134.66650390625,
            4344.25,           7138.79052734375,  8497.98046875
          ],
          'descriptor': {shape: [2, 2, 3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'matmul float32 4D and 3D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            33.75957107543945,  97.24552917480469,  83.7085189819336,
            64.53984069824219,  29.57938003540039,  17.19923973083496,
            67.94749450683594,  97.45838165283203,  54.449283599853516,
            29.552200317382812, 51.99970245361328,  36.03101348876953,
            9.701058387756348,  27.04842185974121,  6.020919322967529,
            22.940902709960938, 53.1243896484375,   15.292234420776367,
            48.21302795410156,  87.40799713134766,  51.34442138671875,
            21.1557559967041,   27.589487075805664, 58.412384033203125,
            5.963276386260986,  84.74938201904297,  55.45738220214844,
            50.858699798583984, 23.763574600219727, 62.330928802490234,
            35.774959564208984, 17.340242385864258, 29.16901397705078,
            23.191360473632812, 27.060928344726562, 1.2828527688980103,
            8.720425605773926,  48.45281219482422,  99.0130386352539,
            65.86412048339844,  92.69683074951172,  85.43540954589844,
            37.49127960205078,  51.397132873535156, 53.19015121459961,
            38.33119201660156,  75.20586395263672,  3.8537938594818115
          ],
          'descriptor': {shape: [2, 2, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            70.38780212402344,  25.48917579650879,  28.254196166992188,
            84.5148696899414,   58.873416900634766, 46.252838134765625,
            24.897335052490234, 44.0944938659668,   51.86564636230469,
            1.123237133026123,  4.187554836273193,  71.24649810791016,
            16.034526824951172, 23.677297592163086, 61.27727508544922,
            65.15946197509766,  58.39249801635742,  70.12741088867188,
            9.11972713470459,   24.179977416992188, 84.4263687133789,
            78.55551147460938,  38.48297119140625,  80.6541519165039,
            57.152122497558594, 46.109710693359375, 28.41227912902832,
            90.92566680908203,  94.66068267822266,  61.77287673950195,
            60.324859619140625, 70.97433471679688,  10.631051063537598,
            55.61628341674805,  35.052310943603516, 22.12839126586914,
            38.19757843017578,  78.17564392089844,  62.57684326171875,
            88.35256958007812
          ],
          'descriptor': {shape: [2, 4, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            11430.05859375,    13014.2392578125,  11110.080078125,
            10467.46484375,    8786.7724609375,   9512.4111328125,
            11714.0283203125,  9518.1435546875,   5889.55419921875,
            8280.9365234375,   7764.95166015625,  7932.3759765625,
            6202.0625,         7694.32373046875,  7296.4384765625,
            2937.076416015625, 2834.196533203125, 4690.14990234375,
            5113.283203125,    4463.22998046875,  10032.8623046875,
            11368.3623046875,  10780.572265625,   13883.421875,
            13393.5166015625,  8267.212890625,    8823.828125,
            8759.2177734375,   11333.4462890625,  10369.25390625,
            7885.7978515625,   9182.943359375,    8361.2900390625,
            6676.46533203125,  5074.3115234375,   5835.33837890625,
            5718.96875,        5209.5322265625,   6246.40185546875,
            4080.533935546875, 3322.71826171875,  3323.802490234375,
            2370.630615234375, 4320.47900390625,  3432.568115234375,
            10400.7939453125,  11604.9267578125,  10942.7744140625,
            14918.220703125,   12781.3603515625,  15164.451171875,
            14333.4267578125,  15752.0966796875,  20865.154296875,
            16430.63671875,    10880.1533203125,  10752.34765625,
            6632.99462890625,  12342.2919921875,  8384.3896484375
          ],
          'descriptor': {shape: [2, 2, 3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'matmul float32 5D and 5D tensors, broadcast the two leftmost dimensions of inputB',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            33.75957107543945,  97.24552917480469,  83.7085189819336,
            64.53984069824219,  29.57938003540039,  17.19923973083496,
            67.94749450683594,  97.45838165283203,  54.449283599853516,
            29.552200317382812, 51.99970245361328,  36.03101348876953,
            9.701058387756348,  27.04842185974121,  6.020919322967529,
            22.940902709960938, 53.1243896484375,   15.292234420776367,
            48.21302795410156,  87.40799713134766,  51.34442138671875,
            21.1557559967041,   27.589487075805664, 58.412384033203125,
            5.963276386260986,  84.74938201904297,  55.45738220214844,
            50.858699798583984, 23.763574600219727, 62.330928802490234,
            35.774959564208984, 17.340242385864258, 29.16901397705078,
            23.191360473632812, 27.060928344726562, 1.2828527688980103,
            8.720425605773926,  48.45281219482422,  99.0130386352539,
            65.86412048339844,  92.69683074951172,  85.43540954589844,
            37.49127960205078,  51.397132873535156, 53.19015121459961,
            38.33119201660156,  75.20586395263672,  3.8537938594818115
          ],
          'descriptor': {shape: [2, 2, 1, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            88.1700439453125,   78.4012680053711,   14.819003105163574,
            3.6923038959503174, 45.906429290771484, 43.083919525146484,
            47.199466705322266, 60.92521667480469,  8.162760734558105,
            20.333263397216797, 20.438398361206055, 27.0194091796875,
            15.601424217224121, 87.46969604492188,  65.79554748535156,
            69.31697082519531,  31.984439849853516, 12.291812896728516,
            13.304834365844727, 85.26705169677734
          ],
          'descriptor': {shape: [1, 1, 1, 4, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            13350.8759765625,  11562.755859375,   8524.271484375,
            9099.0927734375,   14537.8701171875,  11493.283203125,
            8083.90869140625,  3744.22216796875,  7489.62353515625,
            14488.2314453125,  9634.3720703125,   8221.173828125,
            3861.51416015625,  5470.0556640625,   9594.072265625,
            3733.946533203125, 2933.679931640625, 2167.611083984375,
            1088.48193359375,  3347.576416015625, 12387.083984375,
            8985.1884765625,   3545.52783203125,  5701.10595703125,
            13374.9169921875,  10051.3671875,     7637.7470703125,
            3198.221435546875, 3552.6796875,      9583.1220703125,
            8835.94921875,     7592.7666015625,   6742.10400390625,
            6241.31396484375,  9982.404296875,    6713.85205078125,
            6326.3173828125,   4920.9609375,      3956.46875,
            6190.67626953125,  4213.013671875,    4153.708984375,
            2283.152099609375, 2681.085693359375, 3700.47509765625,
            9445.5869140625,   7752.5400390625,   5435.56005859375,
            9964.6591796875,   13516.18359375,    16182.931640625,
            13956.9560546875,  7795.52685546875,  5002.8349609375,
            12841.802734375,   8145.45654296875,  8134.66650390625,
            4344.25,           7138.79052734375,  8497.98046875
          ],
          'descriptor': {shape: [2, 2, 1, 3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'matmul float32 5D and 5D tensors, broadcast the leftmost dimensions of inputB',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            33.75957107543945,  97.24552917480469,  83.7085189819336,
            64.53984069824219,  29.57938003540039,  17.19923973083496,
            67.94749450683594,  97.45838165283203,  54.449283599853516,
            29.552200317382812, 51.99970245361328,  36.03101348876953,
            9.701058387756348,  27.04842185974121,  6.020919322967529,
            22.940902709960938, 53.1243896484375,   15.292234420776367,
            48.21302795410156,  87.40799713134766,  51.34442138671875,
            21.1557559967041,   27.589487075805664, 58.412384033203125,
            5.963276386260986,  84.74938201904297,  55.45738220214844,
            50.858699798583984, 23.763574600219727, 62.330928802490234,
            35.774959564208984, 17.340242385864258, 29.16901397705078,
            23.191360473632812, 27.060928344726562, 1.2828527688980103,
            8.720425605773926,  48.45281219482422,  99.0130386352539,
            65.86412048339844,  92.69683074951172,  85.43540954589844,
            37.49127960205078,  51.397132873535156, 53.19015121459961,
            38.33119201660156,  75.20586395263672,  3.8537938594818115
          ],
          'descriptor': {shape: [2, 2, 1, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            88.1700439453125,   78.4012680053711,   14.819003105163574,
            3.6923038959503174, 45.906429290771484, 43.083919525146484,
            47.199466705322266, 60.92521667480469,  8.162760734558105,
            20.333263397216797, 20.438398361206055, 27.0194091796875,
            15.601424217224121, 87.46969604492188,  65.79554748535156,
            69.31697082519531,  31.984439849853516, 12.291812896728516,
            13.304834365844727, 85.26705169677734,  88.1700439453125,
            78.4012680053711,   14.819003105163574, 3.6923038959503174,
            45.906429290771484, 43.083919525146484, 47.199466705322266,
            60.92521667480469,  8.162760734558105,  20.333263397216797,
            20.438398361206055, 27.0194091796875,   15.601424217224121,
            87.46969604492188,  65.79554748535156,  69.31697082519531,
            31.984439849853516, 12.291812896728516, 13.304834365844727,
            85.26705169677734
          ],
          'descriptor': {shape: [1, 2, 1, 4, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            13350.8759765625,  11562.755859375,   8524.271484375,
            9099.0927734375,   14537.8701171875,  11493.283203125,
            8083.90869140625,  3744.22216796875,  7489.62353515625,
            14488.2314453125,  9634.3720703125,   8221.173828125,
            3861.51416015625,  5470.0556640625,   9594.072265625,
            3733.946533203125, 2933.679931640625, 2167.611083984375,
            1088.48193359375,  3347.576416015625, 12387.083984375,
            8985.1884765625,   3545.52783203125,  5701.10595703125,
            13374.9169921875,  10051.3671875,     7637.7470703125,
            3198.221435546875, 3552.6796875,      9583.1220703125,
            8835.94921875,     7592.7666015625,   6742.10400390625,
            6241.31396484375,  9982.404296875,    6713.85205078125,
            6326.3173828125,   4920.9609375,      3956.46875,
            6190.67626953125,  4213.013671875,    4153.708984375,
            2283.152099609375, 2681.085693359375, 3700.47509765625,
            9445.5869140625,   7752.5400390625,   5435.56005859375,
            9964.6591796875,   13516.18359375,    16182.931640625,
            13956.9560546875,  7795.52685546875,  5002.8349609375,
            12841.802734375,   8145.45654296875,  8134.66650390625,
            4344.25,           7138.79052734375,  8497.98046875
          ],
          'descriptor': {shape: [2, 2, 1, 3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'matmul float32 5D and 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            33.75957107543945,  97.24552917480469,  83.7085189819336,
            64.53984069824219,  29.57938003540039,  17.19923973083496,
            67.94749450683594,  97.45838165283203,  54.449283599853516,
            29.552200317382812, 51.99970245361328,  36.03101348876953,
            9.701058387756348,  27.04842185974121,  6.020919322967529,
            22.940902709960938, 53.1243896484375,   15.292234420776367,
            48.21302795410156,  87.40799713134766,  51.34442138671875,
            21.1557559967041,   27.589487075805664, 58.412384033203125,
            5.963276386260986,  84.74938201904297,  55.45738220214844,
            50.858699798583984, 23.763574600219727, 62.330928802490234,
            35.774959564208984, 17.340242385864258, 29.16901397705078,
            23.191360473632812, 27.060928344726562, 1.2828527688980103,
            8.720425605773926,  48.45281219482422,  99.0130386352539,
            65.86412048339844,  92.69683074951172,  85.43540954589844,
            37.49127960205078,  51.397132873535156, 53.19015121459961,
            38.33119201660156,  75.20586395263672,  3.8537938594818115
          ],
          'descriptor': {shape: [2, 2, 1, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            88.1700439453125,   78.4012680053711,   14.819003105163574,
            3.6923038959503174, 45.906429290771484, 43.083919525146484,
            47.199466705322266, 60.92521667480469,  8.162760734558105,
            20.333263397216797, 20.438398361206055, 27.0194091796875,
            15.601424217224121, 87.46969604492188,  65.79554748535156,
            69.31697082519531,  31.984439849853516, 12.291812896728516,
            13.304834365844727, 85.26705169677734
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            13350.8759765625,  11562.755859375,   8524.271484375,
            9099.0927734375,   14537.8701171875,  11493.283203125,
            8083.90869140625,  3744.22216796875,  7489.62353515625,
            14488.2314453125,  9634.3720703125,   8221.173828125,
            3861.51416015625,  5470.0556640625,   9594.072265625,
            3733.946533203125, 2933.679931640625, 2167.611083984375,
            1088.48193359375,  3347.576416015625, 12387.083984375,
            8985.1884765625,   3545.52783203125,  5701.10595703125,
            13374.9169921875,  10051.3671875,     7637.7470703125,
            3198.221435546875, 3552.6796875,      9583.1220703125,
            8835.94921875,     7592.7666015625,   6742.10400390625,
            6241.31396484375,  9982.404296875,    6713.85205078125,
            6326.3173828125,   4920.9609375,      3956.46875,
            6190.67626953125,  4213.013671875,    4153.708984375,
            2283.152099609375, 2681.085693359375, 3700.47509765625,
            9445.5869140625,   7752.5400390625,   5435.56005859375,
            9964.6591796875,   13516.18359375,    16182.931640625,
            13956.9560546875,  7795.52685546875,  5002.8349609375,
            12841.802734375,   8145.45654296875,  8134.66650390625,
            4344.25,           7138.79052734375,  8497.98046875
          ],
          'descriptor': {shape: [2, 2, 1, 3, 5], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'matmul float16 2D and 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            10.4296875, 24.46875, 55.875, 3.240234375, 87.5625, 22.40625,
            79.875, 99.625, 24.734375, 65.9375, 38.125, 87.125
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            88.1875,  78.375,    14.8203125, 3.69140625, 45.90625,
            43.09375, 47.1875,   60.9375,    8.1640625,  20.328125,
            20.4375,  27.015625, 15.6015625, 87.5,       65.8125,
            69.3125,  31.984375, 12.2890625, 13.3046875, 85.25
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            3340, 3586, 2558, 5172, 4928, 17232, 13264, 5132, 8824, 18224,
            11840, 8864, 6052, 5124, 12416
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'matmul float16 3D and 3D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            56.46875,  99.875,  71.0625,   32.46875,  17.3125,  2.5859375,
            92.3125,   97,      26.46875,  77.6875,   29.28125, 82.125,
            89.875,    82.5,    64.375,    23.765625, 6.671875, 81.5625,
            16.140625, 57.4375, 26.828125, 85,        36.1875,  89.625
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            70.375,    25.484375, 28.25,      84.5,        58.875,   46.25,
            24.890625, 44.09375,  51.875,     1.123046875, 4.1875,   71.25,
            16.03125,  23.671875, 61.28125,   65.1875,     58.40625, 70.125,
            9.1171875, 24.1875,   84.4375,    78.5625,     38.46875, 80.625,
            57.15625,  46.125,    28.40625,   90.9375,     94.6875,  61.78125,
            60.3125,   71,        10.6328125, 55.625,      35.0625,  22.125,
            38.1875,   78.1875,   62.5625,    88.375
          ],
          'descriptor': {shape: [2, 4, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            11008, 10888, 9416,  11928, 8576,  8048, 12752, 8888,
            4668,  9024,  10928, 9488,  10400, 7708, 5428,  15800,
            14880, 13504, 20128, 14592, 6568,  6180, 12336, 12752,
            11064, 10352, 10512, 16152, 17824, 15976
          ],
          'descriptor': {shape: [2, 3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'matmul float16 4D and 4D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            56.46875,  99.875,  71.0625,   32.46875,  17.3125,  2.5859375,
            92.3125,   97,      26.46875,  77.6875,   29.28125, 82.125,
            89.875,    82.5,    64.375,    23.765625, 6.671875, 81.5625,
            16.140625, 57.4375, 26.828125, 85,        36.1875,  89.625
          ],
          'descriptor': {shape: [2, 1, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            70.375,    25.484375, 28.25,      84.5,        58.875,   46.25,
            24.890625, 44.09375,  51.875,     1.123046875, 4.1875,   71.25,
            16.03125,  23.671875, 61.28125,   65.1875,     58.40625, 70.125,
            9.1171875, 24.1875,   84.4375,    78.5625,     38.46875, 80.625,
            57.15625,  46.125,    28.40625,   90.9375,     94.6875,  61.78125,
            60.3125,   71,        10.6328125, 55.625,      35.0625,  22.125,
            38.1875,   78.1875,   62.5625,    88.375
          ],
          'descriptor': {shape: [2, 1, 4, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            11008, 10888, 9416,  11928, 8576,  8048, 12752, 8888,
            4668,  9024,  10928, 9488,  10400, 7708, 5428,  15800,
            14880, 13504, 20128, 14592, 6568,  6180, 12336, 12752,
            11064, 10352, 10512, 16152, 17824, 15976
          ],
          'descriptor': {shape: [2, 1, 3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'matmul float16 3D and 3D (broadcast) tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            56.46875,  99.875,  71.0625,   32.46875,  17.3125,  2.5859375,
            92.3125,   97,      26.46875,  77.6875,   29.28125, 82.125,
            89.875,    82.5,    64.375,    23.765625, 6.671875, 81.5625,
            16.140625, 57.4375, 26.828125, 85,        36.1875,  89.625
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [27.828125, 83.125, 34.40625, 83.1875],
          'descriptor': {shape: [1, 4, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [15016, 11944, 15032, 13552, 12296, 16512],
          'descriptor': {shape: [2, 3, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'matmul float16 3D and 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            56.46875,  99.875,  71.0625,   32.46875,  17.3125,  2.5859375,
            92.3125,   97,      26.46875,  77.6875,   29.28125, 82.125,
            89.875,    82.5,    64.375,    23.765625, 6.671875, 81.5625,
            16.140625, 57.4375, 26.828125, 85,        36.1875,  89.625
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [27.828125, 83.125, 34.40625, 83.1875],
          'descriptor': {shape: [4, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [15016, 11944, 15032, 13552, 12296, 16512],
          'descriptor': {shape: [2, 3, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'matmul float16 4D and 4D (broadcast) tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            33.75,       97.25,    83.6875,    64.5625,   29.578125,
            17.203125,   67.9375,  97.4375,    54.4375,   29.546875,
            52,          36.03125, 9.703125,   27.046875, 6.01953125,
            22.9375,     53.125,   15.2890625, 48.21875,  87.4375,
            51.34375,    21.15625, 27.59375,   58.40625,  5.96484375,
            84.75,       55.46875, 50.84375,   23.765625, 62.34375,
            35.78125,    17.34375, 29.171875,  23.1875,   27.0625,
            1.283203125, 8.71875,  48.4375,    99,        65.875,
            92.6875,     85.4375,  37.5,       51.40625,  53.1875,
            38.34375,    75.1875,  3.853515625
          ],
          'descriptor': {shape: [2, 2, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            88.1875,  78.375,    14.8203125, 3.69140625, 45.90625,
            43.09375, 47.1875,   60.9375,    8.1640625,  20.328125,
            20.4375,  27.015625, 15.6015625, 87.5,       65.8125,
            69.3125,  31.984375, 12.2890625, 13.3046875, 85.25
          ],
          'descriptor': {shape: [1, 1, 4, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            13352, 11560, 8528, 9104, 14536, 11488, 8080, 3744, 7492, 14488,
            9632,  8216,  3862, 5472, 9592,  3734,  2934, 2168, 1089, 3346,
            12392, 8984,  3546, 5704, 13376, 10048, 7636, 3198, 3554, 9584,
            8832,  7592,  6744, 6244, 9984,  6716,  6324, 4924, 3958, 6192,
            4212,  4152,  2284, 2682, 3700,  9448,  7752, 5436, 9968, 13520,
            16184, 13952, 7796, 5004, 12840, 8148,  8132, 4344, 7140, 8496
          ],
          'descriptor': {shape: [2, 2, 3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'matmul float16 4D and 3D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            33.75,       97.25,    83.6875,    64.5625,   29.578125,
            17.203125,   67.9375,  97.4375,    54.4375,   29.546875,
            52,          36.03125, 9.703125,   27.046875, 6.01953125,
            22.9375,     53.125,   15.2890625, 48.21875,  87.4375,
            51.34375,    21.15625, 27.59375,   58.40625,  5.96484375,
            84.75,       55.46875, 50.84375,   23.765625, 62.34375,
            35.78125,    17.34375, 29.171875,  23.1875,   27.0625,
            1.283203125, 8.71875,  48.4375,    99,        65.875,
            92.6875,     85.4375,  37.5,       51.40625,  53.1875,
            38.34375,    75.1875,  3.853515625
          ],
          'descriptor': {shape: [2, 2, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            70.375,    25.484375, 28.25,      84.5,        58.875,   46.25,
            24.890625, 44.09375,  51.875,     1.123046875, 4.1875,   71.25,
            16.03125,  23.671875, 61.28125,   65.1875,     58.40625, 70.125,
            9.1171875, 24.1875,   84.4375,    78.5625,     38.46875, 80.625,
            57.15625,  46.125,    28.40625,   90.9375,     94.6875,  61.78125,
            60.3125,   71,        10.6328125, 55.625,      35.0625,  22.125,
            38.1875,   78.1875,   62.5625,    88.375
          ],
          'descriptor': {shape: [2, 4, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            11432, 13016, 11112, 10464, 8784,  9512,  11712, 9520,  5888,
            8280,  7764,  7932,  6200,  7692,  7296,  2938,  2834,  4692,
            5112,  4464,  10032, 11368, 10784, 13880, 13400, 8264,  8824,
            8760,  11328, 10368, 7888,  9184,  8360,  6676,  5076,  5836,
            5720,  5212,  6248,  4082,  3322,  3324,  2370,  4320,  3432,
            10400, 11608, 10944, 14920, 12784, 15168, 14336, 15752, 20864,
            16432, 10880, 10752, 6632,  12344, 8384
          ],
          'descriptor': {shape: [2, 2, 3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'matmul float16 5D and 5D tensors, broadcast the two leftmost dimensions of inputB',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            33.75,       97.25,    83.6875,    64.5625,   29.578125,
            17.203125,   67.9375,  97.4375,    54.4375,   29.546875,
            52,          36.03125, 9.703125,   27.046875, 6.01953125,
            22.9375,     53.125,   15.2890625, 48.21875,  87.4375,
            51.34375,    21.15625, 27.59375,   58.40625,  5.96484375,
            84.75,       55.46875, 50.84375,   23.765625, 62.34375,
            35.78125,    17.34375, 29.171875,  23.1875,   27.0625,
            1.283203125, 8.71875,  48.4375,    99,        65.875,
            92.6875,     85.4375,  37.5,       51.40625,  53.1875,
            38.34375,    75.1875,  3.853515625
          ],
          'descriptor': {shape: [2, 2, 1, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            88.1875,  78.375,    14.8203125, 3.69140625, 45.90625,
            43.09375, 47.1875,   60.9375,    8.1640625,  20.328125,
            20.4375,  27.015625, 15.6015625, 87.5,       65.8125,
            69.3125,  31.984375, 12.2890625, 13.3046875, 85.25
          ],
          'descriptor': {shape: [1, 1, 1, 4, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            13352, 11560, 8528, 9104, 14536, 11488, 8080, 3744, 7492, 14488,
            9632,  8216,  3862, 5472, 9592,  3734,  2934, 2168, 1089, 3346,
            12392, 8984,  3546, 5704, 13376, 10048, 7636, 3198, 3554, 9584,
            8832,  7592,  6744, 6244, 9984,  6716,  6324, 4924, 3958, 6192,
            4212,  4152,  2284, 2682, 3700,  9448,  7752, 5436, 9968, 13520,
            16184, 13952, 7796, 5004, 12840, 8148,  8132, 4344, 7140, 8496
          ],
          'descriptor': {shape: [2, 2, 1, 3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'matmul float16 5D and 5D tensors, broadcast the leftmost dimensions of inputB',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            33.75,       97.25,    83.6875,    64.5625,   29.578125,
            17.203125,   67.9375,  97.4375,    54.4375,   29.546875,
            52,          36.03125, 9.703125,   27.046875, 6.01953125,
            22.9375,     53.125,   15.2890625, 48.21875,  87.4375,
            51.34375,    21.15625, 27.59375,   58.40625,  5.96484375,
            84.75,       55.46875, 50.84375,   23.765625, 62.34375,
            35.78125,    17.34375, 29.171875,  23.1875,   27.0625,
            1.283203125, 8.71875,  48.4375,    99,        65.875,
            92.6875,     85.4375,  37.5,       51.40625,  53.1875,
            38.34375,    75.1875,  3.853515625
          ],
          'descriptor': {shape: [2, 2, 1, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            88.1875,  78.375,    14.8203125, 3.69140625, 45.90625,
            43.09375, 47.1875,   60.9375,    8.1640625,  20.328125,
            20.4375,  27.015625, 15.6015625, 87.5,       65.8125,
            69.3125,  31.984375, 12.2890625, 13.3046875, 85.25,
            88.1875,  78.375,    14.8203125, 3.69140625, 45.90625,
            43.09375, 47.1875,   60.9375,    8.1640625,  20.328125,
            20.4375,  27.015625, 15.6015625, 87.5,       65.8125,
            69.3125,  31.984375, 12.2890625, 13.3046875, 85.25
          ],
          'descriptor': {shape: [1, 2, 1, 4, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            13352, 11560, 8528, 9104, 14536, 11488, 8080, 3744, 7492, 14488,
            9632,  8216,  3862, 5472, 9592,  3734,  2934, 2168, 1089, 3346,
            12392, 8984,  3546, 5704, 13376, 10048, 7636, 3198, 3554, 9584,
            8832,  7592,  6744, 6244, 9984,  6716,  6324, 4924, 3958, 6192,
            4212,  4152,  2284, 2682, 3700,  9448,  7752, 5436, 9968, 13520,
            16184, 13952, 7796, 5004, 12840, 8148,  8132, 4344, 7140, 8496
          ],
          'descriptor': {shape: [2, 2, 1, 3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'matmul float16 5D and 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            33.75,       97.25,    83.6875,    64.5625,   29.578125,
            17.203125,   67.9375,  97.4375,    54.4375,   29.546875,
            52,          36.03125, 9.703125,   27.046875, 6.01953125,
            22.9375,     53.125,   15.2890625, 48.21875,  87.4375,
            51.34375,    21.15625, 27.59375,   58.40625,  5.96484375,
            84.75,       55.46875, 50.84375,   23.765625, 62.34375,
            35.78125,    17.34375, 29.171875,  23.1875,   27.0625,
            1.283203125, 8.71875,  48.4375,    99,        65.875,
            92.6875,     85.4375,  37.5,       51.40625,  53.1875,
            38.34375,    75.1875,  3.853515625
          ],
          'descriptor': {shape: [2, 2, 1, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            88.1875,  78.375,    14.8203125, 3.69140625, 45.90625,
            43.09375, 47.1875,   60.9375,    8.1640625,  20.328125,
            20.4375,  27.015625, 15.6015625, 87.5,       65.8125,
            69.3125,  31.984375, 12.2890625, 13.3046875, 85.25
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'matmul',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            13352, 11560, 8528, 9104, 14536, 11488, 8080, 3744, 7492, 14488,
            9632,  8216,  3862, 5472, 9592,  3734,  2934, 2168, 1089, 3346,
            12392, 8984,  3546, 5704, 13376, 10048, 7636, 3198, 3554, 9584,
            8832,  7592,  6744, 6244, 9984,  6716,  6324, 4924, 3958, 6192,
            4212,  4152,  2284, 2682, 3700,  9448,  7752, 5436, 9968, 13520,
            16184, 13952, 7796, 5004, 12840, 8148,  8132, 4344, 7140, 8496
          ],
          'descriptor': {shape: [2, 2, 1, 3, 5], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(
    matmulTests, buildAndExecuteGraph, getPrecisionTolerance);
