// META: title=test WebNN API element-wise lesser operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-logical
// Compare if the values of the first input tensor is lesser, element-wise.
//
// MLOperand lesser(MLOperand a, MLOperand b);

const lesserTests = [
  {
    'name': 'lesser float32 0D scalar',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-0.5228080153465271],
          'descriptor': {shape: [], dataType: 'float32'}
        },
        'inputB': {
          'data': [0.8150388598442078],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {'data': [1], 'descriptor': {shape: [], dataType: 'uint8'}}
      }
    }
  },
  {
    'name': 'lesser float32 1D constant tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        },
        'inputB': {
          'data': [
            2.945375680923462,   3.730471611022949,    4.0253753662109375,
            -4.718355178833008,  6.7732744216918945,   -2.042813539505005,
            -6.526762008666992,  6.826299667358398,    -9.267172813415527,
            6.118423938751221,   -2.001732349395752,   1.779831051826477,
            9.660094261169434,   -2.7473158836364746,  -3.4345006942749023,
            -4.751097679138184,  -6.092621803283691,   -0.4334806203842163,
            -1.4069052934646606, -0.23742099106311798, -9.10597038269043,
            6.811779975891113,   -6.768326759338379,   -8.952353477478027
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [24], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float32 1D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            2.945375680923462,   3.730471611022949,    4.0253753662109375,
            -4.718355178833008,  6.7732744216918945,   -2.042813539505005,
            -6.526762008666992,  6.826299667358398,    -9.267172813415527,
            6.118423938751221,   -2.001732349395752,   1.779831051826477,
            9.660094261169434,   -2.7473158836364746,  -3.4345006942749023,
            -4.751097679138184,  -6.092621803283691,   -0.4334806203842163,
            -1.4069052934646606, -0.23742099106311798, -9.10597038269043,
            6.811779975891113,   -6.768326759338379,   -8.952353477478027
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [24], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float32 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            2.945375680923462,   3.730471611022949,    4.0253753662109375,
            -4.718355178833008,  6.7732744216918945,   -2.042813539505005,
            -6.526762008666992,  6.826299667358398,    -9.267172813415527,
            6.118423938751221,   -2.001732349395752,   1.779831051826477,
            9.660094261169434,   -2.7473158836364746,  -3.4345006942749023,
            -4.751097679138184,  -6.092621803283691,   -0.4334806203842163,
            -1.4069052934646606, -0.23742099106311798, -9.10597038269043,
            6.811779975891113,   -6.768326759338379,   -8.952353477478027
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [4, 6], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float32 3D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            2.945375680923462,   3.730471611022949,    4.0253753662109375,
            -4.718355178833008,  6.7732744216918945,   -2.042813539505005,
            -6.526762008666992,  6.826299667358398,    -9.267172813415527,
            6.118423938751221,   -2.001732349395752,   1.779831051826477,
            9.660094261169434,   -2.7473158836364746,  -3.4345006942749023,
            -4.751097679138184,  -6.092621803283691,   -0.4334806203842163,
            -1.4069052934646606, -0.23742099106311798, -9.10597038269043,
            6.811779975891113,   -6.768326759338379,   -8.952353477478027
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float32 4D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            2.945375680923462,   3.730471611022949,    4.0253753662109375,
            -4.718355178833008,  6.7732744216918945,   -2.042813539505005,
            -6.526762008666992,  6.826299667358398,    -9.267172813415527,
            6.118423938751221,   -2.001732349395752,   1.779831051826477,
            9.660094261169434,   -2.7473158836364746,  -3.4345006942749023,
            -4.751097679138184,  -6.092621803283691,   -0.4334806203842163,
            -1.4069052934646606, -0.23742099106311798, -9.10597038269043,
            6.811779975891113,   -6.768326759338379,   -8.952353477478027
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float32 5D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            2.945375680923462,   3.730471611022949,    4.0253753662109375,
            -4.718355178833008,  6.7732744216918945,   -2.042813539505005,
            -6.526762008666992,  6.826299667358398,    -9.267172813415527,
            6.118423938751221,   -2.001732349395752,   1.779831051826477,
            9.660094261169434,   -2.7473158836364746,  -3.4345006942749023,
            -4.751097679138184,  -6.092621803283691,   -0.4334806203842163,
            -1.4069052934646606, -0.23742099106311798, -9.10597038269043,
            6.811779975891113,   -6.768326759338379,   -8.952353477478027
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float32 broadcast 0D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-5.678369998931885],
          'descriptor': {shape: [], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0,
            1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float32 broadcast 1D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-5.678369998931885],
          'descriptor': {shape: [1], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0,
            1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float32 broadcast 2D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            3.5869946479797363, -2.853332042694092, -3.684652805328369,
            2.4055018424987793, -4.358371257781982, 5.5484747886657715
          ],
          'descriptor': {shape: [2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1,
            1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float32 broadcast 3D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -4.439523696899414, 2.7518322467803955, 3.635943651199341,
            -2.8089921474456787
          ],
          'descriptor': {shape: [2, 2, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 1, 1,
            0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float32 broadcast 4D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-5.678369998931885],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -1.147218942642212,  -8.409374237060547,  -2.2753310203552246,
            -0.5770801305770874, 8.171789169311523,   -0.907120943069458,
            5.2908453941345215,  -3.9134645462036133, 9.825095176696777,
            -8.931730270385742,  -3.457401752471924,  -7.331232070922852,
            1.232004165649414,   4.312077045440674,   1.2715545892715454,
            4.184540748596191,   -6.710920333862305,  3.0768423080444336,
            1.0030865669250488,  -9.076244354248047,  8.907161712646484,
            4.232614994049072,   2.1005890369415283,  -6.201345443725586
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0,
            1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'lesser float16 0D scalar',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-0.52294921875],
          'descriptor': {shape: [], dataType: 'float16'}
        },
        'inputB': {
          'data': [0.81494140625],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {'data': [1], 'descriptor': {shape: [], dataType: 'uint8'}}
      }
    }
  },
  {
    'name': 'lesser float16 1D constant tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        },
        'inputB': {
          'data': [
            2.9453125, 3.73046875,   4.0234375,     -4.71875,
            6.7734375, -2.04296875,  -6.52734375,   6.828125,
            -9.265625, 6.1171875,    -2.001953125,  1.7802734375,
            9.65625,   -2.748046875, -3.43359375,   -4.75,
            -6.09375,  -0.43359375,  -1.4072265625, -0.2374267578125,
            -9.109375, 6.8125,       -6.76953125,   -8.953125
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [24], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float16 1D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            2.9453125, 3.73046875,   4.0234375,     -4.71875,
            6.7734375, -2.04296875,  -6.52734375,   6.828125,
            -9.265625, 6.1171875,    -2.001953125,  1.7802734375,
            9.65625,   -2.748046875, -3.43359375,   -4.75,
            -6.09375,  -0.43359375,  -1.4072265625, -0.2374267578125,
            -9.109375, 6.8125,       -6.76953125,   -8.953125
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [24], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float16 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            2.9453125, 3.73046875,   4.0234375,     -4.71875,
            6.7734375, -2.04296875,  -6.52734375,   6.828125,
            -9.265625, 6.1171875,    -2.001953125,  1.7802734375,
            9.65625,   -2.748046875, -3.43359375,   -4.75,
            -6.09375,  -0.43359375,  -1.4072265625, -0.2374267578125,
            -9.109375, 6.8125,       -6.76953125,   -8.953125
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [4, 6], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float16 3D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            2.9453125, 3.73046875,   4.0234375,     -4.71875,
            6.7734375, -2.04296875,  -6.52734375,   6.828125,
            -9.265625, 6.1171875,    -2.001953125,  1.7802734375,
            9.65625,   -2.748046875, -3.43359375,   -4.75,
            -6.09375,  -0.43359375,  -1.4072265625, -0.2374267578125,
            -9.109375, 6.8125,       -6.76953125,   -8.953125
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float16 4D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            2.9453125, 3.73046875,   4.0234375,     -4.71875,
            6.7734375, -2.04296875,  -6.52734375,   6.828125,
            -9.265625, 6.1171875,    -2.001953125,  1.7802734375,
            9.65625,   -2.748046875, -3.43359375,   -4.75,
            -6.09375,  -0.43359375,  -1.4072265625, -0.2374267578125,
            -9.109375, 6.8125,       -6.76953125,   -8.953125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float16 5D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            2.9453125, 3.73046875,   4.0234375,     -4.71875,
            6.7734375, -2.04296875,  -6.52734375,   6.828125,
            -9.265625, 6.1171875,    -2.001953125,  1.7802734375,
            9.65625,   -2.748046875, -3.43359375,   -4.75,
            -6.09375,  -0.43359375,  -1.4072265625, -0.2374267578125,
            -9.109375, 6.8125,       -6.76953125,   -8.953125
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 1, 0, 0, 0, 0, 1, 0, 1, 1, 1,
            1, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float16 broadcast 0D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-5.6796875],
          'descriptor': {shape: [], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0,
            1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float16 broadcast 1D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-5.6796875],
          'descriptor': {shape: [1], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0,
            1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float16 broadcast 2D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            3.587890625, -2.853515625, -3.685546875, 2.40625, -4.359375,
            5.546875
          ],
          'descriptor': {shape: [2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 1,
            1, 0, 0, 0, 1, 1, 1, 1, 0, 0, 0, 1
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float16 broadcast 3D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [-4.44140625, 2.751953125, 3.63671875, -2.80859375],
          'descriptor': {shape: [2, 2, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            0, 1, 0, 1, 0, 1, 0, 1, 0, 1, 1, 1,
            0, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 1
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  },
  {
    'name': 'lesser float16 broadcast 4D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-5.6796875],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -1.1474609375, -8.40625,      -2.275390625, -0.5771484375,
            8.171875,      -0.9072265625, 5.2890625,    -3.9140625,
            9.828125,      -8.9296875,    -3.45703125,  -7.33203125,
            1.232421875,   4.3125,        1.271484375,  4.18359375,
            -6.7109375,    3.076171875,   1.0029296875, -9.078125,
            8.90625,       4.234375,      2.1015625,    -6.203125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'lesser',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0,
            1, 1, 1, 1, 0, 1, 1, 0, 1, 1, 1, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'uint8'}
        }
      }
    }
  }
];

webnn_conformance_test(lesserTests, buildAndExecuteGraph, getZeroULPTolerance);
