// META: title=test WebNN API layerNormalization operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-layernorm
// Normalize the input using Layer-Normalization.
//
// dictionary MLLayerNormalizationOptions {
//   MLOperand scale;
//   MLOperand bias;
//   sequence<[EnforceRange] unsigned long> axes;
//   double epsilon = 1e-5;
// };
//
// MLOperand layerNormalization(
//     MLOperand input, optional MLLayerNormalizationOptions options = {});


const getLayerNormPrecisionTolerance = () => {
  return {metricType: 'ULP', value: 14};
};

const layerNormTests = [
  {
    'name': 'layerNormalization float32 0D tensor default options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [-35.51446533203125],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput':
            {'data': [0], 'descriptor': {shape: [], dataType: 'float32'}}
      }
    }
  },
  {
    'name': 'layerNormalization float32 2D tensor default options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -5.712825298309326, 1.4681644439697266,     6.143280029296875,
            9.427258491516113,  2.0522539615631104,     -8.829475402832031,
            9.143593788146973,  -7.643154144287109,     -2.0325264930725098,
            6.063992500305176,  4.094968318939209,      0.8910917043685913,
            8.712732315063477,  -0.0006124831270426512, 5.505736827850342,
            -9.155109405517578, -9.89109992980957,      1.0480059385299683,
            -5.925083637237549, 7.741676330566406,      0.700584352016449,
            -5.662013530731201, 1.3204102516174316,     2.7849292755126953
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.0228718519210815, 0.11223962903022766, 0.8512431979179382,
            1.3703473806381226,  0.20456767082214355, -1.5155260562896729,
            1.3417094945907593,  -1.705802321434021,  -0.6872337460517883,
            0.7826303243637085,  0.42516833543777466, -0.1564721316099167,
            1.3518258333206177,  0.09107562154531479, 0.8877996206283569,
            -1.2335057258605957, -1.3399975299835205, 0.2428021878004074,
            -1.273769736289978,  1.58700692653656,    0.1131395623087883,
            -1.2187029123306274, 0.2428838163614273,  0.5494423508644104
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float32 2D tensor axes=[] and options.bias',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        },
        'layerNormBias': {
          'data': [7.862982749938965],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'},
          {'options': {'axes': [], 'bias': 'layerNormBias'}}
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            7.862982749938965, 7.862982749938965, 7.862982749938965,
            7.862982749938965, 7.862982749938965, 7.862982749938965,
            7.862982749938965, 7.862982749938965, 7.862982749938965,
            7.862982749938965, 7.862982749938965, 7.862982749938965,
            7.862982749938965, 7.862982749938965, 7.862982749938965,
            7.862982749938965, 7.862982749938965, 7.862982749938965,
            7.862982749938965, 7.862982749938965, 7.862982749938965,
            7.862982749938965, 7.862982749938965, 7.862982749938965
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float32 2D tensor axes=[]',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}, {'options': {'axes': []}}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float32 3D tensor default options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.4057259559631348, 0.5396455526351929,  -0.21643976867198944,
            -0.9825550317764282, 0.7713912725448608,  -0.08366990834474564,
            1.46259605884552,    -0.8138729333877563, 0.7165266871452332,
            -1.6945916414260864, 1.3408818244934082,  0.3658137917518616,
            -1.5234858989715576, 0.5162702202796936,  0.43863821029663086,
            1.0831668376922607,  -1.2419193983078003, 0.29146093130111694,
            -1.7796510457992554, -0.5852779150009155, 1.3068104982376099,
            0.10783683508634567, 1.0319640636444092,  0.35418668389320374
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float32 4D tensor default options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.4057259559631348, 0.5396455526351929,  -0.21643976867198944,
            -0.9825550317764282, 0.7713912725448608,  -0.08366990834474564,
            1.46259605884552,    -0.8138729333877563, 0.7165266871452332,
            -1.6945916414260864, 1.3408818244934082,  0.3658137917518616,
            -1.5234858989715576, 0.5162702202796936,  0.43863821029663086,
            1.0831668376922607,  -1.2419193983078003, 0.29146093130111694,
            -1.7796510457992554, -0.5852779150009155, 1.3068104982376099,
            0.10783683508634567, 1.0319640636444092,  0.35418668389320374
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float32 5D tensor default options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [2, 1, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.4057259559631348, 0.5396455526351929,  -0.21643976867198944,
            -0.9825550317764282, 0.7713912725448608,  -0.08366990834474564,
            1.46259605884552,    -0.8138729333877563, 0.7165266871452332,
            -1.6945916414260864, 1.3408818244934082,  0.3658137917518616,
            -1.5234858989715576, 0.5162702202796936,  0.43863821029663086,
            1.0831668376922607,  -1.2419193983078003, 0.29146093130111694,
            -1.7796510457992554, -0.5852779150009155, 1.3068104982376099,
            0.10783683508634567, 1.0319640636444092,  0.35418668389320374
          ],
          'descriptor': {shape: [2, 1, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float32 4D tensor options.scale',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        },
        'layerNormScale': {
          'data': [
            -3.8228423595428467, -5.452458381652832, 0.6776165962219238,
            -4.027037620544434, -3.7771618366241455, -9.327335357666016,
            7.1816911697387695, 1.5054303407669067, 3.120894193649292,
            0.5214731693267822, 2.6719748973846436, -3.571370840072632
          ],
          'descriptor': {shape: [1, 4, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'}, {'options': {'scale': 'layerNormScale'}}
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            5.373868465423584,   -2.942394971847534,  -0.14666318893432617,
            3.9567861557006836,  -2.9136698246002197, 0.780417263507843,
            10.503913879394531,  -1.225229024887085,  2.236203908920288,
            -0.8836840987205505, 3.5828025341033936,  -1.3064566850662231,
            5.824046611785889,   -2.814941883087158,  0.29722854495048523,
            -4.3619537353515625, 4.6909308433532715,  -2.7185537815093994,
            -12.780903816223145, -0.8810951709747314, 4.0784173011779785,
            0.05623401328921318, 2.7573819160461426,  -1.2649319171905518
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float32 4D tensor options.bias',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        },
        'layerNormBias': {
          'data': [
            7.862982749938965, -3.6603047847747803, -6.955524444580078,
            -6.397322654724121, 3.268958568572998, -2.7498080730438232,
            -4.080942153930664, -7.137991905212402, 8.465653419494629,
            2.762545108795166, 0.8230442404747009, -3.827561378479004
          ],
          'descriptor': {shape: [1, 4, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'}, {'options': {'bias': 'layerNormBias'}}
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            6.45725679397583,    -3.120659112930298,  -7.171964168548584,
            -7.37987756729126,   4.040349960327148,   -2.8334779739379883,
            -2.6183459758758545, -7.951864719390869,  9.182180404663086,
            1.0679534673690796,  2.163926124572754,   -3.461747646331787,
            6.339496612548828,   -3.1440346240997314, -6.516886234283447,
            -5.314155578613281,  2.027039051055908,   -2.4583470821380615,
            -5.860593318939209,  -7.723269939422607,  9.77246379852295,
            2.8703818321228027,  1.8550082445144653,  -3.473374605178833
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float32 4D tensor options.axes=[2]',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}, {'options': {'axes': [2]}}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -0.6012066006660461,  0.10132180899381638, -1.112992763519287,
            -0.26228588819503784, 0.3943416476249695,  -0.7543209195137024,
            1.6960537433624268,   -1.6100702285766602, 1.4073745012283325,
            -0.8325613141059875,  1.114406704902649,   0.45993921160697937,
            -0.8445013165473938,  0.6554933190345764,  -0.3856155574321747,
            1.3668763637542725,   -1.3111618757247925, -0.7422532439231873,
            -1.0618212223052979,  -0.5766634941101074, 1.7181260585784912,
            0.539446234703064,    1.2323321104049683,  -0.5902572274208069
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float32 4D tensor options.epsilon',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments':
            [{'input': 'layerNormInput'}, {'options': {'epsilon': 0.0001}}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.4057258367538452, 0.5396455526351929,  -0.21643976867198944,
            -0.9825550317764282, 0.7713912725448608,  -0.08366990089416504,
            1.46259605884552,    -0.8138729333877563, 0.7165266871452332,
            -1.6945916414260864, 1.3408817052841187,  0.3658137619495392,
            -1.5234858989715576, 0.5162702202796936,  0.43863821029663086,
            1.0831668376922607,  -1.2419193983078003, 0.29146093130111694,
            -1.7796509265899658, -0.5852779150009155, 1.3068104982376099,
            0.10783682763576508, 1.0319639444351196,  0.35418668389320374
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'layerNormalization float32 4D tensor options.scale and options.axes=[0, 2]',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        },
        'layerNormScale': {
          'data': [
            8.72657299041748, -5.388210773468018, -6.811323165893555,
            4.707905292510986, -4.705780029296875, -5.143046855926514,
            -1.1115549802780151, 5.250569820404053
          ],
          'descriptor': {shape: [2, 4], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'},
          {'options': {'scale': 'layerNormScale', 'axes': [0, 2]}}
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -3.3744184970855713, 5.22746467590332,     -7.580371856689453,
            0.3324689269065857,  -4.414334774017334,   2.973374605178833,
            -12.369945526123047, 4.680946350097656,    -9.247408866882324,
            -2.8648624420166016, 6.40486478805542,     2.4516794681549072,
            4.884079456329346,   -0.44672244787216187, 2.521172285079956,
            -6.083702564239502,  9.044846534729004,    4.759283065795898,
            1.3962621688842773,  1.185346245765686,    -1.959165334701538,
            1.8479242324829102,  3.3530402183532715,   -3.986907958984375
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'layerNormalization float32 4D tensor options.bias and options.axes=[3, 1, 2]',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        },
        'layerNormBias': {
          'data': [
            -0.1396923065185547, -6.156772136688232, 4.363296031951904,
            8.8598051071167, 9.772650718688965, -3.4626545906066895,
            9.744950294494629, -0.3958968222141266, -8.497353553771973,
            6.172536849975586, -2.8930461406707764, 1.7220044136047363
          ],
          'descriptor': {shape: [3, 1, 4], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'},
          {'options': {'bias': 'layerNormBias', 'axes': [3, 1, 2]}}
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.5454182624816895, 10.312295913696289, -8.713793754577637,
            -7.139327049255371,  -2.691263198852539, 6.088866710662842,
            5.825891971588135,   8.931077003479004,  -2.1765193939208984,
            7.165213584899902,   0.9449849724769592, 2.087818145751953,
            -1.6631782054901123, 10.288921356201172, -8.058714866638184,
            -5.073605060577393,  -4.704574108123779, 6.463997840881348,
            2.5836451053619385,  9.159672737121582,  -1.5862356424331665,
            8.967641830444336,   0.6360672116279602, 2.0761911869049072
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float32 4D tensor all options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.51446533203125,  54.735408782958984,  19.659019470214844,
            -15.882678031921387, 65.48657989501953,   25.818492889404297,
            97.55302429199219,   -8.057161331176758,  62.9412956237793,
            -48.91555404663086,  91.90644073486328,   46.67098617553711,
            -74.85331726074219,  30.126361846923828,  26.13089370727539,
            59.30270767211914,   -60.361995697021484, 18.55615234375,
            -88.03730773925781,  -26.5667724609375,   70.81292724609375,
            9.105611801147461,   56.66746139526367,   21.78444480895996
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        },
        'layerNormScale': {
          'data': [
            7.715926647186279, 1.7371079921722412, 9.13965129852295,
            5.758823394775391, -2.8198351860046387, -0.6866958141326904
          ],
          'descriptor': {shape: [2, 3, 1], dataType: 'float32'},
          'constant': true
        },
        'layerNormBias': {
          'data': [
            -8.710672378540039, -7.642981052398682, 4.937538146972656,
            -2.1876745223999023, -4.067612648010254, -6.836254596710205
          ],
          'descriptor': {shape: [2, 3, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'}, {
            'options': {
              'scale': 'layerNormScale',
              'bias': 'layerNormBias',
              'axes': [0, 3, 1],
              'epsilon': 0.0001
            }
          }
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -15.487034797668457, -5.628695964813232,  8.29687786102295,
            -14.294686317443848, -5.639192581176758,  7.11608362197876,
            0.7769554257392883,  -8.346451759338379,  11.279659271240234,
            -22.506288528442383, -5.173816204071045,  8.506545066833496,
            -12.360523223876953, -5.77052116394043,   -7.18900203704834,
            3.6336634159088135,  0.8666883707046509,  -6.884884357452393,
            -11.648612976074219, -2.117840528488159,  -7.396423816680908,
            -4.869131088256836,  -5.8111701011657715, -6.714934349060059
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'layerNormalization float16 2D tensor default options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -5.7109375,    1.4677734375,
            6.14453125,    9.4296875,
            2.052734375,   -8.828125,
            9.140625,      -7.64453125,
            -2.033203125,  6.0625,
            4.09375,       0.89111328125,
            8.7109375,     -0.0006122589111328125,
            5.50390625,    -9.15625,
            -9.890625,     1.0478515625,
            -5.92578125,   7.7421875,
            0.70068359375, -5.66015625,
            1.3203125,     2.78515625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.0224609375,    0.11199951171875, 0.85107421875,   1.3701171875,
            0.2044677734375,  -1.515625,        1.341796875,     -1.7060546875,
            -0.68701171875,   0.78271484375,    0.42529296875,   -0.15625,
            1.3515625,        0.0911865234375,  0.8876953125,    -1.2333984375,
            -1.33984375,      0.242919921875,   -1.2744140625,   1.5869140625,
            0.11309814453125, -1.21875,         0.2427978515625, 0.54931640625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float16 3D tensor default options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.5,    54.75,      19.65625, -15.8828125, 65.5,     25.8125,
            97.5625,  -8.0546875, 62.9375,  -48.90625,   91.9375,  46.65625,
            -74.875,  30.125,     26.125,   59.3125,     -60.375,  18.5625,
            -88.0625, -26.5625,   70.8125,  9.109375,    56.65625, 21.78125
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.4052734375, 0.5400390625,      -0.216552734375, -0.982421875,
            0.771484375,   -0.08392333984375, 1.462890625,     -0.81396484375,
            0.71630859375, -1.6943359375,     1.341796875,     0.365234375,
            -1.5234375,    0.51611328125,     0.4384765625,    1.0830078125,
            -1.2421875,    0.291748046875,    -1.7802734375,   -0.5849609375,
            1.306640625,   0.10797119140625,  1.03125,         0.354248046875
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float16 4D tensor default options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.5,    54.75,      19.65625, -15.8828125, 65.5,     25.8125,
            97.5625,  -8.0546875, 62.9375,  -48.90625,   91.9375,  46.65625,
            -74.875,  30.125,     26.125,   59.3125,     -60.375,  18.5625,
            -88.0625, -26.5625,   70.8125,  9.109375,    56.65625, 21.78125
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.4052734375, 0.5400390625,      -0.216552734375, -0.982421875,
            0.771484375,   -0.08392333984375, 1.462890625,     -0.81396484375,
            0.71630859375, -1.6943359375,     1.341796875,     0.365234375,
            -1.5234375,    0.51611328125,     0.4384765625,    1.0830078125,
            -1.2421875,    0.291748046875,    -1.7802734375,   -0.5849609375,
            1.306640625,   0.10797119140625,  1.03125,         0.354248046875
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float16 5D tensor default options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.5,    54.75,      19.65625, -15.8828125, 65.5,     25.8125,
            97.5625,  -8.0546875, 62.9375,  -48.90625,   91.9375,  46.65625,
            -74.875,  30.125,     26.125,   59.3125,     -60.375,  18.5625,
            -88.0625, -26.5625,   70.8125,  9.109375,    56.65625, 21.78125
          ],
          'descriptor': {shape: [2, 1, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.4052734375, 0.5400390625,      -0.216552734375, -0.982421875,
            0.771484375,   -0.08392333984375, 1.462890625,     -0.81396484375,
            0.71630859375, -1.6943359375,     1.341796875,     0.365234375,
            -1.5234375,    0.51611328125,     0.4384765625,    1.0830078125,
            -1.2421875,    0.291748046875,    -1.7802734375,   -0.5849609375,
            1.306640625,   0.10797119140625,  1.03125,         0.354248046875
          ],
          'descriptor': {shape: [2, 1, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float16 4D tensor options.scale',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.5,    54.75,      19.65625, -15.8828125, 65.5,     25.8125,
            97.5625,  -8.0546875, 62.9375,  -48.90625,   91.9375,  46.65625,
            -74.875,  30.125,     26.125,   59.3125,     -60.375,  18.5625,
            -88.0625, -26.5625,   70.8125,  9.109375,    56.65625, 21.78125
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        },
        'layerNormScale': {
          'data': [
            -3.822265625, -5.453125, 0.677734375, -4.02734375, -3.77734375,
            -9.328125, 7.18359375, 1.505859375, 3.12109375, 0.521484375,
            2.671875, -3.572265625
          ],
          'descriptor': {shape: [1, 4, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'}, {'options': {'scale': 'layerNormScale'}}
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            5.37109375,  -2.943359375,      -0.1468505859375, 3.95703125,
            -2.9140625,  0.78271484375,     10.5078125,       -1.2255859375,
            2.236328125, -0.8837890625,     3.583984375,      -1.3046875,
            5.82421875,  -2.814453125,      0.297119140625,   -4.36328125,
            4.69140625,  -2.720703125,      -12.7890625,      -0.880859375,
            4.078125,    0.056304931640625, 2.755859375,      -1.265625
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float16 4D tensor options.bias',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.5,    54.75,      19.65625, -15.8828125, 65.5,     25.8125,
            97.5625,  -8.0546875, 62.9375,  -48.90625,   91.9375,  46.65625,
            -74.875,  30.125,     26.125,   59.3125,     -60.375,  18.5625,
            -88.0625, -26.5625,   70.8125,  9.109375,    56.65625, 21.78125
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        },
        'layerNormBias': {
          'data': [
            7.86328125, -3.66015625, -6.95703125, -6.3984375, 3.26953125, -2.75,
            -4.08203125, -7.13671875, 8.46875, 2.76171875, 0.8232421875,
            -3.828125
          ],
          'descriptor': {shape: [1, 4, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'}, {'options': {'bias': 'layerNormBias'}}
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            6.45703125,   -3.12109375,  -7.171875,    -7.3828125,  4.04296875,
            -2.833984375, -2.619140625, -7.94921875,  9.1875,      1.0673828125,
            2.1640625,    -3.462890625, 6.33984375,   -3.14453125, -6.51953125,
            -5.31640625,  2.02734375,   -2.458984375, -5.86328125, -7.72265625,
            9.7734375,    2.869140625,  1.8544921875, -3.474609375
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float16 4D tensor options.axes=[2]',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.5,    54.75,      19.65625, -15.8828125, 65.5,     25.8125,
            97.5625,  -8.0546875, 62.9375,  -48.90625,   91.9375,  46.65625,
            -74.875,  30.125,     26.125,   59.3125,     -60.375,  18.5625,
            -88.0625, -26.5625,   70.8125,  9.109375,    56.65625, 21.78125
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [{'input': 'layerNormInput'}, {'options': {'axes': [2]}}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -0.60107421875, 0.10125732421875, -1.11328125,   -0.262451171875,
            0.394287109375, -0.75439453125,   1.6962890625,  -1.6103515625,
            1.4072265625,   -0.83251953125,   1.1142578125,  0.45947265625,
            -0.8447265625,  0.65576171875,    -0.3857421875, 1.3671875,
            -1.3115234375,  -0.74169921875,   -1.0615234375, -0.57666015625,
            1.7177734375,   0.53955078125,    1.232421875,   -0.59033203125
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float16 4D tensor options.epsilon',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.5,    54.75,      19.65625, -15.8828125, 65.5,     25.8125,
            97.5625,  -8.0546875, 62.9375,  -48.90625,   91.9375,  46.65625,
            -74.875,  30.125,     26.125,   59.3125,     -60.375,  18.5625,
            -88.0625, -26.5625,   70.8125,  9.109375,    56.65625, 21.78125
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments':
            [{'input': 'layerNormInput'}, {'options': {'epsilon': 0.0001}}],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.4052734375, 0.5400390625,      -0.216552734375, -0.982421875,
            0.771484375,   -0.08392333984375, 1.462890625,     -0.81396484375,
            0.71630859375, -1.6943359375,     1.341796875,     0.365234375,
            -1.5234375,    0.51611328125,     0.4384765625,    1.0830078125,
            -1.2421875,    0.291748046875,    -1.7802734375,   -0.5849609375,
            1.306640625,   0.10797119140625,  1.03125,         0.354248046875
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'layerNormalization float16 4D tensor options.scale and options.axes=[0, 2]',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.5,    54.75,      19.65625, -15.8828125, 65.5,     25.8125,
            97.5625,  -8.0546875, 62.9375,  -48.90625,   91.9375,  46.65625,
            -74.875,  30.125,     26.125,   59.3125,     -60.375,  18.5625,
            -88.0625, -26.5625,   70.8125,  9.109375,    56.65625, 21.78125
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        },
        'layerNormScale': {
          'data': [
            8.7265625, -5.38671875, -6.8125, 4.70703125, -4.70703125,
            -5.14453125, -1.111328125, 5.25
          ],
          'descriptor': {shape: [2, 4], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'},
          {'options': {'scale': 'layerNormScale', 'axes': [0, 2]}}
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -3.37109375,  5.2265625,       -7.58203125, 0.332275390625,
            -4.4140625,   2.97265625,      -12.375,     4.6796875,
            -9.25,        -2.86328125,     6.40625,     2.44921875,
            4.88671875,   -0.446044921875, 2.5234375,   -6.0859375,
            9.046875,     4.7578125,       1.396484375, 1.1845703125,
            -1.958984375, 1.84765625,      3.349609375, -3.986328125
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'layerNormalization float16 4D tensor options.bias and options.axes=[3, 1, 2]',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.5,    54.75,      19.65625, -15.8828125, 65.5,     25.8125,
            97.5625,  -8.0546875, 62.9375,  -48.90625,   91.9375,  46.65625,
            -74.875,  30.125,     26.125,   59.3125,     -60.375,  18.5625,
            -88.0625, -26.5625,   70.8125,  9.109375,    56.65625, 21.78125
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        },
        'layerNormBias': {
          'data': [
            -0.1396484375, -6.15625, 4.36328125, 8.859375, 9.7734375,
            -3.462890625, 9.7421875, -0.39599609375, -8.5, 6.171875,
            -2.892578125, 1.7216796875
          ],
          'descriptor': {shape: [3, 1, 4], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'},
          {'options': {'bias': 'layerNormBias', 'axes': [3, 1, 2]}}
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -1.544921875, 10.3125,     -8.71875,      -7.140625,   -2.69140625,
            6.08984375,   5.82421875,  8.9296875,     -2.17578125, 7.1640625,
            0.9453125,    2.087890625, -1.6630859375, 10.2890625,  -8.0625,
            -5.07421875,  -4.703125,   6.46484375,    2.583984375, 9.15625,
            -1.5859375,   8.96875,     0.6357421875,  2.076171875
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'layerNormalization float16 4D tensor all options',
    'graph': {
      'inputs': {
        'layerNormInput': {
          'data': [
            -35.5,    54.75,      19.65625, -15.8828125, 65.5,     25.8125,
            97.5625,  -8.0546875, 62.9375,  -48.90625,   91.9375,  46.65625,
            -74.875,  30.125,     26.125,   59.3125,     -60.375,  18.5625,
            -88.0625, -26.5625,   70.8125,  9.109375,    56.65625, 21.78125
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        },
        'layerNormScale': {
          'data': [
            7.71484375, 1.7373046875, 9.140625, 5.7578125, -2.8203125,
            -0.6865234375
          ],
          'descriptor': {shape: [2, 3, 1], dataType: 'float16'},
          'constant': true
        },
        'layerNormBias': {
          'data': [
            -8.7109375, -7.64453125, 4.9375, -2.1875, -4.06640625, -6.8359375
          ],
          'descriptor': {shape: [2, 3, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'layerNormalization',
        'arguments': [
          {'input': 'layerNormInput'}, {
            'options': {
              'scale': 'layerNormScale',
              'bias': 'layerNormBias',
              'axes': [0, 3, 1],
              'epsilon': 0.0001
            }
          }
        ],
        'outputs': 'layerNormOutput'
      }],
      'expectedOutputs': {
        'layerNormOutput': {
          'data': [
            -15.484375,  -5.62890625,   8.296875,    -14.296875,  -5.640625,
            7.11328125,  0.775390625,   -8.3515625,  11.28125,    -22.5,
            -5.17578125, 8.5,           -12.359375,  -5.76953125, -7.1875,
            3.6328125,   0.86865234375, -6.8828125,  -11.6484375, -2.1171875,
            -7.39453125, -4.8671875,    -5.80859375, -6.71484375
          ],
          'descriptor': {shape: [2, 1, 4, 3], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(
    layerNormTests, buildAndExecuteGraph, getLayerNormPrecisionTolerance);
