// META: title=test WebNN API l2Pool2d operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-pool2d
// Compute a pooling operation across all the elements within the moving window
// over the input tensor.
//
// enum MLRoundingType {
//   "floor",
//   "ceil"
// };
//
// dictionary MLPool2dOptions {
//   sequence<[EnforceRange] unsigned long> windowDimensions;
//   sequence<[EnforceRange] unsigned long> padding;
//   sequence<[EnforceRange] unsigned long> strides;
//   sequence<[EnforceRange] unsigned long> dilations;
//   MLInputOperandLayout layout = "nchw";
//   MLRoundingType roundingType = "floor";
//   sequence<[EnforceRange] unsigned long> outputSizes;
// };
//
// MLOperand l2Pool2d(
//     MLOperand input, optional MLPool2dOptions options = {});

const l2Pool2dTests = [
  // float32 tests
  {
    'name': 'l2Pool2d float32 4D constant tensor all positive default options',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [{'input': 'l2Pool2dInput'}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289.01953125, 292.6146545410156],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor all positive default options',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [{'input': 'l2Pool2dInput'}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289.01953125, 292.6146545410156],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor all negative default options',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            -1.1957088708877563, -9.706199645996094,  -39.54935836791992,
            -82.34971618652344,  -32.87415313720703,  -50.22603225708008,
            -31.17849349975586,  -55.817893981933594, -46.70829391479492,
            -38.68181228637695,  -63.299320220947266, -35.09224319458008,
            -80.93848419189453,  -82.8619613647461,   -40.41627502441406,
            -34.86458206176758,  -84.33639526367188,  -84.11852264404297,
            -5.525088787078857,  -99.03114318847656,  -75.505126953125,
            -91.43389129638672,  -96.71258544921875,  -16.722585678100586,
            -17.98292350769043,  -58.06570816040039,  -11.846800804138184,
            -97.90313720703125,  -38.69822692871094,  -80.19510650634766,
            -48.72047805786133,  -90.86722564697266,  -99.10758209228516,
            -79.70288848876953,  -59.3824462890625,   -9.967330932617188,
            -39.27534866333008,  -10.469644546508789, -27.565326690673828,
            -2.0468990802764893, -81.88761901855469,  -66.88040161132812,
            -85.98504638671875,  -29.674592971801758, -19.649417877197266,
            -89.39192199707031,  -61.13504409790039,  -84.16869354248047,
            -77.36112213134766,  -91.17266082763672
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [{'input': 'l2Pool2dInput'}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [298.928955078125, 326.83587646484375],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.windowDimensions',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {'options': {'windowDimensions': [3, 3]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            194.45481872558594, 189.54539489746094, 189.85488891601562,
            160.0518341064453, 167.1435546875, 149.63897705078125,
            161.15570068359375, 190.5449981689453, 168.4636688232422,
            170.331787109375, 155.60073852539062, 174.72145080566406,
            165.07762145996094, 165.45819091796875, 161.11062622070312,
            176.6307373046875, 174.245361328125, 180.60714721679688
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.padding',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {'options': {'padding': [1, 0, 0, 1]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            254.81358337402344, 233.14259338378906, 289.01953125,
            269.777587890625, 241.52200317382812, 212.99337768554688,
            292.6146545410156, 253.77178955078125
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.strides',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'strides': [2, 2]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            194.45481872558594, 189.85488891601562, 161.15570068359375,
            168.4636688232422, 170.331787109375, 174.72145080566406,
            176.6307373046875, 180.60714721679688
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.dilations',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'dilations': [2, 2]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [189.47933959960938, 207.25343322753906],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.layout=nchw',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments':
            [{'input': 'l2Pool2dInput'}, {'options': {'layout': 'nchw'}}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289.01953125, 292.6146545410156],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.layout=nhwc',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   52.027313232421875, 76.55464172363281,
            6.397815227508545,   62.71847152709961,  84.54785919189453,
            83.8726577758789,    18.622516632080078, 73.10235595703125,
            34.10626220703125,   41.52470779418945,  73.96932220458984,
            39.3339729309082,    36.1437873840332,   86.59486389160156,
            60.73781967163086,   23.09039306640625,  55.09187316894531,
            53.650146484375,     63.8924446105957,   0.00902052316814661,
            59.36124038696289,   42.78899383544922,  50.91202926635742,
            81.03960418701172,   50.339813232421875, 33.48585510253906,
            59.31963348388672,   33.67196273803711,  70.78031921386719,
            0.42822372913360596, 35.56179428100586,  80.07991790771484,
            82.53382873535156,   5.929991722106934,  7.572360038757324,
            48.89164733886719,   61.90089416503906,  15.282920837402344,
            14.084012985229492,  13.335721969604492, 90.86540985107422,
            39.06557846069336,   39.56248474121094,  97.06050109863281,
            67.77167510986328,   83.68133544921875,  69.69512176513672,
            21.79571533203125,   89.54518127441406
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments':
            [{'input': 'l2Pool2dInput'}, {'options': {'layout': 'nhwc'}}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289.01953125, 292.6146545410156],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.roundingType=floor',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            171.5061492919922, 164.9919891357422, 160.0518341064453,
            149.63897705078125, 142.6990966796875, 139.51637268066406,
            165.07762145996094, 161.11062622070312
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.roundingType=ceil',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            171.5061492919922, 164.9919891357422, 90.6768569946289,
            160.0518341064453, 149.63897705078125, 65.15908813476562,
            132.56260681152344, 139.84808349609375, 26.61993408203125,
            142.6990966796875, 139.51637268066406, 72.42569732666016,
            165.07762145996094, 161.11062622070312, 96.38701629638672,
            150.1616668701172, 146.8201904296875, 90.64601135253906
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'l2Pool2d float32 4D tensor options.outputSizes ignores options.roundingType=floor',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor',
              'outputSizes': [3, 3]
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            171.5061492919922, 164.9919891357422, 90.6768569946289,
            160.0518341064453, 149.63897705078125, 65.15908813476562,
            132.56260681152344, 139.84808349609375, 26.61993408203125,
            142.6990966796875, 139.51637268066406, 72.42569732666016,
            165.07762145996094, 161.11062622070312, 96.38701629638672,
            150.1616668701172, 146.8201904296875, 90.64601135253906
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'l2Pool2d float32 4D tensor options.outputSizes ignores options.roundingType=ceil',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil',
              'outputSizes': [2, 2]
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            171.5061492919922, 164.9919891357422, 160.0518341064453,
            149.63897705078125, 142.6990966796875, 139.51637268066406,
            165.07762145996094, 161.11062622070312
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.dilations with options.strides',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            6.5550384521484375, 26.254413604736328, 28.47271156311035,
            64.81202697753906,  39.65838623046875,  10.465584754943848,
            47.94060134887695,  42.208946228027344, 36.834041595458984,
            68.50249481201172,  2.0496721267700195, 49.73927688598633,
            59.97947311401367,  71.08380889892578,  0.20033331215381622,
            19.39293670654297,  70.1269302368164,   86.8837661743164,
            84.28858184814453,  9.695697784423828,  62.69126510620117,
            51.924110412597656, 5.412675857543945,  70.82118225097656,
            81.61302947998047,  29.148712158203125, 85.83409881591797,
            71.36548614501953,  44.09445571899414,  58.343570709228516,
            43.37118148803711,  54.025882720947266, 85.50556945800781,
            93.19215393066406,  10.992993354797363, 34.864158630371094,
            96.2605209350586,   44.29584503173828,  61.12482833862305,
            79.62699127197266,  4.066447734832764,  64.89644622802734,
            97.5897445678711,   11.257055282592773, 61.151283264160156,
            20.312341690063477, 39.862640380859375, 68.747314453125,
            89.61034393310547,  22.28224754333496,  41.36311721801758,
            62.9378662109375,   79.54936218261719,  55.64254379272461,
            54.47548294067383,  77.04864501953125,  56.83576965332031,
            80.57747650146484,  70.43293762207031,  85.67094421386719,
            19.527807235717773, 33.87490463256836,  14.498117446899414,
            92.85955810546875,  96.8167724609375,   28.399721145629883,
            99.917236328125,    48.76692199707031,  86.08634948730469,
            47.32324981689453,  7.223662376403809,  82.97200775146484,
            38.374778747558594, 22.10988426208496,  14.797550201416016,
            2.3872148990631104, 83.26342010498047,  46.41500473022461,
            28.659175872802734, 13.919462203979492, 55.413089752197266,
            62.68498992919922,  78.54127502441406,  31.142845153808594,
            4.806727886199951,  33.233642578125,    24.749773025512695,
            1.529007077217102,  42.976322174072266, 93.08572387695312,
            77.908935546875,    45.74395751953125,  62.868892669677734,
            60.689762115478516, 20.046878814697266, 13.203198432922363,
            33.33952713012695,  0.5279953479766846
          ],
          'descriptor': {shape: [1, 7, 7, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'dilations': [1, 1],
              'layout': 'nhwc'
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            120.20333862304688, 114.0977783203125, 127.63969421386719,
            119.95613861083984, 137.89837646484375, 152.24261474609375,
            194.9647216796875, 168.20205688476562, 197.7173309326172,
            169.85887145996094, 195.1484832763672, 190.96127319335938,
            158.64576721191406, 166.2051544189453, 171.07916259765625,
            148.70985412597656, 218.7123260498047, 153.33311462402344
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'l2Pool2d float16 4D constant tensor all positive default options',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [{'input': 'l2Pool2dInput'}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289, 292.75],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor all positive default options',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [{'input': 'l2Pool2dInput'}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289, 292.75],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor all negative default options',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            -1.1953125, -9.703125,  -39.5625,  -82.375,   -32.875,  -50.21875,
            -31.171875, -55.8125,   -46.71875, -38.6875,  -63.3125, -35.09375,
            -80.9375,   -82.875,    -40.40625, -34.875,   -84.3125, -84.125,
            -5.5234375, -99,        -75.5,     -91.4375,  -96.6875, -16.71875,
            -17.984375, -58.0625,   -11.84375, -97.875,   -38.6875, -80.1875,
            -48.71875,  -90.875,    -99.125,   -79.6875,  -59.375,  -9.96875,
            -39.28125,  -10.46875,  -27.5625,  -2.046875, -81.875,  -66.875,
            -86,        -29.671875, -19.65625, -89.375,   -61.125,  -84.1875,
            -77.375,    -91.1875
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [{'input': 'l2Pool2dInput'}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [299, 326.75],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor options.windowDimensions',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {'options': {'windowDimensions': [3, 3]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            194.5, 189.625, 189.875, 160.125, 167.125, 149.625, 161.125, 190.5,
            168.5, 170.375, 155.625, 174.75, 165.125, 165.5, 161.125, 176.625,
            174.25, 180.625
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor options.padding',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {'options': {'padding': [1, 0, 0, 1]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [254.875, 233.125, 289, 269.75, 241.5, 213, 292.75, 253.75],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor options.strides',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'strides': [2, 2]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            194.5, 189.875, 161.125, 168.5, 170.375, 174.75, 176.625, 180.625
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor options.dilations',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'dilations': [2, 2]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [189.5, 207.25],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor options.layout=nchw',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments':
            [{'input': 'l2Pool2dInput'}, {'options': {'layout': 'nchw'}}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289, 292.75],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor options.layout=nhwc',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       52.03125,   76.5625,
            6.3984375,     62.71875,   84.5625,
            83.875,        18.625,     73.125,
            34.09375,      41.53125,   74,
            39.34375,      36.15625,   86.625,
            60.75,         23.09375,   55.09375,
            53.65625,      63.90625,   0.0090179443359375,
            59.375,        42.78125,   50.90625,
            81.0625,       50.34375,   33.5,
            59.3125,       33.6875,    70.75,
            0.42822265625, 35.5625,    80.0625,
            82.5625,       5.9296875,  7.57421875,
            48.90625,      61.90625,   15.28125,
            14.0859375,    13.3359375, 90.875,
            39.0625,       39.5625,    97.0625,
            67.75,         83.6875,    69.6875,
            21.796875,     89.5625
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments':
            [{'input': 'l2Pool2dInput'}, {'options': {'layout': 'nhwc'}}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289, 292.75],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor options.roundingType=floor',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data':
              [171.5, 165, 160.125, 149.625, 142.75, 139.5, 165.125, 161.125],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor options.roundingType=ceil',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            171.5, 165, 90.6875, 160.125, 149.625, 65.1875, 132.5, 139.875,
            26.625, 142.75, 139.5, 72.4375, 165.125, 161.125, 96.375, 150.125,
            146.875, 90.6875
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'l2Pool2d float16 4D tensor options.outputSizes ignores options.roundingType=floor',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor',
              'outputSizes': [3, 3]
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            171.5, 165, 90.6875, 160.125, 149.625, 65.1875, 132.5, 139.875,
            26.625, 142.75, 139.5, 72.4375, 165.125, 161.125, 96.375, 150.125,
            146.875, 90.6875
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'l2Pool2d float16 4D tensor options.outputSizes ignores options.roundingType=ceil',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.0625,       76.5625,            62.71875,
            83.875,        73.125,             41.53125,
            39.34375,      86.625,             23.09375,
            53.65625,      0.0090179443359375, 42.78125,
            81.0625,       33.5,               33.6875,
            0.42822265625, 80.0625,            5.9296875,
            48.90625,      15.28125,           13.3359375,
            39.0625,       97.0625,            83.6875,
            21.796875,     52.03125,           6.3984375,
            84.5625,       18.625,             34.09375,
            74,            36.15625,           60.75,
            55.09375,      63.90625,           59.375,
            50.90625,      50.34375,           59.3125,
            70.75,         35.5625,            82.5625,
            7.57421875,    61.90625,           14.0859375,
            90.875,        39.5625,            67.75,
            69.6875,       89.5625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil',
              'outputSizes': [2, 2]
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data':
              [171.5, 165, 160.125, 149.625, 142.75, 139.5, 165.125, 161.125],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float16 4D tensor options.dilations with options.strides',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            6.5546875,   26.25,    28.46875,     64.8125,    39.65625,
            10.46875,    47.9375,  42.21875,     36.84375,   68.5,
            2.048828125, 49.75,    59.96875,     71.0625,    0.2003173828125,
            19.390625,   70.125,   86.875,       84.3125,    9.6953125,
            62.6875,     51.9375,  5.4140625,    70.8125,    81.625,
            29.15625,    85.8125,  71.375,       44.09375,   58.34375,
            43.375,      54.03125, 85.5,         93.1875,    10.9921875,
            34.875,      96.25,    44.28125,     61.125,     79.625,
            4.06640625,  64.875,   97.5625,      11.2578125, 61.15625,
            20.3125,     39.875,   68.75,        89.625,     22.28125,
            41.375,      62.9375,  79.5625,      55.65625,   54.46875,
            77.0625,     56.84375, 80.5625,      70.4375,    85.6875,
            19.53125,    33.875,   14.5,         92.875,     96.8125,
            28.40625,    99.9375,  48.78125,     86.0625,    47.3125,
            7.22265625,  83,       38.375,       22.109375,  14.796875,
            2.38671875,  83.25,    46.40625,     28.65625,   13.921875,
            55.40625,    62.6875,  78.5625,      31.140625,  4.80859375,
            33.21875,    24.75,    1.529296875,  42.96875,   93.0625,
            77.9375,     45.75,    62.875,       60.6875,    20.046875,
            13.203125,   33.34375, 0.52783203125
          ],
          'descriptor': {shape: [1, 7, 7, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'dilations': [1, 1],
              'layout': 'nhwc'
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            120.1875, 114.0625, 127.625, 119.9375, 137.875, 152.25, 195, 168.25,
            197.75, 169.875, 195.125, 191, 158.625, 166.25, 171.125, 148.75,
            218.75, 153.375
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(
    l2Pool2dTests, buildAndExecuteGraph, getPrecisionTolerance);
