// META: title=test WebNN API hardSigmoid operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-hard-sigmoid
// Calculate the non-smooth hard sigmoid function on the input tensor, used
// instead of the sigmoid function for faster computation.
//
// dictionary MLHardSigmoidOptions {
//   double alpha = 0.2;
//   double beta = 0.5;
// };
//
// MLOperand hardSigmoid(
//     MLOperand input, optional MLHardSigmoidOptions options = {});

const hardSigmoidTests = [
  {
    'name': 'hardSigmoid float32 positive 0D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [0.05907066911458969],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [0.5118141174316406],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float32 positive 1D constant tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05907066911458969, 0.7076089382171631, 0.5228404998779297,
            0.4231015741825104,  0.6643692851066589, 0.950294017791748,
            0.10918906331062317, 0.0129771139472723, 0.4755297303199768,
            0.5322551727294922,  0.684307873249054,  0.4662107527256012,
            0.3048996329307556,  0.8025872707366943, 0.2485964000225067,
            0.663689911365509,   0.5547611713409424, 0.554258406162262,
            0.7311381697654724,  0.4880960285663605, 0.7766845226287842,
            0.8455570340156555,  0.555302083492279,  0.5603444576263428
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.5118141174316406, 0.6415218114852905, 0.6045681238174438,
            0.5846202969551086, 0.6328738331794739, 0.6900588274002075,
            0.5218378305435181, 0.5025954246520996, 0.5951059460639954,
            0.6064510345458984, 0.6368615627288818, 0.5932421684265137,
            0.5609799027442932, 0.6605174541473389, 0.5497192740440369,
            0.6327379941940308, 0.6109522581100464, 0.6108517050743103,
            0.6462276577949524, 0.5976191759109497, 0.6553369164466858,
            0.669111430644989,  0.6110604405403137, 0.6120688915252686
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float32 positive 1D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05907066911458969, 0.7076089382171631, 0.5228404998779297,
            0.4231015741825104,  0.6643692851066589, 0.950294017791748,
            0.10918906331062317, 0.0129771139472723, 0.4755297303199768,
            0.5322551727294922,  0.684307873249054,  0.4662107527256012,
            0.3048996329307556,  0.8025872707366943, 0.2485964000225067,
            0.663689911365509,   0.5547611713409424, 0.554258406162262,
            0.7311381697654724,  0.4880960285663605, 0.7766845226287842,
            0.8455570340156555,  0.555302083492279,  0.5603444576263428
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.5118141174316406, 0.6415218114852905, 0.6045681238174438,
            0.5846202969551086, 0.6328738331794739, 0.6900588274002075,
            0.5218378305435181, 0.5025954246520996, 0.5951059460639954,
            0.6064510345458984, 0.6368615627288818, 0.5932421684265137,
            0.5609799027442932, 0.6605174541473389, 0.5497192740440369,
            0.6327379941940308, 0.6109522581100464, 0.6108517050743103,
            0.6462276577949524, 0.5976191759109497, 0.6553369164466858,
            0.669111430644989,  0.6110604405403137, 0.6120688915252686
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float32 positive 2D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05907066911458969, 0.7076089382171631, 0.5228404998779297,
            0.4231015741825104,  0.6643692851066589, 0.950294017791748,
            0.10918906331062317, 0.0129771139472723, 0.4755297303199768,
            0.5322551727294922,  0.684307873249054,  0.4662107527256012,
            0.3048996329307556,  0.8025872707366943, 0.2485964000225067,
            0.663689911365509,   0.5547611713409424, 0.554258406162262,
            0.7311381697654724,  0.4880960285663605, 0.7766845226287842,
            0.8455570340156555,  0.555302083492279,  0.5603444576263428
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.5118141174316406, 0.6415218114852905, 0.6045681238174438,
            0.5846202969551086, 0.6328738331794739, 0.6900588274002075,
            0.5218378305435181, 0.5025954246520996, 0.5951059460639954,
            0.6064510345458984, 0.6368615627288818, 0.5932421684265137,
            0.5609799027442932, 0.6605174541473389, 0.5497192740440369,
            0.6327379941940308, 0.6109522581100464, 0.6108517050743103,
            0.6462276577949524, 0.5976191759109497, 0.6553369164466858,
            0.669111430644989,  0.6110604405403137, 0.6120688915252686
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float32 positive 3D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05907066911458969, 0.7076089382171631, 0.5228404998779297,
            0.4231015741825104,  0.6643692851066589, 0.950294017791748,
            0.10918906331062317, 0.0129771139472723, 0.4755297303199768,
            0.5322551727294922,  0.684307873249054,  0.4662107527256012,
            0.3048996329307556,  0.8025872707366943, 0.2485964000225067,
            0.663689911365509,   0.5547611713409424, 0.554258406162262,
            0.7311381697654724,  0.4880960285663605, 0.7766845226287842,
            0.8455570340156555,  0.555302083492279,  0.5603444576263428
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.5118141174316406, 0.6415218114852905, 0.6045681238174438,
            0.5846202969551086, 0.6328738331794739, 0.6900588274002075,
            0.5218378305435181, 0.5025954246520996, 0.5951059460639954,
            0.6064510345458984, 0.6368615627288818, 0.5932421684265137,
            0.5609799027442932, 0.6605174541473389, 0.5497192740440369,
            0.6327379941940308, 0.6109522581100464, 0.6108517050743103,
            0.6462276577949524, 0.5976191759109497, 0.6553369164466858,
            0.669111430644989,  0.6110604405403137, 0.6120688915252686
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float32 positive 4D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05907066911458969, 0.7076089382171631, 0.5228404998779297,
            0.4231015741825104,  0.6643692851066589, 0.950294017791748,
            0.10918906331062317, 0.0129771139472723, 0.4755297303199768,
            0.5322551727294922,  0.684307873249054,  0.4662107527256012,
            0.3048996329307556,  0.8025872707366943, 0.2485964000225067,
            0.663689911365509,   0.5547611713409424, 0.554258406162262,
            0.7311381697654724,  0.4880960285663605, 0.7766845226287842,
            0.8455570340156555,  0.555302083492279,  0.5603444576263428
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.5118141174316406, 0.6415218114852905, 0.6045681238174438,
            0.5846202969551086, 0.6328738331794739, 0.6900588274002075,
            0.5218378305435181, 0.5025954246520996, 0.5951059460639954,
            0.6064510345458984, 0.6368615627288818, 0.5932421684265137,
            0.5609799027442932, 0.6605174541473389, 0.5497192740440369,
            0.6327379941940308, 0.6109522581100464, 0.6108517050743103,
            0.6462276577949524, 0.5976191759109497, 0.6553369164466858,
            0.669111430644989,  0.6110604405403137, 0.6120688915252686
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float32 positive 5D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05907066911458969, 0.7076089382171631, 0.5228404998779297,
            0.4231015741825104,  0.6643692851066589, 0.950294017791748,
            0.10918906331062317, 0.0129771139472723, 0.4755297303199768,
            0.5322551727294922,  0.684307873249054,  0.4662107527256012,
            0.3048996329307556,  0.8025872707366943, 0.2485964000225067,
            0.663689911365509,   0.5547611713409424, 0.554258406162262,
            0.7311381697654724,  0.4880960285663605, 0.7766845226287842,
            0.8455570340156555,  0.555302083492279,  0.5603444576263428
          ],
          'descriptor': {shape: [1, 2, 1, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.5118141174316406, 0.6415218114852905, 0.6045681238174438,
            0.5846202969551086, 0.6328738331794739, 0.6900588274002075,
            0.5218378305435181, 0.5025954246520996, 0.5951059460639954,
            0.6064510345458984, 0.6368615627288818, 0.5932421684265137,
            0.5609799027442932, 0.6605174541473389, 0.5497192740440369,
            0.6327379941940308, 0.6109522581100464, 0.6108517050743103,
            0.6462276577949524, 0.5976191759109497, 0.6553369164466858,
            0.669111430644989,  0.6110604405403137, 0.6120688915252686
          ],
          'descriptor': {shape: [1, 2, 1, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float32 positive 4D tensor specified positive options.alpha default options.beta',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05907066911458969, 0.7076089382171631, 0.5228404998779297,
            0.4231015741825104,  0.6643692851066589, 0.950294017791748,
            0.10918906331062317, 0.0129771139472723, 0.4755297303199768,
            0.5322551727294922,  0.684307873249054,  0.4662107527256012,
            0.3048996329307556,  0.8025872707366943, 0.2485964000225067,
            0.663689911365509,   0.5547611713409424, 0.554258406162262,
            0.7311381697654724,  0.4880960285663605, 0.7766845226287842,
            0.8455570340156555,  0.555302083492279,  0.5603444576263428
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'},
          {'options': {'alpha': 0.7854232544278235}}
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.546395480632782,
            1,
            0.9106510877609253,
            0.8323138356208801,
            1,
            1,
            0.5857596397399902,
            0.5101925134658813,
            0.8734921216964722,
            0.9180455803871155,
            1,
            0.8661727905273438,
            0.7394752502441406,
            1,
            0.6952533721923828,
            1,
            0.9357223510742188,
            0.9353274703025818,
            1,
            0.8833619952201843,
            1,
            1,
            0.936147153377533,
            0.9401075839996338
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float32 negative 4D tensor specified negative options.alpha default options.beta',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            -0.05907066911458969, -0.7076089382171631, -0.5228404998779297,
            -0.4231015741825104,  -0.6643692851066589, -0.950294017791748,
            -0.10918906331062317, -0.0129771139472723, -0.4755297303199768,
            -0.5322551727294922,  -0.684307873249054,  -0.4662107527256012,
            -0.3048996329307556,  -0.8025872707366943, -0.2485964000225067,
            -0.663689911365509,   -0.5547611713409424, -0.554258406162262,
            -0.7311381697654724,  -0.4880960285663605, -0.7766845226287842,
            -0.8455570340156555,  -0.555302083492279,  -0.5603444576263428
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'},
          {'options': {'alpha': -0.7854232544278235}}
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.546395480632782,
            1,
            0.9106510877609253,
            0.8323138356208801,
            1,
            1,
            0.5857596397399902,
            0.5101925134658813,
            0.8734921216964722,
            0.9180455803871155,
            1,
            0.8661727905273438,
            0.7394752502441406,
            1,
            0.6952533721923828,
            1,
            0.9357223510742188,
            0.9353274703025818,
            1,
            0.8833619952201843,
            1,
            1,
            0.936147153377533,
            0.9401075839996338
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float32 positive 4D tensor specified positive options.beta default options.alpha',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05907066911458969, 0.7076089382171631, 0.5228404998779297,
            0.4231015741825104,  0.6643692851066589, 0.950294017791748,
            0.10918906331062317, 0.0129771139472723, 0.4755297303199768,
            0.5322551727294922,  0.684307873249054,  0.4662107527256012,
            0.3048996329307556,  0.8025872707366943, 0.2485964000225067,
            0.663689911365509,   0.5547611713409424, 0.554258406162262,
            0.7311381697654724,  0.4880960285663605, 0.7766845226287842,
            0.8455570340156555,  0.555302083492279,  0.5603444576263428
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'},
          {'options': {'beta': 0.4361860418530341}}
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.4480001926422119, 0.577707827091217,  0.5407541394233704,
            0.5208063721656799, 0.5690599083900452, 0.626244843006134,
            0.4580238461494446, 0.4387814700603485, 0.5312919616699219,
            0.5426371097564697, 0.5730476379394531, 0.5294281840324402,
            0.4971659779548645, 0.5967035293579102, 0.48590531945228577,
            0.5689240097999573, 0.5471382737159729, 0.5470377206802368,
            0.5824136734008789, 0.533805251121521,  0.5915229320526123,
            0.6052974462509155, 0.5472464561462402, 0.5482549667358398
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float32 negative 4D tensor specified negative options.beta default options.alpha',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            -0.05907066911458969, -0.7076089382171631, -0.5228404998779297,
            -0.4231015741825104,  -0.6643692851066589, -0.950294017791748,
            -0.10918906331062317, -0.0129771139472723, -0.4755297303199768,
            -0.5322551727294922,  -0.684307873249054,  -0.4662107527256012,
            -0.3048996329307556,  -0.8025872707366943, -0.2485964000225067,
            -0.663689911365509,   -0.5547611713409424, -0.554258406162262,
            -0.7311381697654724,  -0.4880960285663605, -0.7766845226287842,
            -0.8455570340156555,  -0.555302083492279,  -0.5603444576263428
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'},
          {'options': {'beta': -0.436186041853034}}
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float32 positive 4D tensor specified all options (positive options.alpha and positive options.beta)',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05907066911458969, 0.7076089382171631, 0.5228404998779297,
            0.4231015741825104,  0.6643692851066589, 0.950294017791748,
            0.10918906331062317, 0.0129771139472723, 0.4755297303199768,
            0.5322551727294922,  0.684307873249054,  0.4662107527256012,
            0.3048996329307556,  0.8025872707366943, 0.2485964000225067,
            0.663689911365509,   0.5547611713409424, 0.554258406162262,
            0.7311381697654724,  0.4880960285663605, 0.7766845226287842,
            0.8455570340156555,  0.555302083492279,  0.5603444576263428
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'},
          {'options': {'alpha': 0.7854232544278235, 'beta': 0.4361860418530341}}
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.4825815260410309,
            0.9919585585594177,
            0.8468371629714966,
            0.7684998512268066,
            0.9579971432685852,
            1,
            0.5219456553459167,
            0.44637855887413025,
            0.8096781373023987,
            0.8542316555976868,
            0.9736573696136475,
            0.8023588061332703,
            0.6756613254547119,
            1,
            0.6314394474029541,
            0.9574635624885559,
            0.8719083666801453,
            0.8715134859085083,
            1,
            0.8195480108261108,
            1,
            1,
            0.8723332285881042,
            0.8762935996055603
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float32 positive 4D tensor specified all options (negative options.alpha and negative options.beta)',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05907066911458969, 0.7076089382171631, 0.5228404998779297,
            0.4231015741825104,  0.6643692851066589, 0.950294017791748,
            0.10918906331062317, 0.0129771139472723, 0.4755297303199768,
            0.5322551727294922,  0.684307873249054,  0.4662107527256012,
            0.3048996329307556,  0.8025872707366943, 0.2485964000225067,
            0.663689911365509,   0.5547611713409424, 0.554258406162262,
            0.7311381697654724,  0.4880960285663605, 0.7766845226287842,
            0.8455570340156555,  0.555302083492279,  0.5603444576263428
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'}, {
            'options':
                {'alpha': -0.7854232544278235, 'beta': -0.4361860418530341}
          }
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float32 negative 4D tensor all options (positive options.alpha and negative options.beta)',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            -0.05907066911458969, -0.7076089382171631, -0.5228404998779297,
            -0.4231015741825104,  -0.6643692851066589, -0.950294017791748,
            -0.10918906331062317, -0.0129771139472723, -0.4755297303199768,
            -0.5322551727294922,  -0.684307873249054,  -0.4662107527256012,
            -0.3048996329307556,  -0.8025872707366943, -0.2485964000225067,
            -0.663689911365509,   -0.5547611713409424, -0.554258406162262,
            -0.7311381697654724,  -0.4880960285663605, -0.7766845226287842,
            -0.8455570340156555,  -0.555302083492279,  -0.5603444576263428
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'}, {
            'options':
                {'alpha': 0.7854232544278235, 'beta': -0.4361860418530341}
          }
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float32 negative 4D tensor specified all options (negative options.alpha and positive options.beta)',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            -0.05907066911458969, -0.7076089382171631, -0.5228404998779297,
            -0.4231015741825104,  -0.6643692851066589, -0.950294017791748,
            -0.10918906331062317, -0.0129771139472723, -0.4755297303199768,
            -0.5322551727294922,  -0.684307873249054,  -0.4662107527256012,
            -0.3048996329307556,  -0.8025872707366943, -0.2485964000225067,
            -0.663689911365509,   -0.5547611713409424, -0.554258406162262,
            -0.7311381697654724,  -0.4880960285663605, -0.7766845226287842,
            -0.8455570340156555,  -0.555302083492279,  -0.5603444576263428
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'}, {
            'options':
                {'alpha': -0.7854232544278235, 'beta': 0.4361860418530341}
          }
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.4825815260410309,
            0.9919585585594177,
            0.8468371629714966,
            0.7684998512268066,
            0.9579971432685852,
            1,
            0.5219456553459167,
            0.44637855887413025,
            0.8096781373023987,
            0.8542316555976868,
            0.9736573696136475,
            0.8023588061332703,
            0.6756613254547119,
            1,
            0.6314394474029541,
            0.9574635624885559,
            0.8719083666801453,
            0.8715134859085083,
            1,
            0.8195480108261108,
            1,
            1,
            0.8723332285881042,
            0.8762935996055603
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'hardSigmoid float16 positive 0D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [0.05908203125],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [0.51171875],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float16 positive 1D constant tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05908203125,    0.70751953125,       0.52294921875,
            0.423095703125,   0.66455078125,       0.9501953125,
            0.10919189453125, 0.01297760009765625, 0.4755859375,
            0.5322265625,     0.68408203125,       0.46630859375,
            0.304931640625,   0.802734375,         0.2486572265625,
            0.66357421875,    0.5546875,           0.55419921875,
            0.73095703125,    0.488037109375,      0.77685546875,
            0.845703125,      0.55517578125,       0.560546875
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.51171875,    0.6416015625,  0.6044921875,  0.58447265625,
            0.6328125,     0.68994140625, 0.52197265625, 0.50244140625,
            0.59521484375, 0.6064453125,  0.63671875,    0.59326171875,
            0.56103515625, 0.66064453125, 0.5498046875,  0.6328125,
            0.61083984375, 0.61083984375, 0.64599609375, 0.59765625,
            0.6552734375,  0.6689453125,  0.61083984375, 0.6123046875
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float16 positive 1D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05908203125,    0.70751953125,       0.52294921875,
            0.423095703125,   0.66455078125,       0.9501953125,
            0.10919189453125, 0.01297760009765625, 0.4755859375,
            0.5322265625,     0.68408203125,       0.46630859375,
            0.304931640625,   0.802734375,         0.2486572265625,
            0.66357421875,    0.5546875,           0.55419921875,
            0.73095703125,    0.488037109375,      0.77685546875,
            0.845703125,      0.55517578125,       0.560546875
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.51171875,    0.6416015625,  0.6044921875,  0.58447265625,
            0.6328125,     0.68994140625, 0.52197265625, 0.50244140625,
            0.59521484375, 0.6064453125,  0.63671875,    0.59326171875,
            0.56103515625, 0.66064453125, 0.5498046875,  0.6328125,
            0.61083984375, 0.61083984375, 0.64599609375, 0.59765625,
            0.6552734375,  0.6689453125,  0.61083984375, 0.6123046875
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float16 positive 2D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05908203125,    0.70751953125,       0.52294921875,
            0.423095703125,   0.66455078125,       0.9501953125,
            0.10919189453125, 0.01297760009765625, 0.4755859375,
            0.5322265625,     0.68408203125,       0.46630859375,
            0.304931640625,   0.802734375,         0.2486572265625,
            0.66357421875,    0.5546875,           0.55419921875,
            0.73095703125,    0.488037109375,      0.77685546875,
            0.845703125,      0.55517578125,       0.560546875
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.51171875,    0.6416015625,  0.6044921875,  0.58447265625,
            0.6328125,     0.68994140625, 0.52197265625, 0.50244140625,
            0.59521484375, 0.6064453125,  0.63671875,    0.59326171875,
            0.56103515625, 0.66064453125, 0.5498046875,  0.6328125,
            0.61083984375, 0.61083984375, 0.64599609375, 0.59765625,
            0.6552734375,  0.6689453125,  0.61083984375, 0.6123046875
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float16 positive 3D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05908203125,    0.70751953125,       0.52294921875,
            0.423095703125,   0.66455078125,       0.9501953125,
            0.10919189453125, 0.01297760009765625, 0.4755859375,
            0.5322265625,     0.68408203125,       0.46630859375,
            0.304931640625,   0.802734375,         0.2486572265625,
            0.66357421875,    0.5546875,           0.55419921875,
            0.73095703125,    0.488037109375,      0.77685546875,
            0.845703125,      0.55517578125,       0.560546875
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.51171875,    0.6416015625,  0.6044921875,  0.58447265625,
            0.6328125,     0.68994140625, 0.52197265625, 0.50244140625,
            0.59521484375, 0.6064453125,  0.63671875,    0.59326171875,
            0.56103515625, 0.66064453125, 0.5498046875,  0.6328125,
            0.61083984375, 0.61083984375, 0.64599609375, 0.59765625,
            0.6552734375,  0.6689453125,  0.61083984375, 0.6123046875
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float16 positive 4D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05908203125,    0.70751953125,       0.52294921875,
            0.423095703125,   0.66455078125,       0.9501953125,
            0.10919189453125, 0.01297760009765625, 0.4755859375,
            0.5322265625,     0.68408203125,       0.46630859375,
            0.304931640625,   0.802734375,         0.2486572265625,
            0.66357421875,    0.5546875,           0.55419921875,
            0.73095703125,    0.488037109375,      0.77685546875,
            0.845703125,      0.55517578125,       0.560546875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.51171875,    0.6416015625,  0.6044921875,  0.58447265625,
            0.6328125,     0.68994140625, 0.52197265625, 0.50244140625,
            0.59521484375, 0.6064453125,  0.63671875,    0.59326171875,
            0.56103515625, 0.66064453125, 0.5498046875,  0.6328125,
            0.61083984375, 0.61083984375, 0.64599609375, 0.59765625,
            0.6552734375,  0.6689453125,  0.61083984375, 0.6123046875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'hardSigmoid float16 positive 5D tensor default options',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05908203125,    0.70751953125,       0.52294921875,
            0.423095703125,   0.66455078125,       0.9501953125,
            0.10919189453125, 0.01297760009765625, 0.4755859375,
            0.5322265625,     0.68408203125,       0.46630859375,
            0.304931640625,   0.802734375,         0.2486572265625,
            0.66357421875,    0.5546875,           0.55419921875,
            0.73095703125,    0.488037109375,      0.77685546875,
            0.845703125,      0.55517578125,       0.560546875
          ],
          'descriptor': {shape: [1, 2, 1, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [{'input': 'hardSigmoidInput'}],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.51171875,    0.6416015625,  0.6044921875,  0.58447265625,
            0.6328125,     0.68994140625, 0.52197265625, 0.50244140625,
            0.59521484375, 0.6064453125,  0.63671875,    0.59326171875,
            0.56103515625, 0.66064453125, 0.5498046875,  0.6328125,
            0.61083984375, 0.61083984375, 0.64599609375, 0.59765625,
            0.6552734375,  0.6689453125,  0.61083984375, 0.6123046875
          ],
          'descriptor': {shape: [1, 2, 1, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float16 positive 4D tensor specified positive options.alpha default options.beta',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05908203125,    0.70751953125,       0.52294921875,
            0.423095703125,   0.66455078125,       0.9501953125,
            0.10919189453125, 0.01297760009765625, 0.4755859375,
            0.5322265625,     0.68408203125,       0.46630859375,
            0.304931640625,   0.802734375,         0.2486572265625,
            0.66357421875,    0.5546875,           0.55419921875,
            0.73095703125,    0.488037109375,      0.77685546875,
            0.845703125,      0.55517578125,       0.560546875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'},
          {'options': {'alpha': 0.7854232544278235}}
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.54638671875,
            1,
            0.91064453125,
            0.83251953125,
            1,
            1,
            0.5859375,
            0.51025390625,
            0.87353515625,
            0.91796875,
            1,
            0.8662109375,
            0.7392578125,
            1,
            0.6953125,
            1,
            0.935546875,
            0.93505859375,
            1,
            0.88330078125,
            1,
            1,
            0.93603515625,
            0.9404296875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float16 negative 4D tensor specified negative options.alpha default options.beta',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            -0.05908203125,    -0.70751953125,       -0.52294921875,
            -0.423095703125,   -0.66455078125,       -0.9501953125,
            -0.10919189453125, -0.01297760009765625, -0.4755859375,
            -0.5322265625,     -0.68408203125,       -0.46630859375,
            -0.304931640625,   -0.802734375,         -0.2486572265625,
            -0.66357421875,    -0.5546875,           -0.55419921875,
            -0.73095703125,    -0.488037109375,      -0.77685546875,
            -0.845703125,      -0.55517578125,       -0.560546875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'},
          {'options': {'alpha': -0.7854232544278235}}
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.54638671875,
            1,
            0.91064453125,
            0.83251953125,
            1,
            1,
            0.5859375,
            0.51025390625,
            0.87353515625,
            0.91796875,
            1,
            0.8662109375,
            0.7392578125,
            1,
            0.6953125,
            1,
            0.935546875,
            0.93505859375,
            1,
            0.88330078125,
            1,
            1,
            0.93603515625,
            0.9404296875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float16 positive 4D tensor specified positive options.beta default options.alpha',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05908203125,    0.70751953125,       0.52294921875,
            0.423095703125,   0.66455078125,       0.9501953125,
            0.10919189453125, 0.01297760009765625, 0.4755859375,
            0.5322265625,     0.68408203125,       0.46630859375,
            0.304931640625,   0.802734375,         0.2486572265625,
            0.66357421875,    0.5546875,           0.55419921875,
            0.73095703125,    0.488037109375,      0.77685546875,
            0.845703125,      0.55517578125,       0.560546875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'},
          {'options': {'beta': 0.4361860418530341}}
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.447998046875, 0.57763671875, 0.541015625,   0.52099609375,
            0.5693359375,   0.62646484375, 0.4580078125,  0.438720703125,
            0.53125,        0.54248046875, 0.5732421875,  0.529296875,
            0.4970703125,   0.5966796875,  0.48583984375, 0.56884765625,
            0.54736328125,  0.546875,      0.58251953125, 0.53369140625,
            0.591796875,    0.60546875,    0.54736328125, 0.54833984375
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float16 negative 4D tensor specified negative options.beta default options.alpha',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            -0.05908203125,    -0.70751953125,       -0.52294921875,
            -0.423095703125,   -0.66455078125,       -0.9501953125,
            -0.10919189453125, -0.01297760009765625, -0.4755859375,
            -0.5322265625,     -0.68408203125,       -0.46630859375,
            -0.304931640625,   -0.802734375,         -0.2486572265625,
            -0.66357421875,    -0.5546875,           -0.55419921875,
            -0.73095703125,    -0.488037109375,      -0.77685546875,
            -0.845703125,      -0.55517578125,       -0.560546875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'},
          {'options': {'beta': -0.436186041853034}}
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float16 positive 4D tensor specified all options (positive options.alpha and positive options.beta)',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05908203125,    0.70751953125,       0.52294921875,
            0.423095703125,   0.66455078125,       0.9501953125,
            0.10919189453125, 0.01297760009765625, 0.4755859375,
            0.5322265625,     0.68408203125,       0.46630859375,
            0.304931640625,   0.802734375,         0.2486572265625,
            0.66357421875,    0.5546875,           0.55419921875,
            0.73095703125,    0.488037109375,      0.77685546875,
            0.845703125,      0.55517578125,       0.560546875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'},
          {'options': {'alpha': 0.7854232544278235, 'beta': 0.4361860418530341}}
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.482666015625,
            0.99169921875,
            0.8466796875,
            0.7685546875,
            0.9580078125,
            1,
            0.52197265625,
            0.4462890625,
            0.8095703125,
            0.85400390625,
            0.9736328125,
            0.80224609375,
            0.67578125,
            1,
            0.63134765625,
            0.95751953125,
            0.8720703125,
            0.87158203125,
            1,
            0.8193359375,
            1,
            1,
            0.8720703125,
            0.87646484375
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float16 positive 4D tensor specified all options (negative options.alpha and negative options.beta)',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            0.05908203125,    0.70751953125,       0.52294921875,
            0.423095703125,   0.66455078125,       0.9501953125,
            0.10919189453125, 0.01297760009765625, 0.4755859375,
            0.5322265625,     0.68408203125,       0.46630859375,
            0.304931640625,   0.802734375,         0.2486572265625,
            0.66357421875,    0.5546875,           0.55419921875,
            0.73095703125,    0.488037109375,      0.77685546875,
            0.845703125,      0.55517578125,       0.560546875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'}, {
            'options':
                {'alpha': -0.7854232544278235, 'beta': -0.4361860418530341}
          }
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float16 negative 4D tensor all options (positive options.alpha and negative options.beta)',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            -0.05908203125,    -0.70751953125,       -0.52294921875,
            -0.423095703125,   -0.66455078125,       -0.9501953125,
            -0.10919189453125, -0.01297760009765625, -0.4755859375,
            -0.5322265625,     -0.68408203125,       -0.46630859375,
            -0.304931640625,   -0.802734375,         -0.2486572265625,
            -0.66357421875,    -0.5546875,           -0.55419921875,
            -0.73095703125,    -0.488037109375,      -0.77685546875,
            -0.845703125,      -0.55517578125,       -0.560546875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'}, {
            'options':
                {'alpha': 0.7854232544278235, 'beta': -0.4361860418530341}
          }
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'hardSigmoid float16 negative 4D tensor specified all options (negative options.alpha and positive options.beta)',
    'graph': {
      'inputs': {
        'hardSigmoidInput': {
          'data': [
            -0.05908203125,    -0.70751953125,       -0.52294921875,
            -0.423095703125,   -0.66455078125,       -0.9501953125,
            -0.10919189453125, -0.01297760009765625, -0.4755859375,
            -0.5322265625,     -0.68408203125,       -0.46630859375,
            -0.304931640625,   -0.802734375,         -0.2486572265625,
            -0.66357421875,    -0.5546875,           -0.55419921875,
            -0.73095703125,    -0.488037109375,      -0.77685546875,
            -0.845703125,      -0.55517578125,       -0.560546875
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'hardSigmoid',
        'arguments': [
          {'input': 'hardSigmoidInput'}, {
            'options':
                {'alpha': -0.7854232544278235, 'beta': 0.4361860418530341}
          }
        ],
        'outputs': 'hardSigmoidOutput'
      }],
      'expectedOutputs': {
        'hardSigmoidOutput': {
          'data': [
            0.482666015625,
            0.99169921875,
            0.8466796875,
            0.7685546875,
            0.9580078125,
            1,
            0.52197265625,
            0.4462890625,
            0.8095703125,
            0.85400390625,
            0.9736328125,
            0.80224609375,
            0.67578125,
            1,
            0.63134765625,
            0.95751953125,
            0.8720703125,
            0.87158203125,
            1,
            0.8193359375,
            1,
            1,
            0.8720703125,
            0.87646484375
          ],
          'descriptor': {shape: [1, 2, 3, 4], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(
    hardSigmoidTests, buildAndExecuteGraph, getPrecisionTolerance);
