// META: title=test WebNN API gemm operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-gemm
// Calculate the general matrix multiplication of the Basic Linear Algebra
// Subprograms. The calculation follows the expression alpha * A * B + beta * C
//
// dictionary MLGemmOptions {
//   MLOperand c;
//   double alpha = 1.0;
//   double beta = 1.0;
//   boolean aTranspose = false;
//   boolean bTranspose = false;
// };
//
// MLOperand gemm(
//     MLOperand a, MLOperand b, optional MLGemmOptions options = {});

const gemmTests = [
  {
    'name': 'gemm two float32 2D tensors (b is non-constant) default options',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8161.14697265625, 14025.82421875, 14607.5009765625,
            14048.5322265625, 18757.09765625, 11892.5107421875,
            14659.9755859375, 19125.74609375, 15034.27734375, 18273.080078125,
            5516.6904296875, 9259.267578125, 9167.3916015625, 6347.330078125,
            10868.6025390625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors default options',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8161.14697265625, 14025.82421875, 14607.5009765625,
            14048.5322265625, 18757.09765625, 11892.5107421875,
            14659.9755859375, 19125.74609375, 15034.27734375, 18273.080078125,
            5516.6904296875, 9259.267578125, 9167.3916015625, 6347.330078125,
            10868.6025390625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D constant tensors options.c',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'},
          'constant': true
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            82.77201843261719, 91.38520812988281, 12.651897430419922,
            20.12200355529785, 68.51224517822266, 35.202415466308594,
            13.33466625213623, 50.78546905517578, 88.61195373535156,
            30.577470779418945, 69.47061920166016, 7.166217803955078,
            28.337108612060547, 90.69412231445312, 71.23025512695312
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8243.9189453125, 14117.208984375, 14620.15234375, 14068.654296875,
            18825.609375, 11927.712890625, 14673.310546875, 19176.533203125,
            15122.8896484375, 18303.658203125, 5586.16064453125, 9266.43359375,
            9195.7294921875, 6438.0244140625, 10939.8330078125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors options.c',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            82.77201843261719, 91.38520812988281, 12.651897430419922,
            20.12200355529785, 68.51224517822266, 35.202415466308594,
            13.33466625213623, 50.78546905517578, 88.61195373535156,
            30.577470779418945, 69.47061920166016, 7.166217803955078,
            28.337108612060547, 90.69412231445312, 71.23025512695312
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8243.9189453125, 14117.208984375, 14620.15234375, 14068.654296875,
            18825.609375, 11927.712890625, 14673.310546875, 19176.533203125,
            15122.8896484375, 18303.658203125, 5586.16064453125, 9266.43359375,
            9195.7294921875, 6438.0244140625, 10939.8330078125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors broadcast options.c [1, 5] => [3, 5]',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            85.90813446044922, 39.3753547668457, 50.942604064941406,
            31.87430763244629, 31.210525512695312
          ],
          'descriptor': {shape: [1, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8247.0546875, 14065.19921875, 14658.443359375, 14080.40625,
            18788.30859375, 11978.4189453125, 14699.3505859375, 19176.689453125,
            15066.1513671875, 18304.291015625, 5602.5986328125, 9298.642578125,
            9218.3349609375, 6379.20458984375, 10899.8125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors broadcast options.c [3, 1] => [3, 5]',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [6.20251989364624, 81.40641784667969, 73.00516510009766],
          'descriptor': {shape: [3, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8167.349609375, 14032.0263671875, 14613.703125, 14054.734375,
            18763.30078125, 11973.9169921875, 14741.3818359375, 19207.154296875,
            15115.68359375, 18354.486328125, 5589.6953125, 9332.2724609375,
            9240.3974609375, 6420.33544921875, 10941.607421875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors broadcast options.c [1, 1] => [3, 5]',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [18.78192901611328],
          'descriptor': {shape: [1, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8179.9287109375, 14044.6064453125, 14626.2822265625,
            14067.314453125, 18775.87890625, 11911.29296875, 14678.7578125,
            19144.529296875, 15053.0595703125, 18291.86328125, 5535.47216796875,
            9278.048828125, 9186.173828125, 6366.1123046875, 10887.384765625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors broadcast options.c [5] => [3, 5]',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            85.90813446044922, 39.3753547668457, 50.942604064941406,
            31.87430763244629, 31.210525512695312
          ],
          'descriptor': {shape: [5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8247.0546875, 14065.19921875, 14658.443359375, 14080.40625,
            18788.30859375, 11978.4189453125, 14699.3505859375, 19176.689453125,
            15066.1513671875, 18304.291015625, 5602.5986328125, 9298.642578125,
            9218.3349609375, 6379.20458984375, 10899.8125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors broadcast options.c [1] => [3, 5]',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [18.78192901611328],
          'descriptor': {shape: [1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8179.9287109375, 14044.6064453125, 14626.2822265625,
            14067.314453125, 18775.87890625, 11911.29296875, 14678.7578125,
            19144.529296875, 15053.0595703125, 18291.86328125, 5535.47216796875,
            9278.048828125, 9186.173828125, 6366.1123046875, 10887.384765625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors scalar options.c',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [18.78192901611328],
          'descriptor': {shape: [], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8179.9287109375, 14044.6064453125, 14626.2822265625,
            14067.314453125, 18775.87890625, 11911.29296875, 14678.7578125,
            19144.529296875, 15053.0595703125, 18291.86328125, 5535.47216796875,
            9278.048828125, 9186.173828125, 6366.1123046875, 10887.384765625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors options.alpha',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'alpha': 74.43204170482103}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            607450.875, 1043970.75, 1087266.125, 1045661, 1396129.125,
            885183.875, 1091172, 1423568.5, 1119032, 1360102.75, 410618.53125,
            689186.1875, 682347.75, 472444.78125, 808972.3125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors options.beta',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'beta': 62.01828598608989}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8161.14697265625, 14025.82421875, 14607.5009765625,
            14048.5322265625, 18757.09765625, 11892.5107421875,
            14659.9755859375, 19125.74609375, 15034.27734375, 18273.080078125,
            5516.6904296875, 9259.267578125, 9167.3916015625, 6347.330078125,
            10868.6025390625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors options.c and options.beta',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            82.77201843261719, 91.38520812988281, 12.651897430419922,
            20.12200355529785, 68.51224517822266, 35.202415466308594,
            13.33466625213623, 50.78546905517578, 88.61195373535156,
            30.577470779418945, 69.47061920166016, 7.166217803955078,
            28.337108612060547, 90.69412231445312, 71.23025512695312
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'c': 'inputC', 'beta': 62.01828598608989}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            13294.525390625, 19693.37890625, 15392.1494140625, 15296.4638671875,
            23006.109375, 14075.7041015625, 15486.96875, 22275.375,
            20529.83984375, 20169.443359375, 9825.138671875, 9703.7041015625,
            10924.810546875, 11972.0244140625, 15286.1806640625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors options.aTranspose being true',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 18.9648494720459, 50.51683807373047,
            90.51641082763672, 43.89479446411133, 40.45679473876953,
            59.638519287109375, 98.89488220214844, 50.76741409301758,
            36.271873474121094, 91.46013641357422, 9.336554527282715
          ],
          'descriptor': {shape: [4, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {'options': {'aTranspose': true}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8161.14697265625, 14025.82421875, 14607.5009765625,
            14048.5322265625, 18757.09765625, 11892.5107421875,
            14659.9755859375, 19125.74609375, 15034.27734375, 18273.080078125,
            5516.6904296875, 9259.267578125, 9167.3916015625, 6347.330078125,
            10868.6025390625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'gemm two float32 2D tensors options.aTranspose being explicit false',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {'options': {'aTranspose': false}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8161.14697265625, 14025.82421875, 14607.5009765625,
            14048.5322265625, 18757.09765625, 11892.5107421875,
            14659.9755859375, 19125.74609375, 15034.27734375, 18273.080078125,
            5516.6904296875, 9259.267578125, 9167.3916015625, 6347.330078125,
            10868.6025390625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors options.bTranspose being true',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  0.0037256532814353704, 75.74880981445312,
            42.90679168701172,  66.6923828125,         16.4991455078125,
            96.60688018798828,  34.08055114746094,     63.29909896850586,
            3.036668062210083,  99.10041809082031,     87.37654876708984,
            10.629964828491211, 93.14022064208984,     23.2437744140625,
            92.34209442138672,  61.32737731933594,     70.08265686035156,
            86.11856842041016,  60.32209014892578
          ],
          'descriptor': {shape: [5, 4], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {'options': {'bTranspose': true}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8161.14697265625, 14025.82421875, 14607.5009765625,
            14048.5322265625, 18757.09765625, 11892.5107421875,
            14659.9755859375, 19125.74609375, 15034.27734375, 18273.080078125,
            5516.6904296875, 9259.267578125, 9167.3916015625, 6347.330078125,
            10868.6025390625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'gemm two float32 2D tensors options.bTranspose being explicit false',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {'options': {'bTranspose': false}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8161.14697265625, 14025.82421875, 14607.5009765625,
            14048.5322265625, 18757.09765625, 11892.5107421875,
            14659.9755859375, 19125.74609375, 15034.27734375, 18273.080078125,
            5516.6904296875, 9259.267578125, 9167.3916015625, 6347.330078125,
            10868.6025390625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm two float32 2D tensors all options',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 18.9648494720459, 50.51683807373047,
            90.51641082763672, 43.89479446411133, 40.45679473876953,
            59.638519287109375, 98.89488220214844, 50.76741409301758,
            36.271873474121094, 91.46013641357422, 9.336554527282715
          ],
          'descriptor': {shape: [4, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  0.0037256532814353704, 75.74880981445312,
            42.90679168701172,  66.6923828125,         16.4991455078125,
            96.60688018798828,  34.08055114746094,     63.29909896850586,
            3.036668062210083,  99.10041809082031,     87.37654876708984,
            10.629964828491211, 93.14022064208984,     23.2437744140625,
            92.34209442138672,  61.32737731933594,     70.08265686035156,
            86.11856842041016,  60.32209014892578
          ],
          'descriptor': {shape: [5, 4], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            82.77201843261719, 91.38520812988281, 12.651897430419922,
            20.12200355529785, 68.51224517822266, 35.202415466308594,
            13.33466625213623, 50.78546905517578, 88.61195373535156,
            30.577470779418945, 69.47061920166016, 7.166217803955078,
            28.337108612060547, 90.69412231445312, 71.23025512695312
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {
            'options': {
              'c': 'inputC',
              'alpha': 74.43204170482103,
              'beta': 62.01828598608989,
              'aTranspose': true,
              'bTranspose': true
            }
          }
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            612584.25, 1049638.375, 1088050.75, 1046908.875, 1400378.125,
            887367.0625, 1091999, 1426718.125, 1124527.625, 1361999.125, 414927,
            689630.625, 684105.1875, 478069.46875, 813389.875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm both negative options.alpha and 1st float32 input tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            -87.94973754882812, -14.765121459960938, -80.86310577392578,
            -59.27638244628906, -28.74825096130371, -9.038779258728027,
            -76.046630859375, -83.94807434082031, -95.02689361572266,
            -47.89733123779297, -69.0116195678711, -86.66899108886719
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            82.77201843261719, 91.38520812988281, 12.651897430419922,
            20.12200355529785, 68.51224517822266, 35.202415466308594,
            13.33466625213623, 50.78546905517578, 88.61195373535156,
            30.577470779418945, 69.47061920166016, 7.166217803955078,
            28.337108612060547, 90.69412231445312, 71.23025512695312
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'c': 'inputC', 'alpha': -22.611149749186296}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            246101.234375, 360542.34375, 425213.25, 218520.65625, 383734.40625,
            228076.8125, 277543.625, 378076, 241282.15625, 316800.71875,
            256391.5625, 378711.34375, 465195.8125, 341034.875, 460338.3125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm both negative options.alpha and 2nd float32 input tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -64.35197448730469, -87.64981079101562,  -96.04495239257812,
            -31.07155990600586, -6.548067092895508,  -64.64146423339844,
            -14.02061653137207, -88.93306732177734,  -24.179410934448242,
            -97.31866455078125, -93.4608154296875,   -39.490394592285156,
            -94.37982177734375, -75.83335876464844,  -11.403324127197266,
            -26.2825927734375,  -56.237979888916016, -46.47404861450195,
            -89.20684051513672, -53.495323181152344
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            82.77201843261719, 91.38520812988281, 12.651897430419922,
            20.12200355529785, 68.51224517822266, 35.202415466308594,
            13.33466625213623, 50.78546905517578, 88.61195373535156,
            30.577470779418945, 69.47061920166016, 7.166217803955078,
            28.337108612060547, 90.69412231445312, 71.23025512695312
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'c': 'inputC', 'alpha': -22.611149749186296}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            400725.03125, 292635.75, 527642.375, 283236.40625, 270787.40625,
            355130.96875, 256121.421875, 436658.0625, 391465.40625,
            235557.03125, 245540.5625, 170154.125, 309239.4375, 163583.015625,
            120958.7578125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'gemm both negative options.beta and 3rd float32 input tensor (options.c)',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            -42.662681579589844, -93.6637954711914, -40.708492279052734,
            -45.060977935791016, -87.90338897705078, -71.1618881225586,
            -85.56422424316406, -10.049134254455566, -46.105403900146484,
            -66.76168060302734, -25.392492294311523, -65.62987518310547,
            -40.50155258178711, -81.5303955078125, -41.39629364013672
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'c': 'inputC', 'beta': -11.68521964935509}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            8659.669921875, 15120.3056640625, 15083.1884765625,
            14575.0791015625, 19784.267578125, 12724.052734375, 15659.8125,
            19243.173828125, 15573.029296875, 19053.205078125, 5813.4072265625,
            10026.1669921875, 9640.6611328125, 7300.03076171875,
            11352.3271484375
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'gemm both negative options.alpha and options.beta',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            82.77201843261719, 91.38520812988281, 12.651897430419922,
            20.12200355529785, 68.51224517822266, 35.202415466308594,
            13.33466625213623, 50.78546905517578, 88.61195373535156,
            30.577470779418945, 69.47061920166016, 7.166217803955078,
            28.337108612060547, 90.69412231445312, 71.23025512695312
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {
            'options': {
              'c': 'inputC',
              'alpha': -22.611149749186296,
              'beta': -11.68521964935509
            }
          }
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            -185500.109375, -318207.84375, -330440.21875, -317888.59375,
            -424920.125, -269314.6875, -331634.71875, -433048.5625,
            -340977.71875, -413532.65625, -125550.484375, -209446.40625,
            -207616.390625, -144580.21875, -246583.921875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'gemm both negative options.alpha and 3rd float32 input tensor (options.c)',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            82.98884582519531, 90.51641082763672, 59.638519287109375,
            36.271873474121094, 18.9648494720459, 43.89479446411133,
            98.89488220214844, 91.46013641357422, 50.51683807373047,
            40.45679473876953, 50.76741409301758, 9.336554527282715
          ],
          'descriptor': {shape: [3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            25.14739227294922,  66.6923828125,     63.29909896850586,
            10.629964828491211, 61.32737731933594, 0.0037256532814353704,
            16.4991455078125,   3.036668062210083, 93.14022064208984,
            70.08265686035156,  75.74880981445312, 96.60688018798828,
            99.10041809082031,  23.2437744140625,  86.11856842041016,
            42.90679168701172,  34.08055114746094, 87.37654876708984,
            92.34209442138672,  60.32209014892578
          ],
          'descriptor': {shape: [4, 5], dataType: 'float32'},
          'constant': true
        },
        'inputC': {
          'data': [
            -42.662681579589844, -93.6637954711914, -40.708492279052734,
            -45.060977935791016, -87.90338897705078, -71.1618881225586,
            -85.56422424316406, -10.049134254455566, -46.105403900146484,
            -66.76168060302734, -25.392492294311523, -65.62987518310547,
            -40.50155258178711, -81.5303955078125, -41.39629364013672
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'c': 'inputC', 'alpha': -22.611149749186296}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            -184575.5625, -317233.65625, -330333.09375, -317698.5, -424207.4375,
            -268974.5, -331564.4375, -432465.15625, -339988.375, -413242.09375,
            -124764.09375, -209428.296875, -207325.765625, -143601.96875,
            -245792.984375
          ],
          'descriptor': {shape: [3, 5], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'gemm two float16 2D tensors (b is non-constant) default options',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.654296875, 0.37451171875, 0.708984375, 0.7578125,
            0.07781982421875, 0.1444091796875, 0.214111328125, 0.73291015625,
            0.9384765625, 0.218505859375, 0.69140625, 0.85400390625
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.98974609375, 0.1365966796875, 0.442626953125, 0.501953125,
            0.3447265625,  0.208251953125,  0.64697265625,  0.88037109375,
            0.83203125,    0.826171875,     0.52978515625,  0.51904296875,
            0.57080078125, 0.74072265625,   0.8125,         0.6103515625,
            0.72509765625, 0.487548828125,  0.126220703125, 0.5703125
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            1.5634765625, 1.2490234375, 1.3935546875, 1.2607421875, 1.54296875,
            0.66796875, 0.74658203125, 0.64111328125, 0.410400390625,
            0.73828125, 1.8623046875, 1.248046875, 1.4189453125, 1.2724609375,
            1.552734375
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors default options',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.074951171875, 0.03216552734375, 0.501953125, 0.650390625,
            0.447021484375, 0.6201171875, 0.027099609375, 0.72021484375,
            0.60009765625, 0.246826171875, 0.9169921875, 0.767578125
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.4384765625,   0.80517578125, 0.09515380859375, 0.2376708984375,
            0.84912109375,  0.79296875,    0.33154296875,    0.66455078125,
            0.46826171875,  0.55712890625, 0.587890625,      0.611328125,
            0.464111328125, 0.74267578125, 0.43505859375,    0.68310546875,
            0.309814453125, 0.5908203125,  0.088134765625,   0.07745361328125
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            0.7978515625, 0.57958984375, 0.6455078125, 0.462890625,
            0.350341796875, 1.1953125, 0.80517578125, 0.892578125,
            0.480224609375, 0.79248046875, 1.5224609375, 1.36328125,
            1.1005859375, 1.0068359375, 1.10546875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D constant tensors options.c',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.74365234375, 0.465576171875, 0.99755859375, 0.5576171875,
            0.3115234375, 0.47509765625, 0.468994140625, 0.340576171875,
            0.1337890625, 0.395263671875, 0.80224609375, 0.60986328125
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'},
          'constant': true
        },
        'inputB': {
          'data': [
            0.46435546875,  0.31298828125,  0.2039794921875, 0.39697265625,
            0.76171875,     0.366455078125, 0.35595703125,   0.1937255859375,
            0.450927734375, 0.7958984375,   0.167724609375,  0.2998046875,
            0.54833984375,  0.72265625,     0.56640625,      0.93896484375,
            0.92333984375,  0.67138671875,  0.822265625,     0.55029296875
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.00004869699478149414, 0.143798828125, 0.87158203125, 0.8759765625,
            0.87353515625, 0.9501953125, 0.87353515625, 0.1787109375,
            0.3935546875, 0.23583984375, 0.048431396484375, 0.157958984375,
            0.0894775390625, 0.19287109375, 0.4462890625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            1.20703125, 1.3564453125, 2.03515625, 2.560546875, 2.681640625,
            1.6669921875, 1.595703125, 0.8203125, 1.3505859375, 1.3046875,
            0.96240234375, 1.14453125, 1.04296875, 1.505859375, 1.65234375
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors options.c',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.17431640625, 0.09527587890625, 0.83935546875, 0.2127685546875,
            0.1314697265625, 0.48095703125, 0.248779296875, 0.043670654296875,
            0.1619873046875, 0.9384765625, 0.12188720703125, 0.8056640625
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.1759033203125, 0.8759765625,       0.3544921875,    0.216796875,
            0.8408203125,    0.0220184326171875, 0.7646484375,    0.8779296875,
            0.2392578125,    0.38525390625,      0.0938720703125, 0.57958984375,
            0.4638671875,    0.6982421875,       0.7900390625,    0.77099609375,
            0.744140625,     0.25537109375,      0.97900390625,   0.890625
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.947265625, 0.61767578125, 0.7177734375, 0.175537109375,
            0.468994140625, 0.378662109375, 0.646484375, 0.48583984375,
            0.9921875, 0.046356201171875, 0.200439453125, 0.69677734375,
            0.054534912109375, 0.732421875, 0.97216796875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            1.22265625, 1.48828125, 1.306640625, 1.0302734375, 1.5048828125,
            0.469482421875, 1.3056640625, 1.0810546875, 1.3525390625,
            0.57763671875, 0.88232421875, 2.2265625, 1.1982421875, 1.8662109375,
            2.283203125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors broadcast options.c [1, 5] => [3, 5]',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.98681640625, 0.7919921875, 0.31298828125, 0.0943603515625,
            0.6806640625, 0.7177734375, 0.4150390625, 0.1239013671875,
            0.744140625, 0.311279296875, 0.54345703125, 0.2401123046875
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.249267578125, 0.63623046875,   0.226806640625,
            0.802734375,    0.88525390625,   0.27880859375,
            0.9345703125,   0.2093505859375, 0.77294921875,
            0.86669921875,  0.284912109375,  0.274658203125,
            0.125244140625, 0.43408203125,   0.005023956298828125,
            0.276123046875, 0.583984375,     0.1583251953125,
            0.53662109375,  0.52490234375
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.093994140625, 0.94384765625, 0.751953125, 0.01458740234375,
            0.822265625
          ],
          'descriptor': {shape: [1, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            0.67578125, 2.453125, 1.1953125, 1.60546875, 2.43359375,
            0.6162109375, 2.234375, 1.1279296875, 1.3623046875, 2.11328125,
            0.58740234375, 1.998046875, 1.091796875, 1.2177734375, 1.8798828125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors broadcast options.c [3, 1] => [3, 5]',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.99755859375, 0.139892578125, 0.255126953125, 0.347900390625,
            0.81298828125, 0.271484375, 0.47607421875, 0.137939453125,
            0.8134765625, 0.42138671875, 0.14306640625, 0.595703125
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.82568359375,  0.8203125,     0.93017578125,     0.669921875,
            0.73486328125,  0.974609375,   0.4951171875,      0.95361328125,
            0.393798828125, 0.55419921875, 0.6474609375,      0.1678466796875,
            0.68798828125,  0.89990234375, 0.08148193359375,  0.67626953125,
            0.3095703125,   0.826171875,   0.025726318359375, 0.9970703125
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [0.424072265625, 0.19873046875, 0.69189453125],
          'descriptor': {shape: [3, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            1.7841796875, 1.4619140625, 1.9482421875, 1.3857421875,
            1.6025390625, 1.5361328125, 1.123046875, 1.6552734375, 1.2822265625,
            1.123046875, 2.26953125, 1.7763671875, 2.44140625, 1.546875,
            2.12890625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors broadcast options.c [1, 1] => [3, 5]',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.68896484375, 0.26171875, 0.78662109375, 0.2275390625,
            0.77294921875, 0.97021484375, 0.1256103515625, 0.98486328125,
            0.59326171875, 0.328125, 0.6611328125, 0.339599609375
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.298583984375, 0.99267578125,    0.85546875,    0.94384765625,
            0.994140625,    0.05615234375,    0.16845703125, 0.6943359375,
            0.83203125,     0.06951904296875, 0.9658203125,  0.11956787109375,
            0.7548828125,   0.29638671875,    0.58203125,    0.02618408203125,
            0.55859375,     0.034912109375,   0.97802734375, 0.70263671875
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [0.425048828125],
          'descriptor': {shape: [1, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            1.4111328125, 1.3740234375, 1.7978515625, 1.7490234375, 1.74609375,
            0.857421875, 1.9208984375, 1.888671875, 2.962890625, 2.025390625,
            1.267578125, 1.337890625, 1.6708984375, 1.7861328125, 1.6611328125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors broadcast options.c [5] => [3, 5]',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.0216827392578125, 0.85888671875, 0.48095703125, 0.487548828125,
            0.91064453125, 0.86328125, 0.8046875, 0.224853515625, 0.50341796875,
            0.5478515625, 0.67724609375, 0.66015625
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.83642578125,     0.473876953125, 0.363037109375, 0.5498046875,
            0.4609375,         0.281982421875, 0.421142578125, 0.2330322265625,
            0.80419921875,     0.1865234375,   0.383056640625, 0.2454833984375,
            0.038421630859375, 0.35302734375,  0.431640625,    0.82177734375,
            0.8759765625,      0.34716796875,  0.47119140625,  0.58154296875
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.7314453125, 0.0292816162109375, 0.8896484375, 0.424072265625,
            0.09136962890625
          ],
          'descriptor': {shape: [5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            1.5771484375, 0.9462890625, 1.28515625, 1.5263671875, 0.7529296875,
            2.23046875, 1.21875, 1.5302734375, 2.009765625, 1.150390625,
            2.109375, 1.2431640625, 1.455078125, 1.69140625, 1.1015625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors broadcast options.c [1] => [3, 5]',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.5712890625, 0.83447265625, 0.35693359375, 0.326904296875,
            0.1727294921875, 0.1966552734375, 0.9404296875, 0.724609375,
            0.630859375, 0.53759765625, 0.72021484375, 0.88525390625
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.91552734375, 0.85009765625, 0.2169189453125, 0.430908203125,
            0.30810546875, 0.88916015625, 0.9931640625,    0.705078125,
            0.55126953125, 0.91259765625, 0.92431640625,   0.148681640625,
            0.4306640625,  0.69921875,    0.24658203125,   0.19287109375,
            0.5263671875,  0.98095703125, 0.80322265625,   0.407958984375
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [0.755859375],
          'descriptor': {shape: [1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            2.4140625, 2.294921875, 1.9423828125, 1.974609375, 1.9150390625,
            2.09765625, 1.619140625, 2.046875, 2.177734375, 1.515625, 2.6484375,
            2.3984375, 2.451171875, 2.5390625, 1.9794921875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors scalar options.c',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.966796875, 0.87353515625, 0.9384765625, 0.9091796875,
            0.81396484375, 0.73876953125, 0.1455078125, 0.08917236328125,
            0.7900390625, 0.271484375, 0.21240234375, 0.393310546875
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.52587890625,   0.219482421875,   0.666015625,  0.5078125,
            0.2279052734375, 0.12420654296875, 0.890625,     0.6884765625,
            0.779296875,     0.7548828125,     0.92578125,   0.669921875,
            0.60498046875,   0.1966552734375,  0.986328125,  0.533203125,
            0.62744140625,   0.732421875,      0.5947265625, 0.33837890625
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [0.95361328125],
          'descriptor': {shape: [], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments':
            [{'a': 'inputA'}, {'b': 'inputB'}, {'options': {'c': 'inputC'}}],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            2.923828125, 3.142578125, 3.431640625, 2.849609375, 3.06640625,
            1.6552734375, 1.943359375, 2.158203125, 2.0234375, 1.8701171875,
            1.8095703125, 1.7578125, 2.083984375, 1.841796875, 1.681640625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors options.alpha',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.455810546875, 0.89501953125, 0.1544189453125, 0.25244140625,
            0.359130859375, 0.61376953125, 0.2164306640625, 0.347412109375,
            0.90966796875, 0.91455078125, 0.59765625, 0.409912109375
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.084228515625, 0.6025390625,   0.370361328125, 0.474365234375,
            0.93115234375,  0.72998046875,  0.76806640625,  0.759765625,
            0.673828125,    0.480224609375, 0.12255859375,  0.71044921875,
            0.90478515625,  0.1796875,      0.55322265625,  0.0193939208984375,
            0.290771484375, 0.2197265625,   0.80859375,     0.102783203125
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'alpha': 74.43204170482103}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            53.25, 85.25, 77.6875, 78.25, 71.875, 38.0625, 70.1875, 64.875,
            67.25, 58.40625, 61.4375, 133.5, 123.75, 110.625, 123.5
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors options.beta',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.371337890625, 0.97216796875, 0.51953125, 0.73974609375,
            0.5126953125, 0.0809326171875, 0.44482421875, 0.421875,
            0.2294921875, 0.609375, 0.441162109375, 0.748046875
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.3330078125,   0.7275390625,   0.229736328125, 0.122314453125,
            0.176025390625, 0.84033203125,  0.392578125,    0.85205078125,
            0.053955078125, 0.257080078125, 0.1416015625,   0.272705078125,
            0.7294921875,   0.55615234375,  0.71630859375,  0.560546875,
            0.6728515625,   0.5107421875,   0.36376953125,  0.92578125
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'beta': 62.01828598608989}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            1.4287109375, 1.291015625, 1.6708984375, 0.65576171875,
            1.3720703125, 0.5380859375, 0.81005859375, 0.7265625,
            0.468017578125, 0.8203125, 1.0703125, 1.0302734375, 1.275390625,
            0.57861328125, 1.2060546875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors options.c and options.beta',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.55615234375, 0.2451171875, 0.1129150390625, 0.6279296875,
            0.3349609375, 0.0188751220703125, 0.09759521484375, 0.73828125,
            0.67236328125, 0.4375, 0.271484375, 0.865234375
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.45361328125,   0.7060546875,    0.45751953125,    0.306640625,
            0.41552734375,   0.271484375,     0.28076171875,    0.376708984375,
            0.2457275390625, 0.283935546875,  0.72412109375,    0.52392578125,
            0.7265625,       0.50390625,      0.08197021484375, 0.33154296875,
            0.2176513671875, 0.1204833984375, 0.1644287109375,  0.2138671875
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.736328125, 0.85791015625, 0.9248046875, 0.92236328125,
            0.9267578125, 0.377685546875, 0.319091796875, 0.2371826171875,
            0.79736328125, 0.074462890625, 0.6376953125, 0.413818359375,
            0.72265625, 0.90966796875, 0.767578125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'c': 'inputC', 'beta': 62.01828598608989}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            46.28125, 53.875, 57.84375, 57.59375, 57.90625, 23.890625, 20.25,
            15.03125, 49.71875, 4.9296875, 40.46875, 26.59375, 45.59375, 57,
            48.21875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors options.aTranspose being true',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.335693359375, 0.2005615234375, 0.2440185546875, 0.306396484375,
            0.85693359375, 0.327880859375, 0.1824951171875, 0.47900390625,
            0.6357421875, 0.15185546875, 0.8330078125, 0.55615234375
          ],
          'descriptor': {shape: [4, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.271240234375,  0.0931396484375, 0.84375,
            0.796875,        0.195556640625,  0.79150390625,
            0.59375,         0.8486328125,    0.01042938232421875,
            0.0809326171875, 0.90869140625,   0.36083984375,
            0.66748046875,   0.1376953125,    0.1590576171875,
            0.513671875,     0.810546875,     0.6669921875,
            0.39208984375,   0.97900390625
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {'options': {'aTranspose': true}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            0.57763671875, 0.402099609375, 0.7666015625, 0.35546875,
            0.26806640625, 1.595703125, 1.3759765625, 1.771484375, 0.5615234375,
            1, 1.189453125, 0.8974609375, 1.279296875, 0.50341796875,
            0.7197265625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'gemm two float16 2D tensors options.aTranspose being explicit false',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.61572265625, 0.9892578125, 0.81591796875, 0.8583984375,
            0.47119140625, 0.56494140625, 0.91943359375, 0.174560546875,
            0.295654296875, 0.77783203125, 0.324951171875, 0.708984375
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.650390625,    0.10198974609375,  0.49365234375,
            0.322265625,    0.87060546875,     0.74072265625,
            0.7001953125,   0.7138671875,      0.826171875,
            0.42529296875,  0.57080078125,     0.54150390625,
            0.320556640625, 0.59130859375,     0.1961669921875,
            0.34228515625,  0.034515380859375, 0.669921875,
            0.356201171875, 0.358154296875
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {'options': {'aTranspose': false}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            1.892578125, 1.2265625, 1.8466796875, 1.8037109375, 1.423828125,
            1.3095703125, 0.94775390625, 1.0478515625, 1.224609375,
            0.8935546875, 1.1962890625, 0.775390625, 1.2802734375, 1.1826171875,
            0.90576171875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors options.bTranspose being true',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.58251953125, 0.024688720703125, 0.144775390625, 0.41162109375,
            0.837890625, 0.5419921875, 0.75341796875, 0.435546875,
            0.207275390625, 0.0222930908203125, 0.8896484375, 0.2578125
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.2349853515625, 0.39794921875, 0.64990234375,
            0.62744140625,   0.4853515625,  0.397216796875,
            0.6787109375,    0.65283203125, 0.2154541015625,
            0.7177734375,    0.70166015625, 0.98193359375,
            0.6708984375,    0.26318359375, 0.5478515625,
            0.69873046875,   0.63623046875, 0.006359100341796875,
            0.1671142578125, 0.60791015625
          ],
          'descriptor': {shape: [5, 4], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {'options': {'bTranspose': true}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            0.4990234375, 0.65966796875, 0.64892578125, 0.76416015625,
            0.64501953125, 1.17578125, 1.41796875, 1.525390625, 1.421875,
            0.92724609375, 0.79736328125, 0.88134765625, 0.93798828125, 0.8125,
            0.4375
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'gemm two float16 2D tensors options.bTranspose being explicit false',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.84814453125, 0.66748046875, 0.424072265625, 0.7294921875,
            0.93798828125, 0.787109375, 0.485595703125, 0.341064453125,
            0.364501953125, 0.398681640625, 0.1600341796875, 0.98828125
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.1717529296875,   0.51953125,        0.40576171875,
            0.73388671875,     0.1510009765625,   0.572265625,
            0.890625,          0.69970703125,     0.58642578125,
            0.09619140625,     0.9990234375,      0.6572265625,
            0.035247802734375, 0.251708984375,    0.462158203125,
            0.177001953125,    0.027984619140625, 0.1444091796875,
            0.77099609375,     0.11273193359375
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {'options': {'bTranspose': false}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            1.080078125, 1.333984375, 0.931640625, 1.6826171875, 0.470458984375,
            1.1572265625, 1.5166015625, 0.99755859375, 1.53515625,
            0.480224609375, 0.62548828125, 0.67724609375, 0.5751953125,
            1.3037109375, 0.27880859375
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm two float16 2D tensors all options',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.5048828125, 0.583984375, 0.11492919921875, 0.66845703125,
            0.5654296875, 0.470458984375, 0.76904296875, 0.03497314453125,
            0.76904296875, 0.413330078125, 0.300537109375, 0.73779296875
          ],
          'descriptor': {shape: [4, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.84814453125,    0.861328125,    0.9736328125,  0.7001953125,
            0.06622314453125, 0.79541015625,  0.5458984375,  0.712890625,
            0.58642578125,    0.219970703125, 0.92236328125, 0.09332275390625,
            0.1328125,        0.25439453125,  0.857421875,   0.794921875,
            0.60009765625,    0.422119140625, 0.5068359375,  0.1951904296875
          ],
          'descriptor': {shape: [5, 4], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.79150390625, 0.434326171875, 0.432861328125, 0.85205078125,
            0.314453125, 0.07110595703125, 0.8798828125, 0.037017822265625,
            0.68798828125, 0.84765625, 0.859375, 0.285400390625, 0.63720703125,
            0.6015625, 0.96826171875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {
            'options': {
              'c': 'inputC',
              'alpha': 74.43204170482103,
              'beta': 62.01828598608989,
              'aTranspose': true,
              'bTranspose': true
            }
          }
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            201.125, 122.1875, 115.5, 144, 98.0625, 95.75, 108.3125, 41.53125,
            79.1875, 102.125, 184.875, 116.5, 110.1875, 140.125, 119.6875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm both negative options.alpha and 1st float16 input tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.373779296875, 0.314208984375, 0.775390625, 0.712890625,
            0.1170654296875, 0.1820068359375, 0.81640625, 0.4609375,
            0.59912109375, 0.0888671875, 0.52734375, 0.65869140625
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.673828125,  0.45703125,     0.9619140625,  0.7275390625,
            0.9013671875, 0.837890625,    0.88330078125, 0.744140625,
            0.076171875,  0.283447265625, 0.59228515625, 0.47119140625,
            0.9794921875, 0.595703125,    0.578125,      0.71142578125,
            0.8974609375, 0.83447265625,  0.36962890625, 0.689453125
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.367431640625, 0.8427734375, 0.0340576171875, 0.7353515625,
            0.048309326171875, 0.9326171875, 0.623046875, 0.93017578125,
            0.049896240234375, 0.9501953125, 0.53369140625, 0.12109375,
            0.154296875, 0.9443359375, 0.293212890625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'c': 'inputC', 'alpha': -22.611149749186296}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            -33.125, -32.03125, -44, -22.359375, -30.828125, -22.640625,
            -22.28125, -31.453125, -17.03125, -20.453125, -27.9375, -26.828125,
            -38.46875, -21.671875, -29.65625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'gemm both negative options.alpha and 2nd float16 input tensor',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.7666015625, 0.97216796875, 0.0718994140625, 0.2430419921875,
            0.0016126632690429688, 0.544921875, 0.32177734375, 0.5556640625,
            0.4990234375, 0.384765625, 0.51171875, 0.18212890625
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.265625,        0.7099609375,      0.7265625,
            0.57080078125,   0.8330078125,      0.1624755859375,
            0.28759765625,   0.6552734375,      0.97509765625,
            0.42822265625,   0.016632080078125, 0.13818359375,
            0.1358642578125, 0.525390625,       0.384765625,
            0.17626953125,   0.10430908203125,  0.7548828125,
            0.8330078125,    0.2169189453125
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.285888671875, 0.1680908203125, 0.93798828125, 0.357177734375,
            0.07672119140625, 0.38671875, 0.5224609375, 0.83837890625,
            0.051422119140625, 0.173095703125, 0.00557708740234375,
            0.323486328125, 0.2587890625, 0.30322265625, 0.5576171875
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'c': 'inputC', 'alpha': -22.611149749186296}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            -8.8828125, -19.265625, -30.421875, -36.40625, -25.59375,
            -3.9609375, -5.36328125, -17.734375, -26.265625, -10.65625,
            -5.32421875, -12.21875, -18.328125, -24.125, -17.90625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'gemm both negative options.beta and 3rd float16 input tensor (options.c)',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.85400390625, 0.70458984375, 0.76416015625, 0.982421875,
            0.470458984375, 0.310546875, 0.68408203125, 0.83935546875,
            0.52294921875, 0.5234375, 0.79736328125, 0.10626220703125
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.955078125,    0.25537109375,  0.68505859375,   0.29296875,
            0.36962890625,  0.479248046875, 0.599609375,     0.16455078125,
            0.59375,        0.759765625,    0.325927734375,  0.9853515625,
            0.348388671875, 0.428466796875, 0.1644287109375, 0.371337890625,
            0.343505859375, 0.434814453125, 0.0859375,       0.148193359375
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.09210205078125, 0.869140625, 0.186279296875, 0.9111328125,
            0.80224609375, 0.70703125, 0.51904296875, 0.78466796875,
            0.43310546875, 0.884765625, 0.452880859375, 0.052764892578125,
            0.83740234375, 0.8994140625, 0.68896484375
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'c': 'inputC', 'beta': -11.68521964935509}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            0.69091796875, -8.421875, -0.7822265625, -9.5625, -8.25,
            -7.12890625, -4.796875, -8.1953125, -4.375, -9.6953125, -4.2421875,
            0.65283203125, -9.015625, -9.6953125, -7.3125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'gemm float16 input tensors with both negative options.alpha and options.beta',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.623046875, 0.5673828125, 0.568359375, 0.218017578125,
            0.29833984375, 0.425048828125, 0.59228515625, 0.09576416015625,
            0.14306640625, 0.66259765625, 0.34521484375, 0.505859375
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.65673828125, 0.182373046875, 0.61328125,     0.17578125,
            0.8076171875,  0.46923828125,  0.5927734375,   0.09747314453125,
            0.7041015625,  0.209228515625, 0.955078125,    0.716796875,
            0.955078125,   0.61669921875,  0.45556640625,  0.76611328125,
            0.69970703125, 0.58154296875,  0.492919921875, 0.6513671875
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.40087890625, 0.72509765625, 0.35595703125, 0.0589599609375,
            0.1856689453125, 0.923828125, 0.0650634765625, 0.169677734375,
            0.93603515625, 0.0208892822265625, 0.281005859375, 0.8193359375,
            0.256591796875, 0.043365478515625, 0.222900390625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'}, {
            'options': {
              'c': 'inputC',
              'alpha': -22.611149749186296,
              'beta': -11.68521964935509
            }
          }
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            -36, -31.3125, -29.1875, -22.546875, -25.296875, -34.1875,
            -18.796875, -21.109375, -28.21875, -15.2109375, -28.65625,
            -32.65625, -20.546875, -22.078125, -19.359375
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'gemm both negative options.alpha and 3rd float16 input tensor (options.c)',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            0.95068359375, 0.211669921875, 0.203857421875, 0.198486328125,
            0.9140625, 0.68994140625, 0.9951171875, 0.3525390625,
            0.029266357421875, 0.128662109375, 0.1787109375, 0.91357421875
          ],
          'descriptor': {shape: [3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            0.3720703125,   0.353271484375, 0.1610107421875,  0.301513671875,
            0.75830078125,  0.18359375,     0.06787109375,    0.029083251953125,
            0.9453125,      0.609375,       0.10418701171875, 0.80517578125,
            0.65625,        0.77197265625,  0.5888671875,     0.916015625,
            0.104736328125, 0.7138671875,   0.75341796875,    0.88623046875
          ],
          'descriptor': {shape: [4, 5], dataType: 'float16'},
          'constant': true
        },
        'inputC': {
          'data': [
            0.484375, 0.97412109375, 0.072509765625, 0.439697265625,
            0.46533203125, 0.97119140625, 0.84375, 0.0928955078125, 0.591796875,
            0.0114898681640625, 0.47265625, 0.7490234375, 0.1689453125,
            0.2379150390625, 0.69970703125
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'gemm',
        'arguments': [
          {'a': 'inputA'}, {'b': 'inputB'},
          {'options': {'c': 'inputC', 'alpha': -22.611149749186296}}
        ],
        'outputs': 'gemmOutput'
      }],
      'expectedOutputs': {
        'gemmOutput': {
          'data': [
            -12.984375, -11.125, -9.7578125, -17.5, -25.4375, -19.234375,
            -26.46875, -24.140625, -43.75, -45.46875, -19.65625, -5.09765625,
            -17.421875, -21.390625, -22.265625
          ],
          'descriptor': {shape: [3, 5], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(gemmTests, buildAndExecuteGraph, getPrecisionTolerance);
