// META: title=test WebNN API expand operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-expand
// Expand any dimension of size 1 of the input tensor to a larger size according
// to the new shape.
//
// MLOperand expand(
//     MLOperand input, sequence<[EnforceRange] unsigned long> newShape);

const expandTests = [
  {
    'name': 'expand float32 0D scalar to 1D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [24]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 0D scalar to 2D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [4, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 0D scalar to 3D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 3, 4]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 0D scalar to 4D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 2, 3]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 0D scalar to 5D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 3, 1, 2]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [2, 2, 3, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 1D constant tensor to 1D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [24]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 1D tensor to 1D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [24]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 1D tensor to 2D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [4, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 1D tensor to 3D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 3, 4]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 1D tensor to 4D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 2, 3]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 1D tensor to 5D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 3, 1, 2]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [2, 2, 3, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 2D tensor to 2D (1st dimension)',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [
            10.898762702941895, -29.391416549682617, -73.74250793457031,
            22.456905364990234, -97.5792465209961, -76.95013427734375
          ],
          'descriptor': {shape: [1, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [4, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            10.898762702941895, -29.391416549682617, -73.74250793457031,
            22.456905364990234, -97.5792465209961,   -76.95013427734375,
            10.898762702941895, -29.391416549682617, -73.74250793457031,
            22.456905364990234, -97.5792465209961,   -76.95013427734375,
            10.898762702941895, -29.391416549682617, -73.74250793457031,
            22.456905364990234, -97.5792465209961,   -76.95013427734375,
            10.898762702941895, -29.391416549682617, -73.74250793457031,
            22.456905364990234, -97.5792465209961,   -76.95013427734375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 2D tensor to 2D (2nd dimension)',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [
            4.965915679931641, 66.14382934570312, 75.28175354003906,
            49.998130798339844
          ],
          'descriptor': {shape: [4, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [4, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            4.965915679931641,  4.965915679931641,  4.965915679931641,
            4.965915679931641,  4.965915679931641,  4.965915679931641,
            66.14382934570312,  66.14382934570312,  66.14382934570312,
            66.14382934570312,  66.14382934570312,  66.14382934570312,
            75.28175354003906,  75.28175354003906,  75.28175354003906,
            75.28175354003906,  75.28175354003906,  75.28175354003906,
            49.998130798339844, 49.998130798339844, 49.998130798339844,
            49.998130798339844, 49.998130798339844, 49.998130798339844
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 2D tensor to 2D (all dimensions)',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [1, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [4, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 2D tensor to 3D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [
            4.965915679931641, 66.14382934570312, 75.28175354003906,
            49.998130798339844
          ],
          'descriptor': {shape: [4, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 4, 3]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            4.965915679931641,  4.965915679931641,  4.965915679931641,
            66.14382934570312,  66.14382934570312,  66.14382934570312,
            75.28175354003906,  75.28175354003906,  75.28175354003906,
            49.998130798339844, 49.998130798339844, 49.998130798339844,
            4.965915679931641,  4.965915679931641,  4.965915679931641,
            66.14382934570312,  66.14382934570312,  66.14382934570312,
            75.28175354003906,  75.28175354003906,  75.28175354003906,
            49.998130798339844, 49.998130798339844, 49.998130798339844
          ],
          'descriptor': {shape: [2, 4, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 2D tensor to 4D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [
            10.898762702941895, -29.391416549682617, -73.74250793457031,
            22.456905364990234, -97.5792465209961, -76.95013427734375
          ],
          'descriptor': {shape: [1, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 1, 2, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            10.898762702941895, -29.391416549682617, -73.74250793457031,
            22.456905364990234, -97.5792465209961,   -76.95013427734375,
            10.898762702941895, -29.391416549682617, -73.74250793457031,
            22.456905364990234, -97.5792465209961,   -76.95013427734375,
            10.898762702941895, -29.391416549682617, -73.74250793457031,
            22.456905364990234, -97.5792465209961,   -76.95013427734375,
            10.898762702941895, -29.391416549682617, -73.74250793457031,
            22.456905364990234, -97.5792465209961,   -76.95013427734375
          ],
          'descriptor': {shape: [2, 1, 2, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 2D tensor to 5D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.461850643157959],
          'descriptor': {shape: [1, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 1, 3, 2, 2]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959,
            -6.461850643157959, -6.461850643157959, -6.461850643157959
          ],
          'descriptor': {shape: [2, 1, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 3D tensor to 3D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [21.694129943847656, -72.82571411132812],
          'descriptor': {shape: [1, 2, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            -72.82571411132812, -72.82571411132812, -72.82571411132812,
            -72.82571411132812, -72.82571411132812, -72.82571411132812,
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            -72.82571411132812, -72.82571411132812, -72.82571411132812,
            -72.82571411132812, -72.82571411132812, -72.82571411132812
          ],
          'descriptor': {shape: [2, 2, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 3D tensor to 4D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [21.694129943847656, -72.82571411132812],
          'descriptor': {shape: [1, 2, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 2, 3]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            -72.82571411132812, -72.82571411132812, -72.82571411132812,
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            -72.82571411132812, -72.82571411132812, -72.82571411132812,
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            -72.82571411132812, -72.82571411132812, -72.82571411132812,
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            -72.82571411132812, -72.82571411132812, -72.82571411132812
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 3D tensor to 5D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [21.694129943847656, -72.82571411132812],
          'descriptor': {shape: [1, 2, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 1, 2, 2, 3]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            -72.82571411132812, -72.82571411132812, -72.82571411132812,
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            -72.82571411132812, -72.82571411132812, -72.82571411132812,
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            -72.82571411132812, -72.82571411132812, -72.82571411132812,
            21.694129943847656, 21.694129943847656, 21.694129943847656,
            -72.82571411132812, -72.82571411132812, -72.82571411132812
          ],
          'descriptor': {shape: [2, 1, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 4D tensor to 4D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [12.799123764038086, -26.550199508666992],
          'descriptor': {shape: [2, 1, 1, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 3, 2, 2]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            12.799123764038086,  12.799123764038086,  12.799123764038086,
            12.799123764038086,  12.799123764038086,  12.799123764038086,
            12.799123764038086,  12.799123764038086,  12.799123764038086,
            12.799123764038086,  12.799123764038086,  12.799123764038086,
            -26.550199508666992, -26.550199508666992, -26.550199508666992,
            -26.550199508666992, -26.550199508666992, -26.550199508666992,
            -26.550199508666992, -26.550199508666992, -26.550199508666992,
            -26.550199508666992, -26.550199508666992, -26.550199508666992
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'expand float32 4D tensor to 5D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [12.799123764038086, -26.550199508666992],
          'descriptor': {shape: [2, 1, 1, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 3, 1, 2]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            12.799123764038086,  12.799123764038086,  12.799123764038086,
            12.799123764038086,  12.799123764038086,  12.799123764038086,
            -26.550199508666992, -26.550199508666992, -26.550199508666992,
            -26.550199508666992, -26.550199508666992, -26.550199508666992,
            12.799123764038086,  12.799123764038086,  12.799123764038086,
            12.799123764038086,  12.799123764038086,  12.799123764038086,
            -26.550199508666992, -26.550199508666992, -26.550199508666992,
            -26.550199508666992, -26.550199508666992, -26.550199508666992
          ],
          'descriptor': {shape: [2, 2, 3, 1, 2], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'expand float16 0D scalar to 1D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [24]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 0D scalar to 2D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [4, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 0D scalar to 3D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 3, 4]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 0D scalar to 4D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 2, 3]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 0D scalar to 5D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 3, 1, 2]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [2, 2, 3, 1, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 1D constant tensor to 1D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [24]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 1D tensor to 1D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [24]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 1D tensor to 2D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [4, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 1D tensor to 3D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 3, 4]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 1D tensor to 4D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 2, 3]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 1D tensor to 5D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 3, 1, 2]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [2, 2, 3, 1, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 2D tensor to 2D (1st dimension)',
    'graph': {
      'inputs': {
        'expandInput': {
          'data':
              [10.8984375, -29.390625, -73.75, 22.453125, -97.5625, -76.9375],
          'descriptor': {shape: [1, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [4, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            10.8984375, -29.390625, -73.75, 22.453125, -97.5625, -76.9375,
            10.8984375, -29.390625, -73.75, 22.453125, -97.5625, -76.9375,
            10.8984375, -29.390625, -73.75, 22.453125, -97.5625, -76.9375,
            10.8984375, -29.390625, -73.75, 22.453125, -97.5625, -76.9375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 2D tensor to 2D (2nd dimension)',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [4.96484375, 66.125, 75.3125, 50],
          'descriptor': {shape: [4, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [4, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            4.96484375, 4.96484375, 4.96484375, 4.96484375, 4.96484375,
            4.96484375, 66.125,     66.125,     66.125,     66.125,
            66.125,     66.125,     75.3125,    75.3125,    75.3125,
            75.3125,    75.3125,    75.3125,    50,         50,
            50,         50,         50,         50
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 2D tensor to 2D (all dimensions)',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [1, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [4, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 2D tensor to 3D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [4.96484375, 66.125, 75.3125, 50],
          'descriptor': {shape: [4, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 4, 3]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            4.96484375, 4.96484375, 4.96484375, 66.125, 66.125, 66.125,
            75.3125,    75.3125,    75.3125,    50,     50,     50,
            4.96484375, 4.96484375, 4.96484375, 66.125, 66.125, 66.125,
            75.3125,    75.3125,    75.3125,    50,     50,     50
          ],
          'descriptor': {shape: [2, 4, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 2D tensor to 4D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data':
              [10.8984375, -29.390625, -73.75, 22.453125, -97.5625, -76.9375],
          'descriptor': {shape: [1, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 1, 2, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            10.8984375, -29.390625, -73.75, 22.453125, -97.5625, -76.9375,
            10.8984375, -29.390625, -73.75, 22.453125, -97.5625, -76.9375,
            10.8984375, -29.390625, -73.75, 22.453125, -97.5625, -76.9375,
            10.8984375, -29.390625, -73.75, 22.453125, -97.5625, -76.9375
          ],
          'descriptor': {shape: [2, 1, 2, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 2D tensor to 5D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [-6.4609375],
          'descriptor': {shape: [1, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 1, 3, 2, 2]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375, -6.4609375,
            -6.4609375, -6.4609375, -6.4609375, -6.4609375
          ],
          'descriptor': {shape: [2, 1, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 3D tensor to 3D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [21.6875, -72.8125],
          'descriptor': {shape: [1, 2, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 6]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            21.6875,  21.6875,  21.6875,  21.6875,  21.6875,  21.6875,
            -72.8125, -72.8125, -72.8125, -72.8125, -72.8125, -72.8125,
            21.6875,  21.6875,  21.6875,  21.6875,  21.6875,  21.6875,
            -72.8125, -72.8125, -72.8125, -72.8125, -72.8125, -72.8125
          ],
          'descriptor': {shape: [2, 2, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 3D tensor to 4D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [21.6875, -72.8125],
          'descriptor': {shape: [1, 2, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 2, 3]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            21.6875, 21.6875, 21.6875, -72.8125, -72.8125, -72.8125,
            21.6875, 21.6875, 21.6875, -72.8125, -72.8125, -72.8125,
            21.6875, 21.6875, 21.6875, -72.8125, -72.8125, -72.8125,
            21.6875, 21.6875, 21.6875, -72.8125, -72.8125, -72.8125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 3D tensor to 5D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [21.6875, -72.8125],
          'descriptor': {shape: [1, 2, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 1, 2, 2, 3]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            21.6875, 21.6875, 21.6875, -72.8125, -72.8125, -72.8125,
            21.6875, 21.6875, 21.6875, -72.8125, -72.8125, -72.8125,
            21.6875, 21.6875, 21.6875, -72.8125, -72.8125, -72.8125,
            21.6875, 21.6875, 21.6875, -72.8125, -72.8125, -72.8125
          ],
          'descriptor': {shape: [2, 1, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 4D tensor to 4D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [12.796875, -26.546875],
          'descriptor': {shape: [2, 1, 1, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 3, 2, 2]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            12.796875,  12.796875,  12.796875,  12.796875,  12.796875,
            12.796875,  12.796875,  12.796875,  12.796875,  12.796875,
            12.796875,  12.796875,  -26.546875, -26.546875, -26.546875,
            -26.546875, -26.546875, -26.546875, -26.546875, -26.546875,
            -26.546875, -26.546875, -26.546875, -26.546875
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'expand float16 4D tensor to 5D',
    'graph': {
      'inputs': {
        'expandInput': {
          'data': [12.796875, -26.546875],
          'descriptor': {shape: [2, 1, 1, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'expand',
        'arguments': [{'input': 'expandInput'}, {'newShape': [2, 2, 3, 1, 2]}],
        'outputs': 'expandOutput'
      }],
      'expectedOutputs': {
        'expandOutput': {
          'data': [
            12.796875,  12.796875,  12.796875,  12.796875,  12.796875,
            12.796875,  -26.546875, -26.546875, -26.546875, -26.546875,
            -26.546875, -26.546875, 12.796875,  12.796875,  12.796875,
            12.796875,  12.796875,  12.796875,  -26.546875, -26.546875,
            -26.546875, -26.546875, -26.546875, -26.546875
          ],
          'descriptor': {shape: [2, 2, 3, 1, 2], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(expandTests, buildAndExecuteGraph, getZeroULPTolerance);
