// META: title=test WebNN API element-wise div operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-binary
// Compute the element-wise binary division of the two input tensors.
// MLOperand div(MLOperand a, MLOperand b);


const getDivPrecisionTolerance = (graphResources) => {
  const toleranceValueDict = {float32: 2, float16: 2};
  const expectedDataType =
      getExpectedDataTypeOfSingleOutput(graphResources.expectedOutputs);
  return {metricType: 'ULP', value: toleranceValueDict[expectedDataType]};
};

const divTests = [
  {
    'name': 'div float32 1D constant tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.42374038696289,   -86.92247772216797,  -19.496112823486328,
            -15.150615692138672, 13.455190658569336,  45.433597564697266,
            61.082862854003906,  70.71882629394531,   -31.278579711914062,
            56.08354187011719,   38.992767333984375,  -3.27536940574646,
            32.28932189941406,   -3.676541805267334,  88.4349136352539,
            14.369060516357422,  13.943194389343262,  16.861190795898438,
            4.816806316375732,   44.15916442871094,   -13.083211898803711,
            44.56599807739258,   -34.892784118652344, -74.09375
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        },
        'inputB': {
          'data': [
            -95.0290298461914,   62.804866790771484,  -85.32865905761719,
            -68.20919799804688,  79.45568084716797,   -68.69049072265625,
            -94.46466827392578,  -10.000411033630371, 18.318864822387695,
            -3.6232800483703613, -5.957828044891357,  89.49882507324219,
            94.9579086303711,    -79.0005874633789,   -79.87596893310547,
            74.99787139892578,   25.865373611450195,  91.5443344116211,
            81.65287017822266,   48.2148323059082,    63.370121002197266,
            10.626384735107422,  46.126625061035156,  77.22327423095703
          ],
          'descriptor': {shape: [24], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.6358450651168823,  -1.3840086460113525,  0.22848258912563324,
            0.22211983799934387,  0.16934208571910858,  -0.6614248752593994,
            -0.6466212868690491,  -7.071591854095459,   -1.7074518203735352,
            -15.478666305541992,  -6.544795513153076,   -0.036596786230802536,
            0.3400382697582245,   0.046538159251213074, -1.1071529388427734,
            0.19159291684627533,  0.5390679836273193,   0.18418607115745544,
            0.058991268277168274, 0.9158834218978882,   -0.20645710825920105,
            4.193900108337402,    -0.7564564943313599,  -0.9594743251800537
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'div float32 1D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.42374038696289,   -86.92247772216797,  -19.496112823486328,
            -15.150615692138672, 13.455190658569336,  45.433597564697266,
            61.082862854003906,  70.71882629394531,   -31.278579711914062,
            56.08354187011719,   38.992767333984375,  -3.27536940574646,
            32.28932189941406,   -3.676541805267334,  88.4349136352539,
            14.369060516357422,  13.943194389343262,  16.861190795898438,
            4.816806316375732,   44.15916442871094,   -13.083211898803711,
            44.56599807739258,   -34.892784118652344, -74.09375
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -95.0290298461914,   62.804866790771484,  -85.32865905761719,
            -68.20919799804688,  79.45568084716797,   -68.69049072265625,
            -94.46466827392578,  -10.000411033630371, 18.318864822387695,
            -3.6232800483703613, -5.957828044891357,  89.49882507324219,
            94.9579086303711,    -79.0005874633789,   -79.87596893310547,
            74.99787139892578,   25.865373611450195,  91.5443344116211,
            81.65287017822266,   48.2148323059082,    63.370121002197266,
            10.626384735107422,  46.126625061035156,  77.22327423095703
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.6358450651168823,  -1.3840086460113525,  0.22848258912563324,
            0.22211983799934387,  0.16934208571910858,  -0.6614248752593994,
            -0.6466212868690491,  -7.071591854095459,   -1.7074518203735352,
            -15.478666305541992,  -6.544795513153076,   -0.036596786230802536,
            0.3400382697582245,   0.046538159251213074, -1.1071529388427734,
            0.19159291684627533,  0.5390679836273193,   0.18418607115745544,
            0.058991268277168274, 0.9158834218978882,   -0.20645710825920105,
            4.193900108337402,    -0.7564564943313599,  -0.9594743251800537
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'div float32 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.42374038696289,   -86.92247772216797,  -19.496112823486328,
            -15.150615692138672, 13.455190658569336,  45.433597564697266,
            61.082862854003906,  70.71882629394531,   -31.278579711914062,
            56.08354187011719,   38.992767333984375,  -3.27536940574646,
            32.28932189941406,   -3.676541805267334,  88.4349136352539,
            14.369060516357422,  13.943194389343262,  16.861190795898438,
            4.816806316375732,   44.15916442871094,   -13.083211898803711,
            44.56599807739258,   -34.892784118652344, -74.09375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -95.0290298461914,   62.804866790771484,  -85.32865905761719,
            -68.20919799804688,  79.45568084716797,   -68.69049072265625,
            -94.46466827392578,  -10.000411033630371, 18.318864822387695,
            -3.6232800483703613, -5.957828044891357,  89.49882507324219,
            94.9579086303711,    -79.0005874633789,   -79.87596893310547,
            74.99787139892578,   25.865373611450195,  91.5443344116211,
            81.65287017822266,   48.2148323059082,    63.370121002197266,
            10.626384735107422,  46.126625061035156,  77.22327423095703
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.6358450651168823,  -1.3840086460113525,  0.22848258912563324,
            0.22211983799934387,  0.16934208571910858,  -0.6614248752593994,
            -0.6466212868690491,  -7.071591854095459,   -1.7074518203735352,
            -15.478666305541992,  -6.544795513153076,   -0.036596786230802536,
            0.3400382697582245,   0.046538159251213074, -1.1071529388427734,
            0.19159291684627533,  0.5390679836273193,   0.18418607115745544,
            0.058991268277168274, 0.9158834218978882,   -0.20645710825920105,
            4.193900108337402,    -0.7564564943313599,  -0.9594743251800537
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'div float32 3D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.42374038696289,   -86.92247772216797,  -19.496112823486328,
            -15.150615692138672, 13.455190658569336,  45.433597564697266,
            61.082862854003906,  70.71882629394531,   -31.278579711914062,
            56.08354187011719,   38.992767333984375,  -3.27536940574646,
            32.28932189941406,   -3.676541805267334,  88.4349136352539,
            14.369060516357422,  13.943194389343262,  16.861190795898438,
            4.816806316375732,   44.15916442871094,   -13.083211898803711,
            44.56599807739258,   -34.892784118652344, -74.09375
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -95.0290298461914,   62.804866790771484,  -85.32865905761719,
            -68.20919799804688,  79.45568084716797,   -68.69049072265625,
            -94.46466827392578,  -10.000411033630371, 18.318864822387695,
            -3.6232800483703613, -5.957828044891357,  89.49882507324219,
            94.9579086303711,    -79.0005874633789,   -79.87596893310547,
            74.99787139892578,   25.865373611450195,  91.5443344116211,
            81.65287017822266,   48.2148323059082,    63.370121002197266,
            10.626384735107422,  46.126625061035156,  77.22327423095703
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.6358450651168823,  -1.3840086460113525,  0.22848258912563324,
            0.22211983799934387,  0.16934208571910858,  -0.6614248752593994,
            -0.6466212868690491,  -7.071591854095459,   -1.7074518203735352,
            -15.478666305541992,  -6.544795513153076,   -0.036596786230802536,
            0.3400382697582245,   0.046538159251213074, -1.1071529388427734,
            0.19159291684627533,  0.5390679836273193,   0.18418607115745544,
            0.058991268277168274, 0.9158834218978882,   -0.20645710825920105,
            4.193900108337402,    -0.7564564943313599,  -0.9594743251800537
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'div float32 4D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.42374038696289,   -86.92247772216797,  -19.496112823486328,
            -15.150615692138672, 13.455190658569336,  45.433597564697266,
            61.082862854003906,  70.71882629394531,   -31.278579711914062,
            56.08354187011719,   38.992767333984375,  -3.27536940574646,
            32.28932189941406,   -3.676541805267334,  88.4349136352539,
            14.369060516357422,  13.943194389343262,  16.861190795898438,
            4.816806316375732,   44.15916442871094,   -13.083211898803711,
            44.56599807739258,   -34.892784118652344, -74.09375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -95.0290298461914,   62.804866790771484,  -85.32865905761719,
            -68.20919799804688,  79.45568084716797,   -68.69049072265625,
            -94.46466827392578,  -10.000411033630371, 18.318864822387695,
            -3.6232800483703613, -5.957828044891357,  89.49882507324219,
            94.9579086303711,    -79.0005874633789,   -79.87596893310547,
            74.99787139892578,   25.865373611450195,  91.5443344116211,
            81.65287017822266,   48.2148323059082,    63.370121002197266,
            10.626384735107422,  46.126625061035156,  77.22327423095703
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.6358450651168823,  -1.3840086460113525,  0.22848258912563324,
            0.22211983799934387,  0.16934208571910858,  -0.6614248752593994,
            -0.6466212868690491,  -7.071591854095459,   -1.7074518203735352,
            -15.478666305541992,  -6.544795513153076,   -0.036596786230802536,
            0.3400382697582245,   0.046538159251213074, -1.1071529388427734,
            0.19159291684627533,  0.5390679836273193,   0.18418607115745544,
            0.058991268277168274, 0.9158834218978882,   -0.20645710825920105,
            4.193900108337402,    -0.7564564943313599,  -0.9594743251800537
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'div float32 5D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.42374038696289,   -86.92247772216797,  -19.496112823486328,
            -15.150615692138672, 13.455190658569336,  45.433597564697266,
            61.082862854003906,  70.71882629394531,   -31.278579711914062,
            56.08354187011719,   38.992767333984375,  -3.27536940574646,
            32.28932189941406,   -3.676541805267334,  88.4349136352539,
            14.369060516357422,  13.943194389343262,  16.861190795898438,
            4.816806316375732,   44.15916442871094,   -13.083211898803711,
            44.56599807739258,   -34.892784118652344, -74.09375
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            -95.0290298461914,   62.804866790771484,  -85.32865905761719,
            -68.20919799804688,  79.45568084716797,   -68.69049072265625,
            -94.46466827392578,  -10.000411033630371, 18.318864822387695,
            -3.6232800483703613, -5.957828044891357,  89.49882507324219,
            94.9579086303711,    -79.0005874633789,   -79.87596893310547,
            74.99787139892578,   25.865373611450195,  91.5443344116211,
            81.65287017822266,   48.2148323059082,    63.370121002197266,
            10.626384735107422,  46.126625061035156,  77.22327423095703
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.6358450651168823,  -1.3840086460113525,  0.22848258912563324,
            0.22211983799934387,  0.16934208571910858,  -0.6614248752593994,
            -0.6466212868690491,  -7.071591854095459,   -1.7074518203735352,
            -15.478666305541992,  -6.544795513153076,   -0.036596786230802536,
            0.3400382697582245,   0.046538159251213074, -1.1071529388427734,
            0.19159291684627533,  0.5390679836273193,   0.18418607115745544,
            0.058991268277168274, 0.9158834218978882,   -0.20645710825920105,
            4.193900108337402,    -0.7564564943313599,  -0.9594743251800537
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'div float32 broadcast 1D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-41.827415466308594],
          'descriptor': {shape: [1], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            60.42374038696289,   -86.92247772216797,  -19.496112823486328,
            -15.150615692138672, 13.455190658569336,  45.433597564697266,
            61.082862854003906,  70.71882629394531,   -31.278579711914062,
            56.08354187011719,   38.992767333984375,  -3.27536940574646,
            32.28932189941406,   -3.676541805267334,  88.4349136352539,
            14.369060516357422,  13.943194389343262,  16.861190795898438,
            4.816806316375732,   44.15916442871094,   -13.083211898803711,
            44.56599807739258,   -34.892784118652344, -74.09375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.6922348141670227, 0.48120367527008057, 2.145423412322998,
            2.7607734203338623,  -3.10864520072937,   -0.9206274151802063,
            -0.6847651600837708, -0.5914608240127563, 1.337254285812378,
            -0.7458055019378662, -1.0726968050003052, 12.770289421081543,
            -1.2953946590423584, 11.376836776733398,  -0.4729740023612976,
            -2.910935878753662,  -2.999844551086426,  -2.48069167137146,
            -8.683640480041504,  -0.9471967220306396, 3.1970295906066895,
            -0.9385499358177185, 1.19874107837677,    0.5645201802253723
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'div float32 broadcast 2D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.42374038696289,   -86.92247772216797,  -19.496112823486328,
            -15.150615692138672, 13.455190658569336,  45.433597564697266,
            61.082862854003906,  70.71882629394531,   -31.278579711914062,
            56.08354187011719,   38.992767333984375,  -3.27536940574646,
            32.28932189941406,   -3.676541805267334,  88.4349136352539,
            14.369060516357422,  13.943194389343262,  16.861190795898438,
            4.816806316375732,   44.15916442871094,   -13.083211898803711,
            44.56599807739258,   -34.892784118652344, -74.09375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            97.32406616210938, 36.325218200683594, 26.037858963012695,
            99.47166442871094, 10.395523071289062, -30.788942337036133
          ],
          'descriptor': {shape: [2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            0.6208509802818298,   -2.3928961753845215,  -0.7487602233886719,
            -0.15231086313724518, 1.2943254709243774,   -1.4756466150283813,
            0.627623438835144,    1.946824550628662,    -1.2012730836868286,
            0.5638142228126526,   3.7509193420410156,   0.106381356716156,
            0.33177119493484497,  -0.10121183097362518, 3.396397352218628,
            0.14445380866527557,  1.3412691354751587,   -0.5476378798484802,
            0.049492448568344116, 1.2156614065170288,   -0.5024688243865967,
            0.4480270743370056,   -3.356520175933838,   2.4065051078796387
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'div float32 broadcast 3D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.42374038696289,   -86.92247772216797,  -19.496112823486328,
            -15.150615692138672, 13.455190658569336,  45.433597564697266,
            61.082862854003906,  70.71882629394531,   -31.278579711914062,
            56.08354187011719,   38.992767333984375,  -3.27536940574646,
            32.28932189941406,   -3.676541805267334,  88.4349136352539,
            14.369060516357422,  13.943194389343262,  16.861190795898438,
            4.816806316375732,   44.15916442871094,   -13.083211898803711,
            44.56599807739258,   -34.892784118652344, -74.09375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            75.08295440673828, -46.22666931152344, 15.761880874633789,
            8.9222993850708
          ],
          'descriptor': {shape: [2, 2, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            0.8047597408294678,  -1.1576858758926392,  -0.2596609592437744,
            0.3277462124824524,  -0.2910698652267456,  -0.9828438758850098,
            3.8753535747528076,  4.48669958114624,     -1.9844446182250977,
            6.285772323608398,   4.370259761810303,    -0.36709925532341003,
            0.4300486445426941,  -0.04896639660000801, 1.177829384803772,
            -0.3108392059803009, -0.30162662267684937, -0.36475029587745667,
            0.3055984377861023,  2.801643133163452,    -0.830053985118866,
            4.994900703430176,   -3.910738945007324,   -8.304333686828613
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'div float32 broadcast 4D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-41.827415466308594],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float32'}
        },
        'inputB': {
          'data': [
            60.42374038696289,   -86.92247772216797,  -19.496112823486328,
            -15.150615692138672, 13.455190658569336,  45.433597564697266,
            61.082862854003906,  70.71882629394531,   -31.278579711914062,
            56.08354187011719,   38.992767333984375,  -3.27536940574646,
            32.28932189941406,   -3.676541805267334,  88.4349136352539,
            14.369060516357422,  13.943194389343262,  16.861190795898438,
            4.816806316375732,   44.15916442871094,   -13.083211898803711,
            44.56599807739258,   -34.892784118652344, -74.09375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.6922348141670227, 0.48120367527008057, 2.145423412322998,
            2.7607734203338623,  -3.10864520072937,   -0.9206274151802063,
            -0.6847651600837708, -0.5914608240127563, 1.337254285812378,
            -0.7458055019378662, -1.0726968050003052, 12.770289421081543,
            -1.2953946590423584, 11.376836776733398,  -0.4729740023612976,
            -2.910935878753662,  -2.999844551086426,  -2.48069167137146,
            -8.683640480041504,  -0.9471967220306396, 3.1970295906066895,
            -0.9385499358177185, 1.19874107837677,    0.5645201802253723
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'div float16 1D constant tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.4375,     -86.9375,     -19.5,     -15.1484375, 13.453125,
            45.4375,     61.09375,     70.75,     -31.28125,   56.09375,
            39,          -3.275390625, 32.28125,  -3.67578125, 88.4375,
            14.3671875,  13.9453125,   16.859375, 4.81640625,  44.15625,
            -13.0859375, 44.5625,      -34.90625, -74.125
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        },
        'inputB': {
          'data': [
            -95,      62.8125,  -85.3125, -68.1875,     79.4375,     -68.6875,
            -94.4375, -10,      18.3125,  -3.623046875, -5.95703125, 89.5,
            94.9375,  -79,      -79.875,  75,           25.859375,   91.5625,
            81.625,   48.21875, 63.375,   10.625,       46.125,      77.25
          ],
          'descriptor': {shape: [24], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.63623046875,   -1.3837890625,     0.228515625,
            0.22216796875,    0.1693115234375,   -0.66162109375,
            -0.64697265625,   -7.07421875,       -1.7080078125,
            -15.484375,       -6.546875,         -0.036590576171875,
            0.340087890625,   0.046539306640625, -1.107421875,
            0.1915283203125,  0.5390625,         0.18408203125,
            0.05902099609375, 0.91552734375,     -0.20654296875,
            4.1953125,        -0.7568359375,     -0.95947265625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'div float16 1D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.4375,     -86.9375,     -19.5,     -15.1484375, 13.453125,
            45.4375,     61.09375,     70.75,     -31.28125,   56.09375,
            39,          -3.275390625, 32.28125,  -3.67578125, 88.4375,
            14.3671875,  13.9453125,   16.859375, 4.81640625,  44.15625,
            -13.0859375, 44.5625,      -34.90625, -74.125
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -95,      62.8125,  -85.3125, -68.1875,     79.4375,     -68.6875,
            -94.4375, -10,      18.3125,  -3.623046875, -5.95703125, 89.5,
            94.9375,  -79,      -79.875,  75,           25.859375,   91.5625,
            81.625,   48.21875, 63.375,   10.625,       46.125,      77.25
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.63623046875,   -1.3837890625,     0.228515625,
            0.22216796875,    0.1693115234375,   -0.66162109375,
            -0.64697265625,   -7.07421875,       -1.7080078125,
            -15.484375,       -6.546875,         -0.036590576171875,
            0.340087890625,   0.046539306640625, -1.107421875,
            0.1915283203125,  0.5390625,         0.18408203125,
            0.05902099609375, 0.91552734375,     -0.20654296875,
            4.1953125,        -0.7568359375,     -0.95947265625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'div float16 2D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.4375,     -86.9375,     -19.5,     -15.1484375, 13.453125,
            45.4375,     61.09375,     70.75,     -31.28125,   56.09375,
            39,          -3.275390625, 32.28125,  -3.67578125, 88.4375,
            14.3671875,  13.9453125,   16.859375, 4.81640625,  44.15625,
            -13.0859375, 44.5625,      -34.90625, -74.125
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -95,      62.8125,  -85.3125, -68.1875,     79.4375,     -68.6875,
            -94.4375, -10,      18.3125,  -3.623046875, -5.95703125, 89.5,
            94.9375,  -79,      -79.875,  75,           25.859375,   91.5625,
            81.625,   48.21875, 63.375,   10.625,       46.125,      77.25
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.63623046875,   -1.3837890625,     0.228515625,
            0.22216796875,    0.1693115234375,   -0.66162109375,
            -0.64697265625,   -7.07421875,       -1.7080078125,
            -15.484375,       -6.546875,         -0.036590576171875,
            0.340087890625,   0.046539306640625, -1.107421875,
            0.1915283203125,  0.5390625,         0.18408203125,
            0.05902099609375, 0.91552734375,     -0.20654296875,
            4.1953125,        -0.7568359375,     -0.95947265625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'div float16 3D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.4375,     -86.9375,     -19.5,     -15.1484375, 13.453125,
            45.4375,     61.09375,     70.75,     -31.28125,   56.09375,
            39,          -3.275390625, 32.28125,  -3.67578125, 88.4375,
            14.3671875,  13.9453125,   16.859375, 4.81640625,  44.15625,
            -13.0859375, 44.5625,      -34.90625, -74.125
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -95,      62.8125,  -85.3125, -68.1875,     79.4375,     -68.6875,
            -94.4375, -10,      18.3125,  -3.623046875, -5.95703125, 89.5,
            94.9375,  -79,      -79.875,  75,           25.859375,   91.5625,
            81.625,   48.21875, 63.375,   10.625,       46.125,      77.25
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.63623046875,   -1.3837890625,     0.228515625,
            0.22216796875,    0.1693115234375,   -0.66162109375,
            -0.64697265625,   -7.07421875,       -1.7080078125,
            -15.484375,       -6.546875,         -0.036590576171875,
            0.340087890625,   0.046539306640625, -1.107421875,
            0.1915283203125,  0.5390625,         0.18408203125,
            0.05902099609375, 0.91552734375,     -0.20654296875,
            4.1953125,        -0.7568359375,     -0.95947265625
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'div float16 4D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.4375,     -86.9375,     -19.5,     -15.1484375, 13.453125,
            45.4375,     61.09375,     70.75,     -31.28125,   56.09375,
            39,          -3.275390625, 32.28125,  -3.67578125, 88.4375,
            14.3671875,  13.9453125,   16.859375, 4.81640625,  44.15625,
            -13.0859375, 44.5625,      -34.90625, -74.125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -95,      62.8125,  -85.3125, -68.1875,     79.4375,     -68.6875,
            -94.4375, -10,      18.3125,  -3.623046875, -5.95703125, 89.5,
            94.9375,  -79,      -79.875,  75,           25.859375,   91.5625,
            81.625,   48.21875, 63.375,   10.625,       46.125,      77.25
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.63623046875,   -1.3837890625,     0.228515625,
            0.22216796875,    0.1693115234375,   -0.66162109375,
            -0.64697265625,   -7.07421875,       -1.7080078125,
            -15.484375,       -6.546875,         -0.036590576171875,
            0.340087890625,   0.046539306640625, -1.107421875,
            0.1915283203125,  0.5390625,         0.18408203125,
            0.05902099609375, 0.91552734375,     -0.20654296875,
            4.1953125,        -0.7568359375,     -0.95947265625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'div float16 5D tensors',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.4375,     -86.9375,     -19.5,     -15.1484375, 13.453125,
            45.4375,     61.09375,     70.75,     -31.28125,   56.09375,
            39,          -3.275390625, 32.28125,  -3.67578125, 88.4375,
            14.3671875,  13.9453125,   16.859375, 4.81640625,  44.15625,
            -13.0859375, 44.5625,      -34.90625, -74.125
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            -95,      62.8125,  -85.3125, -68.1875,     79.4375,     -68.6875,
            -94.4375, -10,      18.3125,  -3.623046875, -5.95703125, 89.5,
            94.9375,  -79,      -79.875,  75,           25.859375,   91.5625,
            81.625,   48.21875, 63.375,   10.625,       46.125,      77.25
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.63623046875,   -1.3837890625,     0.228515625,
            0.22216796875,    0.1693115234375,   -0.66162109375,
            -0.64697265625,   -7.07421875,       -1.7080078125,
            -15.484375,       -6.546875,         -0.036590576171875,
            0.340087890625,   0.046539306640625, -1.107421875,
            0.1915283203125,  0.5390625,         0.18408203125,
            0.05902099609375, 0.91552734375,     -0.20654296875,
            4.1953125,        -0.7568359375,     -0.95947265625
          ],
          'descriptor': {shape: [2, 2, 1, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'div float16 broadcast 1D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-41.8125],
          'descriptor': {shape: [1], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            60.4375,     -86.9375,     -19.5,     -15.1484375, 13.453125,
            45.4375,     61.09375,     70.75,     -31.28125,   56.09375,
            39,          -3.275390625, 32.28125,  -3.67578125, 88.4375,
            14.3671875,  13.9453125,   16.859375, 4.81640625,  44.15625,
            -13.0859375, 44.5625,      -34.90625, -74.125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.69189453125, 0.48095703125,  2.14453125,      2.759765625,
            -3.107421875,   -0.92041015625, -0.6845703125,   -0.5908203125,
            1.3369140625,   -0.74560546875, -1.072265625,    12.765625,
            -1.294921875,   11.375,         -0.472900390625, -2.91015625,
            -2.998046875,   -2.48046875,    -8.6796875,      -0.94677734375,
            3.1953125,      -0.9384765625,  1.1982421875,    0.56396484375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'div float16 broadcast 2D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.4375,     -86.9375,     -19.5,     -15.1484375, 13.453125,
            45.4375,     61.09375,     70.75,     -31.28125,   56.09375,
            39,          -3.275390625, 32.28125,  -3.67578125, 88.4375,
            14.3671875,  13.9453125,   16.859375, 4.81640625,  44.15625,
            -13.0859375, 44.5625,      -34.90625, -74.125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [97.3125, 36.3125, 26.03125, 99.5, 10.3984375, -30.78125],
          'descriptor': {shape: [2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            0.62109375,       -2.39453125,      -0.7490234375,
            -0.1522216796875, 1.2939453125,     -1.4765625,
            0.6279296875,     1.9482421875,     -1.2021484375,
            0.56396484375,    3.75,             0.10638427734375,
            0.331787109375,   -0.1011962890625, 3.396484375,
            0.1444091796875,  1.3408203125,     -0.5478515625,
            0.04949951171875, 1.2158203125,     -0.5029296875,
            0.44775390625,    -3.357421875,     2.408203125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'div float16 broadcast 3D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [
            60.4375,     -86.9375,     -19.5,     -15.1484375, 13.453125,
            45.4375,     61.09375,     70.75,     -31.28125,   56.09375,
            39,          -3.275390625, 32.28125,  -3.67578125, 88.4375,
            14.3671875,  13.9453125,   16.859375, 4.81640625,  44.15625,
            -13.0859375, 44.5625,      -34.90625, -74.125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'inputB': {
          'data': [75.0625, -46.21875, 15.765625, 8.921875],
          'descriptor': {shape: [2, 2, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            0.80517578125, -1.158203125,       -0.259765625,   0.32763671875,
            -0.291015625,  -0.98291015625,     3.875,          4.48828125,
            -1.984375,     6.2890625,          4.37109375,     -0.3671875,
            0.43017578125, -0.048980712890625, 1.177734375,    -0.310791015625,
            -0.3017578125, -0.36474609375,     0.305419921875, 2.80078125,
            -0.830078125,  4.99609375,         -3.912109375,   -8.3046875
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'div float16 broadcast 4D to 4D',
    'graph': {
      'inputs': {
        'inputA': {
          'data': [-41.8125],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float16'}
        },
        'inputB': {
          'data': [
            60.4375,     -86.9375,     -19.5,     -15.1484375, 13.453125,
            45.4375,     61.09375,     70.75,     -31.28125,   56.09375,
            39,          -3.275390625, 32.28125,  -3.67578125, 88.4375,
            14.3671875,  13.9453125,   16.859375, 4.81640625,  44.15625,
            -13.0859375, 44.5625,      -34.90625, -74.125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'div',
        'arguments': [{'a': 'inputA'}, {'b': 'inputB'}],
        'outputs': 'output'
      }],
      'expectedOutputs': {
        'output': {
          'data': [
            -0.69189453125, 0.48095703125,  2.14453125,      2.759765625,
            -3.107421875,   -0.92041015625, -0.6845703125,   -0.5908203125,
            1.3369140625,   -0.74560546875, -1.072265625,    12.765625,
            -1.294921875,   11.375,         -0.472900390625, -2.91015625,
            -2.998046875,   -2.48046875,    -8.6796875,      -0.94677734375,
            3.1953125,      -0.9384765625,  1.1982421875,    0.56396484375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(
    divTests, buildAndExecuteGraph, getDivPrecisionTolerance);
