// META: title=test WebNN API conv2d operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-conv2d
// Compute a 2-D convolution given 4-D input and filter tensors.
//
// enum MLConv2dFilterOperandLayout {
//   "oihw",
//   "hwio",
//   "ohwi",
//   "ihwo"
// };
//
// dictionary MLConv2dOptions {
//   sequence<[EnforceRange] unsigned long> padding;
//   sequence<[EnforceRange] unsigned long> strides;
//   sequence<[EnforceRange] unsigned long> dilations;
//   [EnforceRange] unsigned long groups = 1;
//   MLInputOperandLayout inputLayout = "nchw";
//   MLConv2dFilterOperandLayout filterLayout = "oihw";
//   MLOperand bias;
// };
//
// MLOperand conv2d(
//     MLOperand input, MLOperand filter,
//     optional MLConv2dOptions options = {});


const conv2dTests = [
  {
    'name':
        'conv2d float32 4D both input and filter non-constant tensors default options',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6124474406242371,  0.8857858777046204,  0.13667134940624237,
            0.5645291209220886,  0.8965172171592712,  0.36792829632759094,
            0.6811466217041016,  0.0479511022567749,  0.33355462551116943,
            0.19882695376873016, 0.41167140007019043, 0.07934240251779556,
            0.4272463321685791,  0.535800576210022,   0.5910806059837341,
            0.28415432572364807, 0.4147258698940277,  0.026906268671154976,
            0.3621256649494171,  0.9945681691169739,  0.07184549421072006,
            0.12204372137784958, 0.8422137498855591,  0.4537501037120819,
            0.21529443562030792
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.3804761469364166, 0.5280312299728394, 0.21947036683559418,
            0.36689770221710205, 0.33974137902259827, 0.4200059771537781,
            0.3805030882358551, 0.19443586468696594, 0.5686976909637451
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [{'input': 'conv2dInput'}, {'filter': 'conv2dFilter'}],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.5323282480239868, 1.3573521375656128, 1.3641656637191772,
            1.071682333946228, 1.1259644031524658, 1.4713115692138672,
            1.078782320022583, 1.155018925666809, 1.656954288482666
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D both input and filter constant tensors default options',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6124474406242371,  0.8857858777046204,  0.13667134940624237,
            0.5645291209220886,  0.8965172171592712,  0.36792829632759094,
            0.6811466217041016,  0.0479511022567749,  0.33355462551116943,
            0.19882695376873016, 0.41167140007019043, 0.07934240251779556,
            0.4272463321685791,  0.535800576210022,   0.5910806059837341,
            0.28415432572364807, 0.4147258698940277,  0.026906268671154976,
            0.3621256649494171,  0.9945681691169739,  0.07184549421072006,
            0.12204372137784958, 0.8422137498855591,  0.4537501037120819,
            0.21529443562030792
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float32'},
          'constant': true
        },
        'conv2dFilter': {
          'data': [
            0.3804761469364166, 0.5280312299728394, 0.21947036683559418,
            0.36689770221710205, 0.33974137902259827, 0.4200059771537781,
            0.3805030882358551, 0.19443586468696594, 0.5686976909637451
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [{'input': 'conv2dInput'}, {'filter': 'conv2dFilter'}],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.5323282480239868, 1.3573521375656128, 1.3641656637191772,
            1.071682333946228, 1.1259644031524658, 1.4713115692138672,
            1.078782320022583, 1.155018925666809, 1.656954288482666
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'conv2d float32 4D input and filter tensors default options',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6124474406242371,  0.8857858777046204,  0.13667134940624237,
            0.5645291209220886,  0.8965172171592712,  0.36792829632759094,
            0.6811466217041016,  0.0479511022567749,  0.33355462551116943,
            0.19882695376873016, 0.41167140007019043, 0.07934240251779556,
            0.4272463321685791,  0.535800576210022,   0.5910806059837341,
            0.28415432572364807, 0.4147258698940277,  0.026906268671154976,
            0.3621256649494171,  0.9945681691169739,  0.07184549421072006,
            0.12204372137784958, 0.8422137498855591,  0.4537501037120819,
            0.21529443562030792
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.3804761469364166, 0.5280312299728394, 0.21947036683559418,
            0.36689770221710205, 0.33974137902259827, 0.4200059771537781,
            0.3805030882358551, 0.19443586468696594, 0.5686976909637451
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [{'input': 'conv2dInput'}, {'filter': 'conv2dFilter'}],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.5323282480239868, 1.3573521375656128, 1.3641656637191772,
            1.071682333946228, 1.1259644031524658, 1.4713115692138672,
            1.078782320022583, 1.155018925666809, 1.656954288482666
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'conv2d float32 4D input and filter tensors options.padding',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6124474406242371,  0.8857858777046204,  0.13667134940624237,
            0.5645291209220886,  0.8965172171592712,  0.36792829632759094,
            0.6811466217041016,  0.0479511022567749,  0.33355462551116943,
            0.19882695376873016, 0.41167140007019043, 0.07934240251779556,
            0.4272463321685791,  0.535800576210022,   0.5910806059837341,
            0.28415432572364807, 0.4147258698940277,  0.026906268671154976,
            0.3621256649494171,  0.9945681691169739,  0.07184549421072006,
            0.12204372137784958, 0.8422137498855591,  0.4537501037120819,
            0.21529443562030792
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.3804761469364166, 0.5280312299728394, 0.21947036683559418,
            0.36689770221710205, 0.33974137902259827, 0.4200059771537781,
            0.3805030882358551, 0.19443586468696594, 0.5686976909637451
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'padding': [1, 1, 1, 1]}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.0390141010284424, 0.882753312587738,  1.0667248964309692,
            0.8146538734436035, 0.6772860884666443, 1.0540467500686646,
            1.5323282480239868, 1.3573521375656128, 1.3641656637191772,
            1.1969101428985596, 0.8080586791038513, 1.071682333946228,
            1.1259644031524658, 1.4713115692138672, 0.960464596748352,
            0.5888903141021729, 1.078782320022583,  1.155018925666809,
            1.656954288482666,  1.2012416124343872, 0.3167303800582886,
            0.7545653581619263, 0.7729666829109192, 0.9733180403709412,
            0.9025675058364868
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'conv2d float32 4D input and filter tensors options.strides',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6124474406242371,  0.8857858777046204,  0.13667134940624237,
            0.5645291209220886,  0.8965172171592712,  0.36792829632759094,
            0.6811466217041016,  0.0479511022567749,  0.33355462551116943,
            0.19882695376873016, 0.41167140007019043, 0.07934240251779556,
            0.4272463321685791,  0.535800576210022,   0.5910806059837341,
            0.28415432572364807, 0.4147258698940277,  0.026906268671154976,
            0.3621256649494171,  0.9945681691169739,  0.07184549421072006,
            0.12204372137784958, 0.8422137498855591,  0.4537501037120819,
            0.21529443562030792
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.3804761469364166, 0.5280312299728394, 0.21947036683559418,
            0.36689770221710205, 0.33974137902259827, 0.4200059771537781,
            0.3805030882358551, 0.19443586468696594, 0.5686976909637451
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'strides': [2, 2]}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.5323282480239868, 1.3641656637191772, 1.078782320022583,
            1.656954288482666
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'conv2d float32 4D input and filter tensors options.dilations',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6124474406242371,  0.8857858777046204,  0.13667134940624237,
            0.5645291209220886,  0.8965172171592712,  0.36792829632759094,
            0.6811466217041016,  0.0479511022567749,  0.33355462551116943,
            0.19882695376873016, 0.41167140007019043, 0.07934240251779556,
            0.4272463321685791,  0.535800576210022,   0.5910806059837341,
            0.28415432572364807, 0.4147258698940277,  0.026906268671154976,
            0.3621256649494171,  0.9945681691169739,  0.07184549421072006,
            0.12204372137784958, 0.8422137498855591,  0.4537501037120819,
            0.21529443562030792
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.3804761469364166, 0.5280312299728394, 0.21947036683559418,
            0.36689770221710205, 0.33974137902259827, 0.4200059771537781,
            0.3805030882358551, 0.19443586468696594, 0.5686976909637451
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'dilations': [2, 2]}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [1.3599307537078857],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'depthwise conv2d float32 4D input and filter tensors options.groups= input_channels',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.8444867730140686, 0.9432409405708313, 0.6556113362312317,
            0.6982811689376831, 0.9994443655014038, 0.23663610219955444,
            0.36740678548812866, 0.2619246542453766, 0.6254158616065979,
            0.8403863906860352, 0.3783077001571655, 0.4543924033641815,
            0.25327208638191223, 0.5780375599861145, 0.5414554476737976,
            0.37846308946609497
          ],
          'descriptor': {shape: [1, 4, 2, 2], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.27221617102622986, 0.281202495098114, 0.854483962059021,
            0.1796930730342865, 0.7762278318405151, 0.5140685439109802,
            0.6374202966690063, 0.12801742553710938, 0.8373776078224182,
            0.5726001858711243, 0.09855203330516815, 0.5929878950119019,
            0.5900803804397583, 0.9690897464752197, 0.23175589740276337,
            0.14805112779140472
          ],
          'descriptor': {shape: [4, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'groups': 4}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.1808103322982788, 1.165167212486267, 1.311646819114685,
            0.8911385536193848
          ],
          'descriptor': {shape: [1, 4, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors options.inputLayout=\'nchw\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.9671129584312439, 0.10836050659418106,
            0.3202308118343353, 0.6952692270278931, 0.5070913434028625,
            0.08139707148075104, 0.5303338766098022, 0.3072136342525482,
            0.43241235613822937, 0.9849002361297607, 0.4281076192855835
          ],
          'descriptor': {shape: [3, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nchw'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.8845428228378296,  0.7413608431816101, 0.2897796928882599,
            0.4053896367549896,  0.9626783132553101, 0.9108520746231079,
            0.4832426905632019,  0.4878997206687927, 0.8020333051681519,
            0.6277193427085876,  0.4483422338962555, 0.8711439371109009,
            0.6932874917984009,  1.0369365215301514, 0.8282973766326904,
            0.35335418581962585, 1.1787647008895874, 0.8123774528503418,
            0.816078782081604,   0.6780439019203186, 0.9170808792114258,
            1.082636833190918,   1.2353861331939697, 0.9810346961021423
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors options.inputLayout=\'nhwc\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.9671129584312439, 0.10836050659418106,
            0.3202308118343353, 0.6952692270278931, 0.5070913434028625,
            0.08139707148075104, 0.5303338766098022, 0.3072136342525482,
            0.43241235613822937, 0.9849002361297607, 0.4281076192855835
          ],
          'descriptor': {shape: [3, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nhwc'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.8845428228378296,  0.9626783132553101, 0.8020333051681519,
            0.7413608431816101,  0.9108520746231079, 0.6277193427085876,
            0.2897796928882599,  0.4832426905632019, 0.4483422338962555,
            0.4053896367549896,  0.4878997206687927, 0.8711439371109009,
            0.6932874917984009,  1.1787647008895874, 0.9170808792114258,
            1.0369365215301514,  0.8123774528503418, 1.082636833190918,
            0.8282973766326904,  0.816078782081604,  1.2353861331939697,
            0.35335418581962585, 0.6780439019203186, 0.9810346961021423
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors options.filterLayout=\'oihw\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.9671129584312439, 0.10836050659418106,
            0.3202308118343353, 0.6952692270278931, 0.5070913434028625,
            0.08139707148075104, 0.5303338766098022, 0.3072136342525482,
            0.43241235613822937, 0.9849002361297607, 0.4281076192855835
          ],
          'descriptor': {shape: [3, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'filterLayout': 'oihw'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.8845428228378296,  0.7413608431816101, 0.2897796928882599,
            0.4053896367549896,  0.9626783132553101, 0.9108520746231079,
            0.4832426905632019,  0.4878997206687927, 0.8020333051681519,
            0.6277193427085876,  0.4483422338962555, 0.8711439371109009,
            0.6932874917984009,  1.0369365215301514, 0.8282973766326904,
            0.35335418581962585, 1.1787647008895874, 0.8123774528503418,
            0.816078782081604,   0.6780439019203186, 0.9170808792114258,
            1.082636833190918,   1.2353861331939697, 0.9810346961021423
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors options.filterLayout=\'hwio\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.6952692270278931, 0.3072136342525482,
            0.9671129584312439, 0.5070913434028625, 0.43241235613822937,
            0.10836050659418106, 0.08139707148075104, 0.9849002361297607,
            0.3202308118343353, 0.5303338766098022, 0.4281076192855835
          ],
          'descriptor': {shape: [2, 2, 1, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'filterLayout': 'hwio'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.8845428228378296,  0.7413608431816101, 0.2897796928882599,
            0.4053896367549896,  0.9626783132553101, 0.9108520746231079,
            0.4832426905632019,  0.4878997206687927, 0.8020333051681519,
            0.6277193427085876,  0.4483422338962555, 0.8711439371109009,
            0.6932874917984009,  1.0369365215301514, 0.8282973766326904,
            0.35335418581962585, 1.1787647008895874, 0.8123774528503418,
            0.816078782081604,   0.6780439019203186, 0.9170808792114258,
            1.082636833190918,   1.2353861331939697, 0.9810346961021423
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors options.filterLayout=\'ohwi\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.9671129584312439, 0.10836050659418106,
            0.3202308118343353, 0.6952692270278931, 0.5070913434028625,
            0.08139707148075104, 0.5303338766098022, 0.3072136342525482,
            0.43241235613822937, 0.9849002361297607, 0.4281076192855835
          ],
          'descriptor': {shape: [3, 2, 2, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'filterLayout': 'ohwi'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.8845428228378296,  0.7413608431816101, 0.2897796928882599,
            0.4053896367549896,  0.9626783132553101, 0.9108520746231079,
            0.4832426905632019,  0.4878997206687927, 0.8020333051681519,
            0.6277193427085876,  0.4483422338962555, 0.8711439371109009,
            0.6932874917984009,  1.0369365215301514, 0.8282973766326904,
            0.35335418581962585, 1.1787647008895874, 0.8123774528503418,
            0.816078782081604,   0.6780439019203186, 0.9170808792114258,
            1.082636833190918,   1.2353861331939697, 0.9810346961021423
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors options.filterLayout=\'ihwo\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.6952692270278931, 0.3072136342525482,
            0.9671129584312439, 0.5070913434028625, 0.43241235613822937,
            0.10836050659418106, 0.08139707148075104, 0.9849002361297607,
            0.3202308118343353, 0.5303338766098022, 0.4281076192855835
          ],
          'descriptor': {shape: [1, 2, 2, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'filterLayout': 'ihwo'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.8845428228378296,  0.7413608431816101, 0.2897796928882599,
            0.4053896367549896,  0.9626783132553101, 0.9108520746231079,
            0.4832426905632019,  0.4878997206687927, 0.8020333051681519,
            0.6277193427085876,  0.4483422338962555, 0.8711439371109009,
            0.6932874917984009,  1.0369365215301514, 0.8282973766326904,
            0.35335418581962585, 1.1787647008895874, 0.8123774528503418,
            0.816078782081604,   0.6780439019203186, 0.9170808792114258,
            1.082636833190918,   1.2353861331939697, 0.9810346961021423
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors options.inputLayout=\'nhwc\' and options.filterLayout=\'oihw\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.9671129584312439, 0.10836050659418106,
            0.3202308118343353, 0.6952692270278931, 0.5070913434028625,
            0.08139707148075104, 0.5303338766098022, 0.3072136342525482,
            0.43241235613822937, 0.9849002361297607, 0.4281076192855835
          ],
          'descriptor': {shape: [3, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'oihw'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.8845428228378296,  0.9626783132553101, 0.8020333051681519,
            0.7413608431816101,  0.9108520746231079, 0.6277193427085876,
            0.2897796928882599,  0.4832426905632019, 0.4483422338962555,
            0.4053896367549896,  0.4878997206687927, 0.8711439371109009,
            0.6932874917984009,  1.1787647008895874, 0.9170808792114258,
            1.0369365215301514,  0.8123774528503418, 1.082636833190918,
            0.8282973766326904,  0.816078782081604,  1.2353861331939697,
            0.35335418581962585, 0.6780439019203186, 0.9810346961021423
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors options.inputLayout=\'nhwc\' and options.filterLayout=\'hwio\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.6952692270278931, 0.3072136342525482,
            0.9671129584312439, 0.5070913434028625, 0.43241235613822937,
            0.10836050659418106, 0.08139707148075104, 0.9849002361297607,
            0.3202308118343353, 0.5303338766098022, 0.4281076192855835
          ],
          'descriptor': {shape: [2, 2, 1, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'hwio'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.8845428228378296,  0.9626783132553101, 0.8020333051681519,
            0.7413608431816101,  0.9108520746231079, 0.6277193427085876,
            0.2897796928882599,  0.4832426905632019, 0.4483422338962555,
            0.4053896367549896,  0.4878997206687927, 0.8711439371109009,
            0.6932874917984009,  1.1787647008895874, 0.9170808792114258,
            1.0369365215301514,  0.8123774528503418, 1.082636833190918,
            0.8282973766326904,  0.816078782081604,  1.2353861331939697,
            0.35335418581962585, 0.6780439019203186, 0.9810346961021423
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors options.inputLayout=\'nhwc\' and options.filterLayout=\'ohwi\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.9671129584312439, 0.10836050659418106,
            0.3202308118343353, 0.6952692270278931, 0.5070913434028625,
            0.08139707148075104, 0.5303338766098022, 0.3072136342525482,
            0.43241235613822937, 0.9849002361297607, 0.4281076192855835
          ],
          'descriptor': {shape: [3, 2, 2, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'ohwi'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.8845428228378296,  0.9626783132553101, 0.8020333051681519,
            0.7413608431816101,  0.9108520746231079, 0.6277193427085876,
            0.2897796928882599,  0.4832426905632019, 0.4483422338962555,
            0.4053896367549896,  0.4878997206687927, 0.8711439371109009,
            0.6932874917984009,  1.1787647008895874, 0.9170808792114258,
            1.0369365215301514,  0.8123774528503418, 1.082636833190918,
            0.8282973766326904,  0.816078782081604,  1.2353861331939697,
            0.35335418581962585, 0.6780439019203186, 0.9810346961021423
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors options.inputLayout=\'nhwc\' and options.filterLayout=\'ihwo\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.6952692270278931, 0.3072136342525482,
            0.9671129584312439, 0.5070913434028625, 0.43241235613822937,
            0.10836050659418106, 0.08139707148075104, 0.9849002361297607,
            0.3202308118343353, 0.5303338766098022, 0.4281076192855835
          ],
          'descriptor': {shape: [1, 2, 2, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'ihwo'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.8845428228378296,  0.9626783132553101, 0.8020333051681519,
            0.7413608431816101,  0.9108520746231079, 0.6277193427085876,
            0.2897796928882599,  0.4832426905632019, 0.4483422338962555,
            0.4053896367549896,  0.4878997206687927, 0.8711439371109009,
            0.6932874917984009,  1.1787647008895874, 0.9170808792114258,
            1.0369365215301514,  0.8123774528503418, 1.082636833190918,
            0.8282973766326904,  0.816078782081604,  1.2353861331939697,
            0.35335418581962585, 0.6780439019203186, 0.9810346961021423
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'conv2d float32 4D input and filter tensors 1D options.bias',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529087066650391, 0.7520291805267334, 0.5949527621269226,
            0.2163185328245163, 0.07589349150657654, 0.151067852973938,
            0.1212485060095787, 0.5364335179328918, 0.5937089920043945,
            0.991003155708313, 0.3630942404270172, 0.9289674162864685,
            0.22727376222610474, 0.5414124131202698, 0.08445341885089874,
            0.6765284538269043, 0.6193256378173828, 0.3929215967655182
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.14543837308883667, 0.9671129584312439, 0.10836050659418106,
            0.3202308118343353, 0.6952692270278931, 0.5070913434028625,
            0.08139707148075104, 0.5303338766098022, 0.3072136342525482,
            0.43241235613822937, 0.9849002361297607, 0.4281076192855835
          ],
          'descriptor': {shape: [3, 1, 2, 2], dataType: 'float32'},
          'constant': true
        },
        'conv2dBias': {
          'data': [0.8135762214660645, 0.8394582867622375, 0.49444812536239624],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'bias': 'conv2dBias'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.698119044303894,  1.5549371242523193, 1.103355884552002,
            1.2189658880233765, 1.8021366596221924, 1.7503103017807007,
            1.3227009773254395, 1.3273580074310303, 1.2964813709259033,
            1.1221674680709839, 0.9427903890609741, 1.365592122077942,
            1.5068637132644653, 1.8505127429962158, 1.6418735980987549,
            1.1669304370880127, 2.0182230472564697, 1.6518357992172241,
            1.6555371284484863, 1.5175021886825562, 1.4115289449691772,
            1.577085018157959,  1.7298341989517212, 1.4754828214645386
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'conv2d float32 4D input and filter tensors all options',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.09971386939287186,  0.5374298095703125,  0.30570074915885925,
            0.7222362160682678,   0.5066556334495544,  0.3238430619239807,
            0.8721967935562134,   0.20989856123924255, 0.5052645802497864,
            0.026870572939515114, 0.5497115850448608,  0.06430311501026154,
            0.1560668647289276,   0.11970008909702301, 0.036145713180303574,
            0.41425615549087524,  0.2417246550321579,  0.6771785020828247,
            0.2087000161409378,   0.10979551076889038, 0.745035707950592,
            0.7443592548370361,   0.7873413562774658,  0.5887080430984497,
            0.11018028855323792,  0.9045036435127258,  0.11725221574306488,
            0.8508669137954712,   0.4244919717311859,  0.02537914551794529,
            0.747390866279602,    0.4645859897136688,  0.030408121645450592,
            0.4244018793106079,   0.3847547769546509,  0.7581132650375366,
            0.9901952147483826,   0.03716852888464928, 0.014496322721242905,
            0.8263142108917236,   0.21072064340114594, 0.6569713950157166,
            0.25842806696891785,  0.4802338480949402,  0.9704219102859497,
            0.2968284785747528,   0.7524365782737732,  0.029636209830641747,
            0.09028015285730362,  0.77818763256073
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.6385681629180908, 0.07764163613319397, 0.1291629821062088,
            0.45633891224861145, 0.40438535809516907, 0.5943626761436462,
            0.14241264760494232, 0.9036700129508972
          ],
          'descriptor': {shape: [2, 2, 1, 2], dataType: 'float32'},
          'constant': true
        },
        'conv2dBias': {
          'data': [0.542375385761261, 0.8406118750572205],
          'descriptor': {shape: [2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'}, {
            'options': {
              'padding': [1, 0, 0, 1],
              'strides': [1, 1],
              'dilations': [1, 1],
              'groups': 2,
              'inputLayout': 'nchw',
              'filterLayout': 'hwio',
              'bias': 'conv2dBias'
            }
          }
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.6592350006103516, 0.8032397627830505, 0.7688518762588501,
            0.9065912961959839, 0.7472594976425171, 0.9306347966194153,
            1.3076419830322266, 0.987708330154419,  1.277161955833435,
            0.8767756223678589, 1.0932797193527222, 1.1746727228164673,
            0.8218293786048889, 0.9220445156097412, 0.5741508603096008,
            1.103653073310852,  0.7977840900421143, 0.9610581398010254,
            0.7235122323036194, 0.6098566055297852, 1.2454158067703247,
            1.1973347663879395, 1.4039851427078247, 0.9435820579528809,
            0.6570426225662231, 1.4841723442077637, 1.6792051792144775,
            1.729936122894287,  1.115848422050476,  0.8556963205337524,
            1.828399419784546,  1.5416107177734375, 1.5019794702529907,
            1.4850915670394897, 1.0712661743164062, 2.4560532569885254,
            1.5126826763153076, 1.0718353986740112, 1.8044731616973877,
            1.3616151809692383, 2.07026743888855,   1.5584666728973389,
            1.4376858472824097, 2.3811910152435303, 1.4815508127212524,
            2.0131523609161377, 1.4835525751113892, 1.1790242195129395,
            2.0776233673095703, 1.378482699394226
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float32 4D input and filter tensors, both negative input tensor and options.bias',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            -0.8073334693908691,  -0.8839999437332153,  -0.7700487375259399,
            -0.5646049380302429,  -0.39717939496040344, -0.10841356962919235,
            -0.5519214868545532,  -0.3954172134399414,  -0.057589758187532425,
            -0.5144240856170654,  -0.21321901679039001, -0.950609028339386,
            -0.8043696880340576,  -0.8646378517150879,  -0.9607220888137817,
            -0.3264438509941101,  -0.06884296983480453, -0.3203399181365967,
            -0.2692728042602539,  -0.3430887758731842,  -0.8989502191543579,
            -0.9038569331169128,  -0.6369403004646301,  -0.20070797204971313,
            -0.7870702147483826,  -0.3467883765697479,  -0.060042694211006165,
            -0.14985208213329315, -0.6482332348823547,  -0.8934088349342346,
            -0.8149284720420837,  -0.6423668265342712,  -0.032736241817474365,
            -0.6608918905258179,  -0.5843491554260254,  -0.09921254217624664,
            -0.16602523624897003, -0.9508541822433472,  -0.3051462769508362,
            -0.6210587024688721,  -0.5400903820991516,  -0.42009180784225464,
            -0.18824540078639984, -0.3588937520980835,  -0.7114293575286865,
            -0.3751019835472107,  -0.7108227610588074,  -0.36050301790237427,
            -0.5468712449073792,  -0.032261595129966736
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        },
        'conv2dFilter': {
          'data': [
            0.6385681629180908, 0.07764163613319397, 0.1291629821062088,
            0.45633891224861145, 0.40438535809516907, 0.5943626761436462,
            0.14241264760494232, 0.9036700129508972
          ],
          'descriptor': {shape: [2, 2, 1, 2], dataType: 'float32'},
          'constant': true
        },
        'conv2dBias': {
          'data': [-0.37496936321258545, -0.4363507032394409],
          'descriptor': {shape: [2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'}, {
            'options': {
              'padding': [1, 0, 0, 1],
              'groups': 2,
              'filterLayout': 'hwio',
              'bias': 'conv2dBias'
            }
          }
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            -0.8273359537124634, -0.8421106934547424, -0.7667726874351501,
            -0.6598507165908813, -0.5355829000473022, -1.1271283626556396,
            -1.3184267282485962, -1.1077264547348022, -0.8833579421043396,
            -0.8366210460662842, -0.7370880246162415, -1.2774468660354614,
            -1.0833193063735962, -0.9646547436714172, -1.091966152191162,
            -0.7757209539413452, -1.1593523025512695, -1.1681820154190063,
            -1.2089394330978394, -1.127195954322815,  -1.0845609903335571,
            -0.9165211915969849, -0.9004610180854797, -0.78448486328125,
            -0.9123346209526062, -0.6967275738716125, -0.6074546575546265,
            -1.1112061738967896, -1.6289831399917603, -0.9673595428466797,
            -1.5555264949798584, -0.9207774996757507, -1.3604848384857178,
            -1.8152461051940918, -0.8530317544937134, -1.0017603635787964,
            -1.4591015577316284, -1.5813868045806885, -1.4969244003295898,
            -0.8508546352386475, -1.2204514741897583, -1.3029515743255615,
            -1.0856342315673828, -1.5996664762496948, -0.9074177742004395,
            -1.5352842807769775, -1.303133249282837,  -1.3232042789459229,
            -1.1430623531341553, -0.5107623338699341
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name':
        'conv2d float16 4D both input and filter non-constant tensors default options',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6123046875,    0.8857421875,      0.13671875,
            0.564453125,     0.896484375,       0.367919921875,
            0.68115234375,   0.047943115234375, 0.33349609375,
            0.1988525390625, 0.41162109375,     0.079345703125,
            0.42724609375,   0.53564453125,     0.59130859375,
            0.2841796875,    0.414794921875,    0.0269012451171875,
            0.362060546875,  0.99462890625,     0.07183837890625,
            0.1220703125,    0.84228515625,     0.453857421875,
            0.21533203125
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.38037109375, 0.52783203125, 0.219482421875, 0.366943359375,
            0.33984375, 0.419921875, 0.380615234375, 0.1944580078125,
            0.56884765625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [{'input': 'conv2dInput'}, {'filter': 'conv2dFilter'}],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.5322265625, 1.357421875, 1.3642578125, 1.0712890625, 1.1259765625,
            1.4716796875, 1.0791015625, 1.1552734375, 1.6572265625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D both input and filter constant tensors default options',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6123046875,    0.8857421875,      0.13671875,
            0.564453125,     0.896484375,       0.367919921875,
            0.68115234375,   0.047943115234375, 0.33349609375,
            0.1988525390625, 0.41162109375,     0.079345703125,
            0.42724609375,   0.53564453125,     0.59130859375,
            0.2841796875,    0.414794921875,    0.0269012451171875,
            0.362060546875,  0.99462890625,     0.07183837890625,
            0.1220703125,    0.84228515625,     0.453857421875,
            0.21533203125
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float16'},
          'constant': true
        },
        'conv2dFilter': {
          'data': [
            0.38037109375, 0.52783203125, 0.219482421875, 0.366943359375,
            0.33984375, 0.419921875, 0.380615234375, 0.1944580078125,
            0.56884765625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [{'input': 'conv2dInput'}, {'filter': 'conv2dFilter'}],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.5322265625, 1.357421875, 1.3642578125, 1.0712890625, 1.1259765625,
            1.4716796875, 1.0791015625, 1.1552734375, 1.6572265625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'conv2d float16 4D input and filter tensors default options',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6123046875,    0.8857421875,      0.13671875,
            0.564453125,     0.896484375,       0.367919921875,
            0.68115234375,   0.047943115234375, 0.33349609375,
            0.1988525390625, 0.41162109375,     0.079345703125,
            0.42724609375,   0.53564453125,     0.59130859375,
            0.2841796875,    0.414794921875,    0.0269012451171875,
            0.362060546875,  0.99462890625,     0.07183837890625,
            0.1220703125,    0.84228515625,     0.453857421875,
            0.21533203125
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.38037109375, 0.52783203125, 0.219482421875, 0.366943359375,
            0.33984375, 0.419921875, 0.380615234375, 0.1944580078125,
            0.56884765625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [{'input': 'conv2dInput'}, {'filter': 'conv2dFilter'}],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.5322265625, 1.357421875, 1.3642578125, 1.0712890625, 1.1259765625,
            1.4716796875, 1.0791015625, 1.1552734375, 1.6572265625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'conv2d float16 4D input and filter tensors options.padding',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6123046875,    0.8857421875,      0.13671875,
            0.564453125,     0.896484375,       0.367919921875,
            0.68115234375,   0.047943115234375, 0.33349609375,
            0.1988525390625, 0.41162109375,     0.079345703125,
            0.42724609375,   0.53564453125,     0.59130859375,
            0.2841796875,    0.414794921875,    0.0269012451171875,
            0.362060546875,  0.99462890625,     0.07183837890625,
            0.1220703125,    0.84228515625,     0.453857421875,
            0.21533203125
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.38037109375, 0.52783203125, 0.219482421875, 0.366943359375,
            0.33984375, 0.419921875, 0.380615234375, 0.1944580078125,
            0.56884765625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'padding': [1, 1, 1, 1]}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.0390625,      0.8828125,     1.06640625,    0.814453125,
            0.67724609375,  1.0537109375,  1.5322265625,  1.357421875,
            1.3642578125,   1.1962890625,  0.80810546875, 1.0712890625,
            1.1259765625,   1.4716796875,  0.96044921875, 0.5888671875,
            1.0791015625,   1.1552734375,  1.6572265625,  1.201171875,
            0.316650390625, 0.75439453125, 0.77294921875, 0.97314453125,
            0.90234375
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'conv2d float16 4D input and filter tensors options.strides',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6123046875,    0.8857421875,      0.13671875,
            0.564453125,     0.896484375,       0.367919921875,
            0.68115234375,   0.047943115234375, 0.33349609375,
            0.1988525390625, 0.41162109375,     0.079345703125,
            0.42724609375,   0.53564453125,     0.59130859375,
            0.2841796875,    0.414794921875,    0.0269012451171875,
            0.362060546875,  0.99462890625,     0.07183837890625,
            0.1220703125,    0.84228515625,     0.453857421875,
            0.21533203125
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.38037109375, 0.52783203125, 0.219482421875, 0.366943359375,
            0.33984375, 0.419921875, 0.380615234375, 0.1944580078125,
            0.56884765625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'strides': [2, 2]}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [1.5322265625, 1.3642578125, 1.0791015625, 1.6572265625],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'conv2d float16 4D input and filter tensors options.dilations',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.6123046875,    0.8857421875,      0.13671875,
            0.564453125,     0.896484375,       0.367919921875,
            0.68115234375,   0.047943115234375, 0.33349609375,
            0.1988525390625, 0.41162109375,     0.079345703125,
            0.42724609375,   0.53564453125,     0.59130859375,
            0.2841796875,    0.414794921875,    0.0269012451171875,
            0.362060546875,  0.99462890625,     0.07183837890625,
            0.1220703125,    0.84228515625,     0.453857421875,
            0.21533203125
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.38037109375, 0.52783203125, 0.219482421875, 0.366943359375,
            0.33984375, 0.419921875, 0.380615234375, 0.1944580078125,
            0.56884765625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'dilations': [2, 2]}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [1.3603515625],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'depthwise conv2d float16 4D input and filter tensors options.groups= input_channels',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.8447265625, 0.943359375, 0.65576171875, 0.6982421875,
            0.99951171875, 0.2366943359375, 0.367431640625, 0.261962890625,
            0.62548828125, 0.84033203125, 0.37841796875, 0.454345703125,
            0.253173828125, 0.578125, 0.54150390625, 0.37841796875
          ],
          'descriptor': {shape: [1, 4, 2, 2], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.272216796875, 0.28125, 0.8544921875, 0.1796875, 0.7763671875,
            0.51416015625, 0.63720703125, 0.1280517578125, 0.83740234375,
            0.57275390625, 0.09857177734375, 0.5927734375, 0.58984375,
            0.96923828125, 0.2318115234375, 0.1480712890625
          ],
          'descriptor': {shape: [4, 1, 2, 2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'groups': 4}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [1.1806640625, 1.1650390625, 1.3115234375, 0.89111328125],
          'descriptor': {shape: [1, 4, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors options.inputLayout=\'nchw\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.96728515625, 0.10833740234375, 0.3203125,
            0.6953125, 0.50732421875, 0.0814208984375, 0.5302734375,
            0.30712890625, 0.432373046875, 0.98486328125, 0.42822265625
          ],
          'descriptor': {shape: [3, 1, 2, 2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nchw'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.884765625,  0.7412109375,  0.289794921875, 0.405517578125,
            0.962890625,  0.91064453125, 0.4833984375,   0.488037109375,
            0.8017578125, 0.62744140625, 0.448486328125, 0.87158203125,
            0.693359375,  1.037109375,   0.82861328125,  0.353271484375,
            1.1787109375, 0.8125,        0.81640625,     0.67822265625,
            0.9169921875, 1.0830078125,  1.2353515625,   0.98095703125
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors options.inputLayout=\'nhwc\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.96728515625, 0.10833740234375, 0.3203125,
            0.6953125, 0.50732421875, 0.0814208984375, 0.5302734375,
            0.30712890625, 0.432373046875, 0.98486328125, 0.42822265625
          ],
          'descriptor': {shape: [3, 1, 2, 2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nhwc'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.884765625,    0.962890625,    0.8017578125,   0.7412109375,
            0.91064453125,  0.62744140625,  0.289794921875, 0.4833984375,
            0.448486328125, 0.405517578125, 0.488037109375, 0.87158203125,
            0.693359375,    1.1787109375,   0.9169921875,   1.037109375,
            0.8125,         1.0830078125,   0.82861328125,  0.81640625,
            1.2353515625,   0.353271484375, 0.67822265625,  0.98095703125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors options.filterLayout=\'oihw\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.96728515625, 0.10833740234375, 0.3203125,
            0.6953125, 0.50732421875, 0.0814208984375, 0.5302734375,
            0.30712890625, 0.432373046875, 0.98486328125, 0.42822265625
          ],
          'descriptor': {shape: [3, 1, 2, 2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'filterLayout': 'oihw'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.884765625,  0.7412109375,  0.289794921875, 0.405517578125,
            0.962890625,  0.91064453125, 0.4833984375,   0.488037109375,
            0.8017578125, 0.62744140625, 0.448486328125, 0.87158203125,
            0.693359375,  1.037109375,   0.82861328125,  0.353271484375,
            1.1787109375, 0.8125,        0.81640625,     0.67822265625,
            0.9169921875, 1.0830078125,  1.2353515625,   0.98095703125
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors options.filterLayout=\'hwio\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.6953125, 0.30712890625, 0.96728515625,
            0.50732421875, 0.432373046875, 0.10833740234375, 0.0814208984375,
            0.98486328125, 0.3203125, 0.5302734375, 0.42822265625
          ],
          'descriptor': {shape: [2, 2, 1, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'filterLayout': 'hwio'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.884765625,  0.7412109375,  0.289794921875, 0.405517578125,
            0.962890625,  0.91064453125, 0.4833984375,   0.488037109375,
            0.8017578125, 0.62744140625, 0.448486328125, 0.87158203125,
            0.693359375,  1.037109375,   0.82861328125,  0.353271484375,
            1.1787109375, 0.8125,        0.81640625,     0.67822265625,
            0.9169921875, 1.0830078125,  1.2353515625,   0.98095703125
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors options.filterLayout=\'ohwi\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.96728515625, 0.10833740234375, 0.3203125,
            0.6953125, 0.50732421875, 0.0814208984375, 0.5302734375,
            0.30712890625, 0.432373046875, 0.98486328125, 0.42822265625
          ],
          'descriptor': {shape: [3, 2, 2, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'filterLayout': 'ohwi'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.884765625,  0.7412109375,  0.289794921875, 0.405517578125,
            0.962890625,  0.91064453125, 0.4833984375,   0.488037109375,
            0.8017578125, 0.62744140625, 0.448486328125, 0.87158203125,
            0.693359375,  1.037109375,   0.82861328125,  0.353271484375,
            1.1787109375, 0.8125,        0.81640625,     0.67822265625,
            0.9169921875, 1.0830078125,  1.2353515625,   0.98095703125
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors options.filterLayout=\'ihwo\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.6953125, 0.30712890625, 0.96728515625,
            0.50732421875, 0.432373046875, 0.10833740234375, 0.0814208984375,
            0.98486328125, 0.3203125, 0.5302734375, 0.42822265625
          ],
          'descriptor': {shape: [1, 2, 2, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'filterLayout': 'ihwo'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.884765625,  0.7412109375,  0.289794921875, 0.405517578125,
            0.962890625,  0.91064453125, 0.4833984375,   0.488037109375,
            0.8017578125, 0.62744140625, 0.448486328125, 0.87158203125,
            0.693359375,  1.037109375,   0.82861328125,  0.353271484375,
            1.1787109375, 0.8125,        0.81640625,     0.67822265625,
            0.9169921875, 1.0830078125,  1.2353515625,   0.98095703125
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors options.inputLayout=\'nhwc\' and options.filterLayout=\'oihw\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.96728515625, 0.10833740234375, 0.3203125,
            0.6953125, 0.50732421875, 0.0814208984375, 0.5302734375,
            0.30712890625, 0.432373046875, 0.98486328125, 0.42822265625
          ],
          'descriptor': {shape: [3, 1, 2, 2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'oihw'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.884765625,    0.962890625,    0.8017578125,   0.7412109375,
            0.91064453125,  0.62744140625,  0.289794921875, 0.4833984375,
            0.448486328125, 0.405517578125, 0.488037109375, 0.87158203125,
            0.693359375,    1.1787109375,   0.9169921875,   1.037109375,
            0.8125,         1.0830078125,   0.82861328125,  0.81640625,
            1.2353515625,   0.353271484375, 0.67822265625,  0.98095703125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors options.inputLayout=\'nhwc\' and options.filterLayout=\'hwio\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.6953125, 0.30712890625, 0.96728515625,
            0.50732421875, 0.432373046875, 0.10833740234375, 0.0814208984375,
            0.98486328125, 0.3203125, 0.5302734375, 0.42822265625
          ],
          'descriptor': {shape: [2, 2, 1, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'hwio'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.884765625,    0.962890625,    0.8017578125,   0.7412109375,
            0.91064453125,  0.62744140625,  0.289794921875, 0.4833984375,
            0.448486328125, 0.405517578125, 0.488037109375, 0.87158203125,
            0.693359375,    1.1787109375,   0.9169921875,   1.037109375,
            0.8125,         1.0830078125,   0.82861328125,  0.81640625,
            1.2353515625,   0.353271484375, 0.67822265625,  0.98095703125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors options.inputLayout=\'nhwc\' and options.filterLayout=\'ohwi\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.96728515625, 0.10833740234375, 0.3203125,
            0.6953125, 0.50732421875, 0.0814208984375, 0.5302734375,
            0.30712890625, 0.432373046875, 0.98486328125, 0.42822265625
          ],
          'descriptor': {shape: [3, 2, 2, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'ohwi'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.884765625,    0.962890625,    0.8017578125,   0.7412109375,
            0.91064453125,  0.62744140625,  0.289794921875, 0.4833984375,
            0.448486328125, 0.405517578125, 0.488037109375, 0.87158203125,
            0.693359375,    1.1787109375,   0.9169921875,   1.037109375,
            0.8125,         1.0830078125,   0.82861328125,  0.81640625,
            1.2353515625,   0.353271484375, 0.67822265625,  0.98095703125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors options.inputLayout=\'nhwc\' and options.filterLayout=\'ihwo\'',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.6953125, 0.30712890625, 0.96728515625,
            0.50732421875, 0.432373046875, 0.10833740234375, 0.0814208984375,
            0.98486328125, 0.3203125, 0.5302734375, 0.42822265625
          ],
          'descriptor': {shape: [1, 2, 2, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'ihwo'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.884765625,    0.962890625,    0.8017578125,   0.7412109375,
            0.91064453125,  0.62744140625,  0.289794921875, 0.4833984375,
            0.448486328125, 0.405517578125, 0.488037109375, 0.87158203125,
            0.693359375,    1.1787109375,   0.9169921875,   1.037109375,
            0.8125,         1.0830078125,   0.82861328125,  0.81640625,
            1.2353515625,   0.353271484375, 0.67822265625,  0.98095703125
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'conv2d float16 4D input and filter tensors 1D options.bias',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.7529296875, 0.751953125, 0.5947265625, 0.21630859375,
            0.07586669921875, 0.151123046875, 0.12127685546875, 0.53662109375,
            0.59375, 0.9912109375, 0.363037109375, 0.92919921875,
            0.227294921875, 0.54150390625, 0.08447265625, 0.6767578125,
            0.619140625, 0.392822265625
          ],
          'descriptor': {shape: [2, 1, 3, 3], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.1453857421875, 0.96728515625, 0.10833740234375, 0.3203125,
            0.6953125, 0.50732421875, 0.0814208984375, 0.5302734375,
            0.30712890625, 0.432373046875, 0.98486328125, 0.42822265625
          ],
          'descriptor': {shape: [3, 1, 2, 2], dataType: 'float16'},
          'constant': true
        },
        'conv2dBias': {
          'data': [0.8134765625, 0.83935546875, 0.494384765625],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'},
          {'options': {'bias': 'conv2dBias'}}
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            1.6982421875, 1.5546875,    1.103515625,   1.21875,
            1.8017578125, 1.75,         1.322265625,   1.3271484375,
            1.2958984375, 1.1220703125, 0.94287109375, 1.3662109375,
            1.5068359375, 1.8505859375, 1.6416015625,  1.1669921875,
            2.017578125,  1.65234375,   1.6552734375,  1.517578125,
            1.4111328125, 1.5771484375, 1.7294921875,  1.4755859375
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'conv2d float16 4D input and filter tensors all options',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            0.0997314453125,    0.53759765625,    0.3056640625,
            0.72216796875,      0.5068359375,     0.32373046875,
            0.8720703125,       0.2098388671875,  0.50537109375,
            0.0268707275390625, 0.5498046875,     0.0643310546875,
            0.156005859375,     0.11968994140625, 0.0361328125,
            0.414306640625,     0.24169921875,    0.67724609375,
            0.208740234375,     0.10980224609375, 0.7451171875,
            0.744140625,        0.787109375,      0.5888671875,
            0.11016845703125,   0.904296875,      0.11724853515625,
            0.85107421875,      0.424560546875,   0.0253753662109375,
            0.74755859375,      0.464599609375,   0.0304107666015625,
            0.42431640625,      0.384765625,      0.75830078125,
            0.990234375,        0.03717041015625, 0.014495849609375,
            0.826171875,        0.210693359375,   0.65673828125,
            0.258544921875,     0.480224609375,   0.97021484375,
            0.296875,           0.75244140625,    0.029632568359375,
            0.09027099609375,   0.7783203125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.638671875, 0.07763671875, 0.129150390625, 0.456298828125,
            0.404296875, 0.59423828125, 0.1424560546875, 0.90380859375
          ],
          'descriptor': {shape: [2, 2, 1, 2], dataType: 'float16'},
          'constant': true
        },
        'conv2dBias': {
          'data': [0.54248046875, 0.8408203125],
          'descriptor': {shape: [2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'}, {
            'options': {
              'padding': [1, 0, 0, 1],
              'strides': [1, 1],
              'dilations': [1, 1],
              'groups': 2,
              'inputLayout': 'nchw',
              'filterLayout': 'hwio',
              'bias': 'conv2dBias'
            }
          }
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            0.6591796875,  0.80322265625, 0.76904296875, 0.90673828125,
            0.74755859375, 0.9306640625,  1.3076171875,  0.98779296875,
            1.27734375,    0.876953125,   1.09375,       1.1748046875,
            0.82177734375, 0.92236328125, 0.57421875,    1.103515625,
            0.7978515625,  0.9609375,     0.7236328125,  0.60986328125,
            1.2451171875,  1.197265625,   1.404296875,   0.94384765625,
            0.6572265625,  1.484375,      1.6796875,     1.73046875,
            1.1162109375,  0.85595703125, 1.8291015625,  1.5419921875,
            1.501953125,   1.4853515625,  1.0712890625,  2.45703125,
            1.5126953125,  1.072265625,   1.8046875,     1.361328125,
            2.0703125,     1.55859375,    1.4384765625,  2.380859375,
            1.4814453125,  2.013671875,   1.4833984375,  1.1796875,
            2.078125,      1.37890625
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'conv2d float16 4D input and filter tensors, both negative input tensor and options.bias',
    'graph': {
      'inputs': {
        'conv2dInput': {
          'data': [
            -0.80712890625,  -0.8837890625,     -0.77001953125,
            -0.564453125,    -0.397216796875,   -0.1083984375,
            -0.5517578125,   -0.3955078125,     -0.057586669921875,
            -0.5146484375,   -0.2132568359375,  -0.95068359375,
            -0.80419921875,  -0.86474609375,    -0.9609375,
            -0.326416015625, -0.06884765625,    -0.3203125,
            -0.269287109375, -0.343017578125,   -0.89892578125,
            -0.90380859375,  -0.63671875,       -0.20068359375,
            -0.787109375,    -0.3466796875,     -0.060028076171875,
            -0.14990234375,  -0.6484375,        -0.8935546875,
            -0.81494140625,  -0.642578125,      -0.032745361328125,
            -0.6611328125,   -0.58447265625,    -0.09918212890625,
            -0.166015625,    -0.95068359375,    -0.30517578125,
            -0.62109375,     -0.5400390625,     -0.420166015625,
            -0.188232421875, -0.35888671875,    -0.71142578125,
            -0.375,          -0.7109375,        -0.360595703125,
            -0.546875,       -0.032257080078125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        },
        'conv2dFilter': {
          'data': [
            0.638671875, 0.07763671875, 0.129150390625, 0.456298828125,
            0.404296875, 0.59423828125, 0.1424560546875, 0.90380859375
          ],
          'descriptor': {shape: [2, 2, 1, 2], dataType: 'float16'},
          'constant': true
        },
        'conv2dBias': {
          'data': [-0.375, -0.436279296875],
          'descriptor': {shape: [2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'conv2d',
        'arguments': [
          {'input': 'conv2dInput'}, {'filter': 'conv2dFilter'}, {
            'options': {
              'padding': [1, 0, 0, 1],
              'groups': 2,
              'filterLayout': 'hwio',
              'bias': 'conv2dBias'
            }
          }
        ],
        'outputs': 'conv2dOutput'
      }],
      'expectedOutputs': {
        'conv2dOutput': {
          'data': [
            -0.8271484375,  -0.841796875,   -0.7666015625,  -0.65966796875,
            -0.53564453125, -1.126953125,   -1.318359375,   -1.107421875,
            -0.88330078125, -0.8369140625,  -0.7373046875,  -1.27734375,
            -1.0830078125,  -0.96484375,    -1.091796875,   -0.77587890625,
            -1.1591796875,  -1.16796875,    -1.208984375,   -1.126953125,
            -1.0849609375,  -0.91650390625, -0.900390625,   -0.78466796875,
            -0.912109375,   -0.69677734375, -0.607421875,   -1.111328125,
            -1.62890625,    -0.96728515625, -1.5556640625,  -0.9208984375,
            -1.3603515625,  -1.8154296875,  -0.85302734375, -1.001953125,
            -1.458984375,   -1.5810546875,  -1.4970703125,  -0.8505859375,
            -1.220703125,   -1.302734375,   -1.0859375,     -1.599609375,
            -0.9072265625,  -1.53515625,    -1.302734375,   -1.3232421875,
            -1.142578125,   -0.5107421875
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(
    conv2dTests, buildAndExecuteGraph, getPrecisionTolerance);
