// META: title=test WebNN API batchNormalization operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-batchnorm
// Normalize the values of the input tensor using Batch-Normalization.
//
// dictionary MLBatchNormalizationOptions {
//   MLOperand scale;
//   MLOperand bias;
//   [EnforceRange] unsigned long axis = 1;
//   double epsilon = 1e-5;
// };
//
// MLOperand batchNormalization(
//     MLOperand input, MLOperand mean, MLOperand, variance,
//     optional MLBatchNormalizationOptions options = {});

const batchNormTests = [
  {
    'name': 'batchNormalization float32 1D tensor options.axis=0',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406, -63.376670837402344,
            -46.790367126464844, 83.02227020263672, -80.08049011230469
          ],
          'descriptor': {shape: [6], dataType: 'float32'}
        },
        'bnMean': {
          'data': [
            -7.814267635345459, -95.64129638671875, 38.15440368652344,
            -55.95203399658203, -87.86500549316406, -41.63645553588867
          ],
          'descriptor': {shape: [6], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [
            60.31186294555664,  26.43260383605957, 53.275634765625,
            40.146121978759766, 59.41098403930664, 35.99981689453125
          ],
          'descriptor': {shape: [6], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'axis': 0}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -4.312741756439209, 31.068212509155273, -13.910240173339844,
            1.4459478855133057, 22.170541763305664, -6.407354354858398
          ],
          'descriptor': {shape: [6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'batchNormalization float32 2D tensor (mean and variance are non-constant) default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406,    -63.376670837402344,
            -46.790367126464844, 83.02227020263672,    -80.08049011230469,
            -62.144378662109375, -0.10012771934270859, -40.90216064453125,
            56.96306228637695,   37.37249755859375,    57.046478271484375,
            82.05680084228516,   -86.1164321899414,    76.8831787109375,
            97.03362274169922,   -21.35103988647461,   -96.93824005126953,
            -9.359310150146484,  80.20824432373047,    -85.36802673339844,
            62.35185241699219,   -68.4724349975586,    -12.10716724395752
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        },
        'bnMean': {
          'data': [
            -7.814267635345459, -95.64129638671875, 38.15440368652344,
            -55.95203399658203, -87.86500549316406, -41.63645553588867
          ],
          'descriptor': {shape: [6], dataType: 'float32'}
        },
        'bnVariance': {
          'data': [
            60.31186294555664, 26.43260383605957, 53.275634765625,
            40.146121978759766, 59.41098403930664, 35.99981689453125
          ],
          'descriptor': {shape: [6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -4.312741756439209,  31.068212509155273, -13.910240173339844,
            1.4459478855133057,  22.170541763305664, -6.407354354858398,
            -6.995829105377197,  18.583200454711914, -10.831125259399414,
            17.820920944213867,  16.2480411529541,   16.447195053100586,
            11.57226848602295,   1.8526301383972168, 5.306026458740234,
            24.145092010498047,  8.629376411437988,  -9.216986656188965,
            -0.1989477425813675, 34.203548431396484, -16.923160552978516,
            18.671411514282227,  2.5159497261047363, 4.921559810638428
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 2D constant tensors default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406,    -63.376670837402344,
            -46.790367126464844, 83.02227020263672,    -80.08049011230469,
            -62.144378662109375, -0.10012771934270859, -40.90216064453125,
            56.96306228637695,   37.37249755859375,    57.046478271484375,
            82.05680084228516,   -86.1164321899414,    76.8831787109375,
            97.03362274169922,   -21.35103988647461,   -96.93824005126953,
            -9.359310150146484,  80.20824432373047,    -85.36802673339844,
            62.35185241699219,   -68.4724349975586,    -12.10716724395752
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'},
          'constant': true
        },
        'bnMean': {
          'data': [
            -7.814267635345459, -95.64129638671875, 38.15440368652344,
            -55.95203399658203, -87.86500549316406, -41.63645553588867
          ],
          'descriptor': {shape: [6], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [
            60.31186294555664, 26.43260383605957, 53.275634765625,
            40.146121978759766, 59.41098403930664, 35.99981689453125
          ],
          'descriptor': {shape: [6], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -4.312741756439209,  31.068212509155273, -13.910240173339844,
            1.4459478855133057,  22.170541763305664, -6.407354354858398,
            -6.995829105377197,  18.583200454711914, -10.831125259399414,
            17.820920944213867,  16.2480411529541,   16.447195053100586,
            11.57226848602295,   1.8526301383972168, 5.306026458740234,
            24.145092010498047,  8.629376411437988,  -9.216986656188965,
            -0.1989477425813675, 34.203548431396484, -16.923160552978516,
            18.671411514282227,  2.5159497261047363, 4.921559810638428
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 2D tensor default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406,    -63.376670837402344,
            -46.790367126464844, 83.02227020263672,    -80.08049011230469,
            -62.144378662109375, -0.10012771934270859, -40.90216064453125,
            56.96306228637695,   37.37249755859375,    57.046478271484375,
            82.05680084228516,   -86.1164321899414,    76.8831787109375,
            97.03362274169922,   -21.35103988647461,   -96.93824005126953,
            -9.359310150146484,  80.20824432373047,    -85.36802673339844,
            62.35185241699219,   -68.4724349975586,    -12.10716724395752
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        },
        'bnMean': {
          'data': [
            -7.814267635345459, -95.64129638671875, 38.15440368652344,
            -55.95203399658203, -87.86500549316406, -41.63645553588867
          ],
          'descriptor': {shape: [6], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [
            60.31186294555664, 26.43260383605957, 53.275634765625,
            40.146121978759766, 59.41098403930664, 35.99981689453125
          ],
          'descriptor': {shape: [6], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -4.312741756439209,  31.068212509155273, -13.910240173339844,
            1.4459478855133057,  22.170541763305664, -6.407354354858398,
            -6.995829105377197,  18.583200454711914, -10.831125259399414,
            17.820920944213867,  16.2480411529541,   16.447195053100586,
            11.57226848602295,   1.8526301383972168, 5.306026458740234,
            24.145092010498047,  8.629376411437988,  -9.216986656188965,
            -0.1989477425813675, 34.203548431396484, -16.923160552978516,
            18.671411514282227,  2.5159497261047363, 4.921559810638428
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 3D tensor default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406,    -63.376670837402344,
            -46.790367126464844, 83.02227020263672,    -80.08049011230469,
            -62.144378662109375, -0.10012771934270859, -40.90216064453125,
            56.96306228637695,   37.37249755859375,    57.046478271484375,
            82.05680084228516,   -86.1164321899414,    76.8831787109375,
            97.03362274169922,   -21.35103988647461,   -96.93824005126953,
            -9.359310150146484,  80.20824432373047,    -85.36802673339844,
            62.35185241699219,   -68.4724349975586,    -12.10716724395752
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        },
        'bnMean': {
          'data': [12.810380935668945, 63.13715362548828, -61.62983322143555],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [18.358240127563477, 41.847232818603516, 16.12828254699707],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -12.630594253540039, 11.967890739440918,  -17.781383514404297,
            -13.910285949707031, 3.0739352703094482,  -22.139259338378906,
            -19.36661148071289,  -9.775517463684082,  5.161267280578613,
            29.53006935119629,   24.651947021484375,  29.550840377807617,
            16.161500930786133,  -23.088642120361328, 14.954023361206055,
            19.656957626342773,  -13.06058406829834,  -24.745210647583008,
            -11.206846237182617, 2.638929843902588,   -5.910898208618164,
            30.871898651123047,  -1.7038332223892212, 12.331327438354492
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 4D tensor default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406,    -63.376670837402344,
            -46.790367126464844, 83.02227020263672,    -80.08049011230469,
            -62.144378662109375, -0.10012771934270859, -40.90216064453125,
            56.96306228637695,   37.37249755859375,    57.046478271484375,
            82.05680084228516,   -86.1164321899414,    76.8831787109375,
            97.03362274169922,   -21.35103988647461,   -96.93824005126953,
            -9.359310150146484,  80.20824432373047,    -85.36802673339844,
            62.35185241699219,   -68.4724349975586,    -12.10716724395752
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        },
        'bnMean': {
          'data': [51.629150390625, 99.36075592041016, -96.1473617553711],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.448015213012695, 86.36219024658203, 73.88455200195312],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -16.842504501342773, 2.2579827308654785,  -20.842041015625,
            -17.836172103881836, -1.7581257820129395, -19.30902862548828,
            -17.37898826599121,  -10.702629089355469, 6.4271392822265625,
            17.812623977661133,  15.533489227294922,  17.822328567504883,
            5.514280319213867,   -24.963077545166016, 4.576685905456543,
            8.228469848632812,   -12.989363670349121, -21.123029708862305,
            -11.698976516723633, -2.0609331130981445, 1.2540507316589355,
            18.43954849243164,   3.2196571826934814,  9.777103424072266
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 5D tensor default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406,    -63.376670837402344,
            -46.790367126464844, 83.02227020263672,    -80.08049011230469,
            -62.144378662109375, -0.10012771934270859, -40.90216064453125,
            56.96306228637695,   37.37249755859375,    57.046478271484375,
            82.05680084228516,   -86.1164321899414,    76.8831787109375,
            97.03362274169922,   -21.35103988647461,   -96.93824005126953,
            -9.359310150146484,  80.20824432373047,    -85.36802673339844,
            62.35185241699219,   -68.4724349975586,    -12.10716724395752
          ],
          'descriptor': {shape: [6, 1, 1, 2, 2], dataType: 'float32'}
        },
        'bnMean': {
          'data': [35.4078254699707],
          'descriptor': {shape: [1], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [40.93109893798828],
          'descriptor': {shape: [1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -11.990972518920898, 4.4829583168029785,  -15.440524101257324,
            -12.847999572753906, 7.442382335662842,   -18.051416397094727,
            -15.247910499572754, -5.550075531005859,  -11.927642822265625,
            3.369194269180298,   0.30708834528923035, 3.382232427597046,
            7.291474342346191,   -18.99486541748047,  6.4828104972839355,
            9.632428169250488,   -8.871702194213867,  -20.686368942260742,
            -6.99733304977417,   7.002535343170166,   -18.877885818481445,
            4.211489677429199,   -16.237018585205078, -7.42683744430542
          ],
          'descriptor': {shape: [6, 1, 1, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 4D NCHW tensor options.axis=1',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406,    -63.376670837402344,
            -46.790367126464844, 83.02227020263672,    -80.08049011230469,
            -62.144378662109375, -0.10012771934270859, -40.90216064453125,
            56.96306228637695,   37.37249755859375,    57.046478271484375,
            82.05680084228516,   -86.1164321899414,    76.8831787109375,
            97.03362274169922,   -21.35103988647461,   -96.93824005126953,
            -9.359310150146484,  80.20824432373047,    -85.36802673339844,
            62.35185241699219,   -68.4724349975586,    -12.10716724395752
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        },
        'bnMean': {
          'data': [51.629150390625, 99.36075592041016, -96.1473617553711],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.448015213012695, 86.36219024658203, 73.88455200195312],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'axis': 1}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -16.842504501342773, 2.2579827308654785,  -20.842041015625,
            -17.836172103881836, -1.7581257820129395, -19.30902862548828,
            -17.37898826599121,  -10.702629089355469, 6.4271392822265625,
            17.812623977661133,  15.533489227294922,  17.822328567504883,
            5.514280319213867,   -24.963077545166016, 4.576685905456543,
            8.228469848632812,   -12.989363670349121, -21.123029708862305,
            -11.698976516723633, -2.0609331130981445, 1.2540507316589355,
            18.43954849243164,   3.2196571826934814,  9.777103424072266
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 4D NHWC tensor options.axis=3',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  83.02227020263672,    -40.90216064453125,
            64.08863830566406,   -80.08049011230469,   56.96306228637695,
            -63.376670837402344, -62.144378662109375,  37.37249755859375,
            -46.790367126464844, -0.10012771934270859, 57.046478271484375,
            82.05680084228516,   -21.35103988647461,   -85.36802673339844,
            -86.1164321899414,   -96.93824005126953,   62.35185241699219,
            76.8831787109375,    -9.359310150146484,   -68.4724349975586,
            97.03362274169922,   80.20824432373047,    -12.10716724395752
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'bnMean': {
          'data': [51.629150390625, 99.36075592041016, -96.1473617553711],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.448015213012695, 86.36219024658203, 73.88455200195312],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'axis': 3}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -16.842504501342773, -1.7581257820129395, 6.4271392822265625,
            2.2579827308654785,  -19.30902862548828,  17.812623977661133,
            -20.842041015625,    -17.37898826599121,  15.533489227294922,
            -17.836172103881836, -10.702629089355469, 17.822328567504883,
            5.514280319213867,   -12.989363670349121, 1.2540507316589355,
            -24.963077545166016, -21.123029708862305, 18.43954849243164,
            4.576685905456543,   -11.698976516723633, 3.2196571826934814,
            8.228469848632812,   -2.0609331130981445, 9.777103424072266
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 4D NCHW tensor options.scale',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406,    -63.376670837402344,
            -46.790367126464844, 83.02227020263672,    -80.08049011230469,
            -62.144378662109375, -0.10012771934270859, -40.90216064453125,
            56.96306228637695,   37.37249755859375,    57.046478271484375,
            82.05680084228516,   -86.1164321899414,    76.8831787109375,
            97.03362274169922,   -21.35103988647461,   -96.93824005126953,
            -9.359310150146484,  80.20824432373047,    -85.36802673339844,
            62.35185241699219,   -68.4724349975586,    -12.10716724395752
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        },
        'bnMean': {
          'data': [51.629150390625, 99.36075592041016, -96.1473617553711],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.448015213012695, 86.36219024658203, 73.88455200195312],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnScale': {
          'data': [65.50171661376953, -71.007568359375, -5.569730758666992],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'scale': 'bnScale'}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -1103.212890625,     147.90174865722656,  -1365.189453125,
            -1168.2999267578125, 124.84024047851562,  1371.087158203125,
            1234.0396728515625,  759.9676513671875,   -35.79743576049805,
            -99.2115249633789,   -86.51734924316406,  -99.26557159423828,
            361.19482421875,     -1635.1243896484375, 299.78076171875,
            538.9788818359375,   922.3430786132812,   1499.89501953125,
            830.7158813476562,   146.3418426513672,   -6.984724998474121,
            -102.70331573486328, -17.9326229095459,   -54.455833435058594
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 4D NCHW tensor options.bias',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406,    -63.376670837402344,
            -46.790367126464844, 83.02227020263672,    -80.08049011230469,
            -62.144378662109375, -0.10012771934270859, -40.90216064453125,
            56.96306228637695,   37.37249755859375,    57.046478271484375,
            82.05680084228516,   -86.1164321899414,    76.8831787109375,
            97.03362274169922,   -21.35103988647461,   -96.93824005126953,
            -9.359310150146484,  80.20824432373047,    -85.36802673339844,
            62.35185241699219,   -68.4724349975586,    -12.10716724395752
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        },
        'bnMean': {
          'data': [51.629150390625, 99.36075592041016, -96.1473617553711],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.448015213012695, 86.36219024658203, 73.88455200195312],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnBias': {
          'data': [64.2044677734375, 75.28591918945312, -84.57243347167969],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'bias': 'bnBias'}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            47.36196517944336,  66.46244812011719,  43.3624267578125,
            46.36829376220703,  73.52779388427734,  55.976890563964844,
            57.90693283081055,  64.58329010009766,  -78.14529418945312,
            -66.75981140136719, -69.03894805908203, -66.75010681152344,
            69.71875,           39.241390228271484, 68.7811508178711,
            72.43293762207031,  62.29655456542969,  54.16288757324219,
            63.586944580078125, 73.22498321533203,  -83.3183822631836,
            -66.13288879394531, -81.35277557373047, -74.79533386230469
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 4D NCHW tensor options.epsilon',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  64.08863830566406,    -63.376670837402344,
            -46.790367126464844, 83.02227020263672,    -80.08049011230469,
            -62.144378662109375, -0.10012771934270859, -40.90216064453125,
            56.96306228637695,   37.37249755859375,    57.046478271484375,
            82.05680084228516,   -86.1164321899414,    76.8831787109375,
            97.03362274169922,   -21.35103988647461,   -96.93824005126953,
            -9.359310150146484,  80.20824432373047,    -85.36802673339844,
            62.35185241699219,   -68.4724349975586,    -12.10716724395752
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        },
        'bnMean': {
          'data': [51.629150390625, 99.36075592041016, -96.1473617553711],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.448015213012695, 86.36219024658203, 73.88455200195312],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'epsilon': 0.000001}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -16.842506408691406, 2.2579832077026367,  -20.842044830322266,
            -17.8361759185791,   -1.758125901222229,  -19.309030532836914,
            -17.37898826599121,  -10.702629089355469, 6.427139759063721,
            17.812625885009766,  15.533490180969238,  17.822330474853516,
            5.514281272888184,   -24.96308135986328,  4.576686382293701,
            8.228470802307129,   -12.989363670349121, -21.123031616210938,
            -11.698976516723633, -2.0609331130981445, 1.254050850868225,
            18.43954849243164,   3.2196574211120605,  9.777103424072266
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float32 4D NHWC tensor all options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.30733108520508,  83.02227020263672,    -40.90216064453125,
            64.08863830566406,   -80.08049011230469,   56.96306228637695,
            -63.376670837402344, -62.144378662109375,  37.37249755859375,
            -46.790367126464844, -0.10012771934270859, 57.046478271484375,
            82.05680084228516,   -21.35103988647461,   -85.36802673339844,
            -86.1164321899414,   -96.93824005126953,   62.35185241699219,
            76.8831787109375,    -9.359310150146484,   -68.4724349975586,
            97.03362274169922,   80.20824432373047,    -12.10716724395752
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        },
        'bnMean': {
          'data': [51.629150390625, 99.36075592041016, -96.1473617553711],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.448015213012695, 86.36219024658203, 73.88455200195312],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnScale': {
          'data': [65.50171661376953, -71.007568359375, -5.569730758666992],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        },
        'bnBias': {
          'data': [64.2044677734375, 75.28591918945312, -84.57243347167969],
          'descriptor': {shape: [3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {
            'options': {
              'scale': 'bnScale',
              'bias': 'bnBias',
              'axis': 3,
              'epsilon': 0.000001
            }
          }
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -1039.0085734071204, 200.12613597546277, -120.36987167541395,
            212.10626540432202,  1446.3732126569944, -183.78396479879416,
            -1300.9852072279227, 1309.3257094058545, -171.08979404258523,
            -1104.0956031373803, 835.2536189871761,  -183.83801576309426,
            425.3993215144054,   997.6290832897452,  -91.55716013805052,
            -1570.920072497096,  1575.1810627320297, -187.2757593197739,
            363.98524710447384,  906.0018322105,     -102.5050592863526,
            603.1834043179756,   221.6277675074517,  -139.02827100419768
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'batchNormalization float16 1D tensor options.axis=0',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [-41.3125, 64.0625, -63.375, -46.78125, 83, -80.0625],
          'descriptor': {shape: [6], dataType: 'float16'}
        },
        'bnMean': {
          'data': [-7.8125, -95.625, 38.15625, -55.9375, -87.875, -41.625],
          'descriptor': {shape: [6], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [60.3125, 26.4375, 53.28125, 40.15625, 59.40625, 36],
          'descriptor': {shape: [6], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'axis': 0}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [-4.3125, 31.0625, -13.90625, 1.4453125, 22.171875, -6.40625],
          'descriptor': {shape: [6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'batchNormalization float16 2D tensor (mean and variance are non-constant) default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125, 64.0625,   -63.375,        -46.78125, 83,
            -80.0625, -62.15625, -0.10009765625, -40.90625, 56.96875,
            37.375,   57.03125,  82.0625,        -86.125,   76.875,
            97.0625,  -21.34375, -96.9375,       -9.359375, 80.1875,
            -85.375,  62.34375,  -68.5,          -12.109375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        },
        'bnMean': {
          'data': [-7.8125, -95.625, 38.15625, -55.9375, -87.875, -41.625],
          'descriptor': {shape: [6], dataType: 'float16'}
        },
        'bnVariance': {
          'data': [60.3125, 26.4375, 53.28125, 40.15625, 59.40625, 36],
          'descriptor': {shape: [6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -4.3125,    31.0625,     -13.90625,   1.4453125,   22.171875,
            -6.40625,   -6.99609375, 18.578125,   -10.828125,  17.8125,
            16.25,      16.4375,     11.5703125,  1.84765625,  5.3046875,
            24.140625,  8.6328125,   -9.21875,    -0.19921875, 34.1875,
            -16.921875, 18.671875,   2.513671875, 4.91796875
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 2D constant tensors default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125, 64.0625,   -63.375,        -46.78125, 83,
            -80.0625, -62.15625, -0.10009765625, -40.90625, 56.96875,
            37.375,   57.03125,  82.0625,        -86.125,   76.875,
            97.0625,  -21.34375, -96.9375,       -9.359375, 80.1875,
            -85.375,  62.34375,  -68.5,          -12.109375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'},
          'constant': true
        },
        'bnMean': {
          'data': [-7.8125, -95.625, 38.15625, -55.9375, -87.875, -41.625],
          'descriptor': {shape: [6], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [60.3125, 26.4375, 53.28125, 40.15625, 59.40625, 36],
          'descriptor': {shape: [6], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -4.3125,    31.0625,     -13.90625,   1.4453125,   22.171875,
            -6.40625,   -6.99609375, 18.578125,   -10.828125,  17.8125,
            16.25,      16.4375,     11.5703125,  1.84765625,  5.3046875,
            24.140625,  8.6328125,   -9.21875,    -0.19921875, 34.1875,
            -16.921875, 18.671875,   2.513671875, 4.91796875
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 2D tensor default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125, 64.0625,   -63.375,        -46.78125, 83,
            -80.0625, -62.15625, -0.10009765625, -40.90625, 56.96875,
            37.375,   57.03125,  82.0625,        -86.125,   76.875,
            97.0625,  -21.34375, -96.9375,       -9.359375, 80.1875,
            -85.375,  62.34375,  -68.5,          -12.109375
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        },
        'bnMean': {
          'data': [-7.8125, -95.625, 38.15625, -55.9375, -87.875, -41.625],
          'descriptor': {shape: [6], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [60.3125, 26.4375, 53.28125, 40.15625, 59.40625, 36],
          'descriptor': {shape: [6], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -4.3125,    31.0625,     -13.90625,   1.4453125,   22.171875,
            -6.40625,   -6.99609375, 18.578125,   -10.828125,  17.8125,
            16.25,      16.4375,     11.5703125,  1.84765625,  5.3046875,
            24.140625,  8.6328125,   -9.21875,    -0.19921875, 34.1875,
            -16.921875, 18.671875,   2.513671875, 4.91796875
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 3D tensor default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125, 64.0625,   -63.375,        -46.78125, 83,
            -80.0625, -62.15625, -0.10009765625, -40.90625, 56.96875,
            37.375,   57.03125,  82.0625,        -86.125,   76.875,
            97.0625,  -21.34375, -96.9375,       -9.359375, 80.1875,
            -85.375,  62.34375,  -68.5,          -12.109375
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        },
        'bnMean': {
          'data': [12.8125, 63.125, -61.625],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [18.359375, 41.84375, 16.125],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -12.6328125, 11.9609375,  -17.78125,     -13.90625,  3.072265625,
            -22.140625,  -19.375,     -9.7734375,    5.16015625, 29.53125,
            24.65625,    29.546875,   16.15625,      -23.09375,  14.953125,
            19.65625,    -13.0546875, -24.75,        -11.203125, 2.638671875,
            -5.9140625,  30.875,      -1.7119140625, 12.328125
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 4D tensor default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125, 64.0625,   -63.375,        -46.78125, 83,
            -80.0625, -62.15625, -0.10009765625, -40.90625, 56.96875,
            37.375,   57.03125,  82.0625,        -86.125,   76.875,
            97.0625,  -21.34375, -96.9375,       -9.359375, 80.1875,
            -85.375,  62.34375,  -68.5,          -12.109375
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        },
        'bnMean': {
          'data': [51.625, 99.375, -96.125],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.453125, 86.375, 73.875],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -16.84375,    2.25390625,  -20.84375,  -17.828125, -1.76171875,
            -19.3125,     -17.375,     -10.703125, 6.42578125, 17.8125,
            15.53125,     17.8125,     5.515625,   -24.96875,  4.57421875,
            8.234375,     -12.9921875, -21.125,    -11.703125, -2.064453125,
            1.2509765625, 18.4375,     3.21484375, 9.7734375
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 5D tensor default options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125, 64.0625,   -63.375,        -46.78125, 83,
            -80.0625, -62.15625, -0.10009765625, -40.90625, 56.96875,
            37.375,   57.03125,  82.0625,        -86.125,   76.875,
            97.0625,  -21.34375, -96.9375,       -9.359375, 80.1875,
            -85.375,  62.34375,  -68.5,          -12.109375
          ],
          'descriptor': {shape: [6, 1, 1, 2, 2], dataType: 'float16'}
        },
        'bnMean': {
          'data': [35.40625],
          'descriptor': {shape: [1], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [40.9375],
          'descriptor': {shape: [1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -11.9921875,  4.48046875, -15.4375,    -12.84375,   7.4375,
            -18.046875,   -15.25,     -5.55078125, -11.9296875, 3.369140625,
            0.3076171875, 3.37890625, 7.29296875,  -19,         6.48046875,
            9.6328125,    -8.8671875, -20.6875,    -6.99609375, 7,
            -18.875,      4.2109375,  -16.234375,  -7.42578125
          ],
          'descriptor': {shape: [6, 1, 1, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 4D NCHW tensor options.axis=1',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125, 64.0625,   -63.375,        -46.78125, 83,
            -80.0625, -62.15625, -0.10009765625, -40.90625, 56.96875,
            37.375,   57.03125,  82.0625,        -86.125,   76.875,
            97.0625,  -21.34375, -96.9375,       -9.359375, 80.1875,
            -85.375,  62.34375,  -68.5,          -12.109375
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        },
        'bnMean': {
          'data': [51.625, 99.375, -96.125],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.453125, 86.375, 73.875],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'axis': 1}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -16.84375,    2.25390625,  -20.84375,  -17.828125, -1.76171875,
            -19.3125,     -17.375,     -10.703125, 6.42578125, 17.8125,
            15.53125,     17.8125,     5.515625,   -24.96875,  4.57421875,
            8.234375,     -12.9921875, -21.125,    -11.703125, -2.064453125,
            1.2509765625, 18.4375,     3.21484375, 9.7734375
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 4D NHWC tensor options.axis=3',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125,       83,       -40.90625, 64.0625,   -80.0625,
            56.96875,       -63.375,  -62.15625, 37.375,    -46.78125,
            -0.10009765625, 57.03125, 82.0625,   -21.34375, -85.375,
            -86.125,        -96.9375, 62.34375,  76.875,    -9.359375,
            -68.5,          97.0625,  80.1875,   -12.109375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'bnMean': {
          'data': [51.625, 99.375, -96.125],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.453125, 86.375, 73.875],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'axis': 3}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -16.84375,  -1.76171875, 6.42578125,   2.25390625,  -19.3125,
            17.8125,    -20.84375,   -17.375,      15.53125,    -17.828125,
            -10.703125, 17.8125,     5.515625,     -12.9921875, 1.2509765625,
            -24.96875,  -21.125,     18.4375,      4.57421875,  -11.703125,
            3.21484375, 8.234375,    -2.064453125, 9.7734375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 4D NCHW tensor options.scale',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125, 64.0625,   -63.375,        -46.78125, 83,
            -80.0625, -62.15625, -0.10009765625, -40.90625, 56.96875,
            37.375,   57.03125,  82.0625,        -86.125,   76.875,
            97.0625,  -21.34375, -96.9375,       -9.359375, 80.1875,
            -85.375,  62.34375,  -68.5,          -12.109375
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        },
        'bnMean': {
          'data': [51.625, 99.375, -96.125],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.453125, 86.375, 73.875],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnScale': {
          'data': [65.5, -71, -5.5703125],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'scale': 'bnScale'}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -1103,  147.625, -1365,     -1168,     125.125,   1371,
            1234,   760,     -35.78125, -99.1875,  -86.5,     -99.25,
            361.25, -1635,   299.75,    539.5,     922,       1500,
            830.5,  146.625, -6.96875,  -102.6875, -17.90625, -54.4375
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 4D NCHW tensor options.bias',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125, 64.0625,   -63.375,        -46.78125, 83,
            -80.0625, -62.15625, -0.10009765625, -40.90625, 56.96875,
            37.375,   57.03125,  82.0625,        -86.125,   76.875,
            97.0625,  -21.34375, -96.9375,       -9.359375, 80.1875,
            -85.375,  62.34375,  -68.5,          -12.109375
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        },
        'bnMean': {
          'data': [51.625, 99.375, -96.125],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.453125, 86.375, 73.875],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnBias': {
          'data': [64.1875, 75.3125, -84.5625],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'bias': 'bnBias'}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            47.34375, 66.4375,  43.34375, 46.34375, 73.5625, 56,
            57.9375,  64.625,   -78.125,  -66.75,   -69,     -66.75,
            69.6875,  39.21875, 68.75,    72.4375,  62.3125, 54.1875,
            63.625,   73.25,    -83.3125, -66.125,  -81.375, -74.8125
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 4D NCHW tensor options.epsilon',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125, 64.0625,   -63.375,        -46.78125, 83,
            -80.0625, -62.15625, -0.10009765625, -40.90625, 56.96875,
            37.375,   57.03125,  82.0625,        -86.125,   76.875,
            97.0625,  -21.34375, -96.9375,       -9.359375, 80.1875,
            -85.375,  62.34375,  -68.5,          -12.109375
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        },
        'bnMean': {
          'data': [51.625, 99.375, -96.125],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.453125, 86.375, 73.875],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {'options': {'epsilon': 0.000001}}
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -16.84375,    2.25390625,  -20.84375,  -17.828125, -1.76171875,
            -19.3125,     -17.375,     -10.703125, 6.42578125, 17.8125,
            15.53125,     17.8125,     5.515625,   -24.96875,  4.57421875,
            8.234375,     -12.9921875, -21.125,    -11.703125, -2.064453125,
            1.2509765625, 18.4375,     3.21484375, 9.7734375
          ],
          'descriptor': {shape: [2, 3, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'batchNormalization float16 4D NHWC tensor all options',
    'graph': {
      'inputs': {
        'bnInput': {
          'data': [
            -41.3125,       83,       -40.90625, 64.0625,   -80.0625,
            56.96875,       -63.375,  -62.15625, 37.375,    -46.78125,
            -0.10009765625, 57.03125, 82.0625,   -21.34375, -85.375,
            -86.125,        -96.9375, 62.34375,  76.875,    -9.359375,
            -68.5,          97.0625,  80.1875,   -12.109375
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        },
        'bnMean': {
          'data': [51.625, 99.375, -96.125],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnVariance': {
          'data': [30.453125, 86.375, 73.875],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnScale': {
          'data': [65.5, -71, -5.5703125],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        },
        'bnBias': {
          'data': [64.1875, 75.3125, -84.5625],
          'descriptor': {shape: [3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'batchNormalization',
        'arguments': [
          {'input': 'bnInput'}, {'mean': 'bnMean'}, {'variance': 'bnVariance'},
          {
            'options': {
              'scale': 'bnScale',
              'bias': 'bnBias',
              'axis': 3,
              'epsilon': 0.000001
            }
          }
        ],
        'outputs': 'bnOutput'
      }],
      'expectedOutputs': {
        'bnOutput': {
          'data': [
            -1039, 200.375, -120.375,  211.75, 1446,    -183.75,
            -1301, 1309,    -171.125,  -1104,  835.5,   -183.875,
            425.5, 997.5,   -91.5,     -1571,  1575,    -187.25,
            364,   906,     -102.4375, 603.5,  221.875, -139
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(
    batchNormTests, buildAndExecuteGraph, getPrecisionTolerance);
