// META: title=test WebNN API averagePool2d operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-pool2d
// Compute a pooling operation across all the elements within the moving window
// over the input tensor.
//
// enum MLRoundingType {
//   "floor",
//   "ceil"
// };
//
// dictionary MLPool2dOptions {
//   sequence<[EnforceRange] unsigned long> windowDimensions;
//   sequence<[EnforceRange] unsigned long> padding;
//   sequence<[EnforceRange] unsigned long> strides;
//   sequence<[EnforceRange] unsigned long> dilations;
//   MLInputOperandLayout layout = "nchw";
//   MLRoundingType roundingType = "floor";
//   sequence<[EnforceRange] unsigned long> outputSizes;
// };
//
// MLOperand averagePool2d(
//     MLOperand input, optional MLPool2dOptions options = {});

const averagePool2dTests = [
  // float32 tests
  {
    'name':
        'averagePool2d float32 4D constant tensor all positive default options',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [{'input': 'averagePool2dInput'}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor all positive default options',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [{'input': 'averagePool2dInput'}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor all negative default options',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            -83.87757873535156,  -2.0740277767181396, -7.561108589172363,
            -45.274261474609375, -16.36655616760254,  -44.908512115478516,
            -42.04186248779297,  -44.77231979370117,  -1.5066279172897339,
            -52.65203857421875,  -92.01856231689453,  -48.004093170166016,
            -61.522972106933594, -93.44403839111328,  -25.780330657958984,
            -95.51873779296875,  -10.963757514953613, -59.132747650146484,
            -32.60173797607422,  -21.4510440826416,   -87.115966796875,
            -61.326114654541016, -41.989723205566406, -87.8764877319336,
            -71.69316101074219,  -80.24160766601562,  -97.48886108398438,
            -75.89422607421875,  -45.08991622924805,  -88.27134704589844,
            -90.71282958984375,  -93.32392120361328,  -59.14753341674805,
            -45.33106231689453,  -51.32562255859375,  -31.154796600341797,
            -31.62424087524414,  -62.80168151855469,  -63.558509826660156,
            -68.96183013916016,  -43.09415054321289,  -15.803443908691406,
            -64.31092071533203,  -66.45872497558594,  -42.027252197265625,
            -26.032955169677734, -22.73752784729004,  -70.32036590576172,
            -85.28227996826172,  -92.10668182373047
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [{'input': 'averagePool2dInput'}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [-49.258975982666016, -60.52408981323242],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.windowDimensions',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [3, 3]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            43.46498107910156, 49.37273406982422, 42.7481689453125,
            50.038944244384766, 52.452327728271484, 58.46046447753906,
            32.15948486328125, 34.75465393066406, 54.00202560424805,
            49.65404510498047, 41.824440002441406, 35.84912109375,
            43.23125457763672, 37.842769622802734, 32.67961120605469,
            41.17021942138672, 42.79708480834961, 38.987247467041016
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'global averagePool2d float32 4D tensor all positive options.windowDimensions',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [5, 5]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.padding',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'padding': [1, 0, 0, 1]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            52.43666076660156, 49.84208297729492, 47.26926803588867,
            46.15715408325195, 46.63268280029297, 43.616947174072266,
            44.72445297241211, 44.05451583862305
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.strides',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'strides': [2, 2]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            43.46498107910156, 42.7481689453125, 32.15948486328125,
            54.00202560424805, 49.65404510498047, 35.84912109375,
            41.17021942138672, 38.987247467041016
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.dilations',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'dilations': [2, 2]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [32.2001838684082, 42.971012115478516],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.layout=nchw',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments':
            [{'input': 'averagePool2dInput'}, {'options': {'layout': 'nchw'}}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.layout=nhwc',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 91.59549713134766,  78.15438079833984,
            65.64701080322266,  9.686111450195312,  55.14215087890625,
            51.298038482666016, 18.432437896728516, 32.193084716796875,
            49.34624099731445,  87.65037536621094,  15.648024559020996,
            87.25082397460938,  68.02723693847656,  39.49794006347656,
            20.342548370361328, 80.0996322631836,   26.727949142456055,
            10.220142364501953, 64.87446594238281,  52.602699279785156,
            46.5671501159668,   1.4128639698028564, 79.57833099365234,
            11.95406436920166,  4.33846378326416,   85.00074768066406,
            38.183837890625,    64.78374481201172,  45.25398254394531,
            88.03128814697266,  80.9718017578125,   11.333850860595703,
            67.58124542236328,  70.61659240722656,  6.0264997482299805,
            84.90442657470703,  29.7788143157959,   79.06687927246094,
            58.58993148803711,  7.3287248611450195, 2.2384984493255615,
            35.97796630859375,  14.50549030303955,  10.177306175231934,
            68.72449493408203,  1.4140757322311401, 76.45657348632812,
            78.10037994384766,  23.53263282775879
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments':
            [{'input': 'averagePool2dInput'}, {'options': {'layout': 'nhwc'}}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'global averagePool2d float32 4D tensor options.layout=nhwc and options.windowDimensions',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 91.59549713134766,  78.15438079833984,
            65.64701080322266,  9.686111450195312,  55.14215087890625,
            51.298038482666016, 18.432437896728516, 32.193084716796875,
            49.34624099731445,  87.65037536621094,  15.648024559020996,
            87.25082397460938,  68.02723693847656,  39.49794006347656,
            20.342548370361328, 80.0996322631836,   26.727949142456055,
            10.220142364501953, 64.87446594238281,  52.602699279785156,
            46.5671501159668,   1.4128639698028564, 79.57833099365234,
            11.95406436920166,  4.33846378326416,   85.00074768066406,
            38.183837890625,    64.78374481201172,  45.25398254394531,
            88.03128814697266,  80.9718017578125,   11.333850860595703,
            67.58124542236328,  70.61659240722656,  6.0264997482299805,
            84.90442657470703,  29.7788143157959,   79.06687927246094,
            58.58993148803711,  7.3287248611450195, 2.2384984493255615,
            35.97796630859375,  14.50549030303955,  10.177306175231934,
            68.72449493408203,  1.4140757322311401, 76.45657348632812,
            78.10037994384766,  23.53263282775879
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [5, 5], 'layout': 'nhwc'}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.roundingType=floor',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 37.16582489013672, 50.038944244384766,
            58.46046447753906, 52.73374557495117, 39.1442985534668,
            43.23125457763672, 32.67961120605469
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.roundingType=ceil',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 37.16582489013672, 21.206613540649414,
            50.038944244384766, 58.46046447753906, 51.3569221496582,
            37.24428939819336, 54.04661178588867, 78.58363342285156,
            52.73374557495117, 39.1442985534668, 57.1103515625,
            43.23125457763672, 32.67961120605469, 56.23945999145508,
            40.00800323486328, 43.85149002075195, 41.061283111572266
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.roundingType=ceil and no padding',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656
          ],
          'descriptor': {shape: [1, 2, 4, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            51.20364761352539, 40.29140853881836, 50.77684020996094,
            51.70764923095703, 50.63130187988281, 49.3919792175293,
            53.128265380859375, 51.11610412597656
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.layout=nhwc and options.roundingType=floor',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 91.59549713134766,  78.15438079833984,
            65.64701080322266,  9.686111450195312,  55.14215087890625,
            51.298038482666016, 18.432437896728516, 32.193084716796875,
            49.34624099731445,  87.65037536621094,  15.648024559020996,
            87.25082397460938,  68.02723693847656,  39.49794006347656,
            20.342548370361328, 80.0996322631836,   26.727949142456055,
            10.220142364501953, 64.87446594238281,  52.602699279785156,
            46.5671501159668,   1.4128639698028564, 79.57833099365234,
            11.95406436920166,  4.33846378326416,   85.00074768066406,
            38.183837890625,    64.78374481201172,  45.25398254394531,
            88.03128814697266,  80.9718017578125,   11.333850860595703,
            67.58124542236328,  70.61659240722656,  6.0264997482299805,
            84.90442657470703,  29.7788143157959,   79.06687927246094,
            58.58993148803711,  7.3287248611450195, 2.2384984493255615,
            35.97796630859375,  14.50549030303955,  10.177306175231934,
            68.72449493408203,  1.4140757322311401, 76.45657348632812,
            78.10037994384766,  23.53263282775879
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'layout': 'nhwc',
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 52.73374557495117, 37.16582489013672,
            39.1442985534668, 50.038944244384766, 43.23125457763672,
            58.46046447753906, 32.67961120605469
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.layout=nhwc and options.roundingType=ceil',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 91.59549713134766,  78.15438079833984,
            65.64701080322266,  9.686111450195312,  55.14215087890625,
            51.298038482666016, 18.432437896728516, 32.193084716796875,
            49.34624099731445,  87.65037536621094,  15.648024559020996,
            87.25082397460938,  68.02723693847656,  39.49794006347656,
            20.342548370361328, 80.0996322631836,   26.727949142456055,
            10.220142364501953, 64.87446594238281,  52.602699279785156,
            46.5671501159668,   1.4128639698028564, 79.57833099365234,
            11.95406436920166,  4.33846378326416,   85.00074768066406,
            38.183837890625,    64.78374481201172,  45.25398254394531,
            88.03128814697266,  80.9718017578125,   11.333850860595703,
            67.58124542236328,  70.61659240722656,  6.0264997482299805,
            84.90442657470703,  29.7788143157959,   79.06687927246094,
            58.58993148803711,  7.3287248611450195, 2.2384984493255615,
            35.97796630859375,  14.50549030303955,  10.177306175231934,
            68.72449493408203,  1.4140757322311401, 76.45657348632812,
            78.10037994384766,  23.53263282775879
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'layout': 'nhwc',
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 52.73374557495117, 37.16582489013672,
            39.1442985534668, 21.206613540649414, 57.1103515625,
            50.038944244384766, 43.23125457763672, 58.46046447753906,
            32.67961120605469, 51.3569221496582, 56.23945999145508,
            37.24428939819336, 40.00800323486328, 54.04661178588867,
            43.85149002075195, 78.58363342285156, 41.061283111572266
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.outputSizes ignores options.roundingType=floor',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor',
              'outputSizes': [3, 3]
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 37.16582489013672, 21.206613540649414,
            50.038944244384766, 58.46046447753906, 51.3569221496582,
            37.24428939819336, 54.04661178588867, 78.58363342285156,
            52.73374557495117, 39.1442985534668, 57.1103515625,
            43.23125457763672, 32.67961120605469, 56.23945999145508,
            40.00800323486328, 43.85149002075195, 41.061283111572266
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.outputSizes ignores options.roundingType=ceil',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil',
              'outputSizes': [2, 2]
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 37.16582489013672, 50.038944244384766,
            58.46046447753906, 52.73374557495117, 39.1442985534668,
            43.23125457763672, 32.67961120605469
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.dilations with options.strides',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            70.71148681640625,  99.33489990234375,  76.41767883300781,
            39.40980911254883,  38.16328811645508,  45.971256256103516,
            65.3527603149414,   64.51607513427734,  7.725966930389404,
            41.7672004699707,   94.92633819580078,  53.475772857666016,
            95.46460723876953,  58.461795806884766, 15.831390380859375,
            78.41020202636719,  24.454092025756836, 20.630916595458984,
            32.06352233886719,  47.85192108154297,  91.60813903808594,
            72.3534927368164,   74.69429779052734,  28.860214233398438,
            71.82395935058594,  7.989691734313965,  88.16659545898438,
            58.69850540161133,  63.6061897277832,   55.88187789916992,
            52.809974670410156, 72.91474151611328,  46.957664489746094,
            22.10279655456543,  87.14309692382812,  89.6496810913086,
            63.19610595703125,  11.760882377624512, 70.68730926513672,
            57.70444107055664,  1.183821439743042,  25.26912498474121,
            95.29122924804688,  1.9658530950546265, 53.368465423583984,
            21.400854110717773, 55.86185836791992,  27.824508666992188,
            7.642839431762695,  82.34233093261719,  91.75215911865234,
            62.79155731201172,  28.11526107788086,  28.72478675842285,
            29.887035369873047, 66.4310302734375,   7.0103044509887695,
            34.33702087402344,  73.20159912109375,  7.8835601806640625,
            17.82563591003418,  33.799156188964844, 65.01251220703125,
            30.264028549194336, 75.76551818847656,  21.150800704956055,
            60.84249496459961,  98.56522369384766,  62.60990905761719,
            42.42991256713867,  53.142147064208984, 36.29545974731445,
            79.95863342285156,  79.60734558105469,  16.059114456176758,
            19.27552032470703,  53.93022918701172,  48.41620635986328,
            93.00965118408203,  62.086524963378906, 83.50532531738281,
            61.07964324951172,  75.51439666748047,  54.193782806396484,
            2.572873830795288,  59.47652053833008,  34.22541427612305,
            13.07015323638916,  12.419061660766602, 55.82337188720703,
            4.553813934326172,  63.47830581665039,  62.3555908203125,
            56.961090087890625, 34.77016067504883,  0.9611223936080933,
            35.30686950683594,  98.00790405273438
          ],
          'descriptor': {shape: [1, 7, 7, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'layout': 'nhwc'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            42.940242767333984, 55.268165588378906, 51.6013298034668,
            50.220027923583984, 72.13362884521484, 41.542198181152344,
            48.91604232788086, 38.775962829589844, 61.21329879760742,
            49.504154205322266, 57.72294998168945, 38.6922492980957,
            50.19099807739258, 29.15436363220215, 52.98439025878906,
            43.10562515258789, 66.77796936035156, 55.2725830078125
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name':
        'averagePool2d float16 4D constant tensor all positive default options',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [{'input': 'averagePool2dInput'}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.28125, 44.71875],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float16 4D tensor all positive default options',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [{'input': 'averagePool2dInput'}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.28125, 44.71875],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float16 4D tensor all negative default options',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            -83.875,   -2.07421875, -7.5625,   -45.28125,     -16.359375,
            -44.90625, -42.03125,   -44.78125, -1.5068359375, -52.65625,
            -92,       -48,         -61.53125, -93.4375,      -25.78125,
            -95.5,     -10.9609375, -59.125,   -32.59375,     -21.453125,
            -87.125,   -61.3125,    -42,       -87.875,       -71.6875,
            -80.25,    -97.5,       -75.875,   -45.09375,     -88.25,
            -90.6875,  -93.3125,    -59.15625, -45.34375,     -51.3125,
            -31.15625, -31.625,     -62.8125,  -63.5625,      -68.9375,
            -43.09375, -15.8046875, -64.3125,  -66.4375,      -42.03125,
            -26.03125, -22.734375,  -70.3125,  -85.3125,      -92.125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [{'input': 'averagePool2dInput'}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [-49.25, -60.53125],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float16 4D tensor options.windowDimensions',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [3, 3]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            43.46875, 49.375, 42.75, 50.03125, 52.4375, 58.46875, 32.15625,
            34.75, 54, 49.65625, 41.8125, 35.84375, 43.21875, 37.84375, 32.6875,
            41.1875, 42.78125, 39
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'global averagePool2d float16 4D tensor all positive options.windowDimensions',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [5, 5]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.28125, 44.71875],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float16 4D tensor options.padding',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'padding': [1, 0, 0, 1]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            52.4375, 49.84375, 47.28125, 46.15625, 46.625, 43.625, 44.71875,
            44.0625
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float16 4D tensor options.strides',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'strides': [2, 2]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data':
              [43.46875, 42.75, 32.15625, 54, 49.65625, 35.84375, 41.1875, 39],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float16 4D tensor options.dilations',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'dilations': [2, 2]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [32.21875, 42.96875],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float16 4D tensor options.layout=nchw',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments':
            [{'input': 'averagePool2dInput'}, {'options': {'layout': 'nchw'}}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.28125, 44.71875],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float16 4D tensor options.layout=nhwc',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   91.625,     78.125,       65.625,     9.6875,
            55.15625,   51.3125,    18.4375,      32.1875,    49.34375,
            87.625,     15.6484375, 87.25,        68,         39.5,
            20.34375,   80.125,     26.734375,    10.21875,   64.875,
            52.59375,   46.5625,    1.4130859375, 79.5625,    11.953125,
            4.33984375, 85,         38.1875,      64.8125,    45.25,
            88.0625,    81,         11.3359375,   67.5625,    70.625,
            6.02734375, 84.875,     29.78125,     79.0625,    58.59375,
            7.328125,   2.23828125, 35.96875,     14.5078125, 10.1796875,
            68.75,      1.4140625,  76.4375,      78.125,     23.53125
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments':
            [{'input': 'averagePool2dInput'}, {'options': {'layout': 'nhwc'}}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.28125, 44.71875],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'global averagePool2d float16 4D tensor options.layout=nhwc and options.windowDimensions',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   91.625,     78.125,       65.625,     9.6875,
            55.15625,   51.3125,    18.4375,      32.1875,    49.34375,
            87.625,     15.6484375, 87.25,        68,         39.5,
            20.34375,   80.125,     26.734375,    10.21875,   64.875,
            52.59375,   46.5625,    1.4130859375, 79.5625,    11.953125,
            4.33984375, 85,         38.1875,      64.8125,    45.25,
            88.0625,    81,         11.3359375,   67.5625,    70.625,
            6.02734375, 84.875,     29.78125,     79.0625,    58.59375,
            7.328125,   2.23828125, 35.96875,     14.5078125, 10.1796875,
            68.75,      1.4140625,  76.4375,      78.125,     23.53125
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [5, 5], 'layout': 'nhwc'}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.28125, 44.71875],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float16 4D tensor options.roundingType=floor',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.1875, 37.1875, 50.03125, 58.46875, 52.71875, 39.15625, 43.21875,
            32.6875
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float16 4D tensor options.roundingType=ceil',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.1875, 37.1875, 21.203125, 50.03125, 58.46875, 51.375, 37.25,
            54.0625, 78.625, 52.71875, 39.15625, 57.125, 43.21875, 32.6875,
            56.25, 40, 43.84375, 41.0625
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float16 4D tensor options.roundingType=ceil and no padding',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,  9.6875,   51.3125,  32.1875,    87.625,
            87.25,      39.5,    80.125,   10.21875, 52.59375,   1.4130859375,
            11.953125,  85,      64.8125,  88.0625,  11.3359375, 70.625,
            84.875,     79.0625, 7.328125, 35.96875, 10.1796875, 1.4140625,
            78.125,     91.625,  65.625,   55.15625, 18.4375,    49.34375,
            15.6484375, 68
          ],
          'descriptor': {shape: [1, 2, 4, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            51.1875, 40.28125, 50.78125, 51.71875, 50.625, 49.375, 53.125,
            51.09375
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float16 4D tensor options.layout=nhwc and options.roundingType=floor',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   91.625,     78.125,       65.625,     9.6875,
            55.15625,   51.3125,    18.4375,      32.1875,    49.34375,
            87.625,     15.6484375, 87.25,        68,         39.5,
            20.34375,   80.125,     26.734375,    10.21875,   64.875,
            52.59375,   46.5625,    1.4130859375, 79.5625,    11.953125,
            4.33984375, 85,         38.1875,      64.8125,    45.25,
            88.0625,    81,         11.3359375,   67.5625,    70.625,
            6.02734375, 84.875,     29.78125,     79.0625,    58.59375,
            7.328125,   2.23828125, 35.96875,     14.5078125, 10.1796875,
            68.75,      1.4140625,  76.4375,      78.125,     23.53125
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'layout': 'nhwc',
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.1875, 52.71875, 37.1875, 39.15625, 50.03125, 43.21875, 58.46875,
            32.6875
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float16 4D tensor options.layout=nhwc and options.roundingType=ceil',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   91.625,     78.125,       65.625,     9.6875,
            55.15625,   51.3125,    18.4375,      32.1875,    49.34375,
            87.625,     15.6484375, 87.25,        68,         39.5,
            20.34375,   80.125,     26.734375,    10.21875,   64.875,
            52.59375,   46.5625,    1.4130859375, 79.5625,    11.953125,
            4.33984375, 85,         38.1875,      64.8125,    45.25,
            88.0625,    81,         11.3359375,   67.5625,    70.625,
            6.02734375, 84.875,     29.78125,     79.0625,    58.59375,
            7.328125,   2.23828125, 35.96875,     14.5078125, 10.1796875,
            68.75,      1.4140625,  76.4375,      78.125,     23.53125
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'layout': 'nhwc',
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.1875, 52.71875, 37.1875, 39.15625, 21.203125, 57.125, 50.03125,
            43.21875, 58.46875, 32.6875, 51.375, 56.25, 37.25, 40, 54.0625,
            43.84375, 78.625, 41.0625
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float16 4D tensor options.outputSizes ignores options.roundingType=floor',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor',
              'outputSizes': [3, 3]
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.1875, 37.1875, 21.203125, 50.03125, 58.46875, 51.375, 37.25,
            54.0625, 78.625, 52.71875, 39.15625, 57.125, 43.21875, 32.6875,
            56.25, 40, 43.84375, 41.0625
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float16 4D tensor options.outputSizes ignores options.roundingType=ceil',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.96875,   78.125,       9.6875,     51.3125,   32.1875,
            87.625,     87.25,        39.5,       80.125,    10.21875,
            52.59375,   1.4130859375, 11.953125,  85,        64.8125,
            88.0625,    11.3359375,   70.625,     84.875,    79.0625,
            7.328125,   35.96875,     10.1796875, 1.4140625, 78.125,
            91.625,     65.625,       55.15625,   18.4375,   49.34375,
            15.6484375, 68,           20.34375,   26.734375, 64.875,
            46.5625,    79.5625,      4.33984375, 38.1875,   45.25,
            81,         67.5625,      6.02734375, 29.78125,  58.59375,
            2.23828125, 14.5078125,   68.75,      76.4375,   23.53125
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil',
              'outputSizes': [2, 2]
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.1875, 37.1875, 50.03125, 58.46875, 52.71875, 39.15625, 43.21875,
            32.6875
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float16 4D tensor options.dilations with options.strides',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            70.6875,    99.3125,    76.4375,    39.40625,     38.15625,
            45.96875,   65.375,     64.5,       7.7265625,    41.78125,
            94.9375,    53.46875,   95.4375,    58.46875,     15.828125,
            78.4375,    24.453125,  20.625,     32.0625,      47.84375,
            91.625,     72.375,     74.6875,    28.859375,    71.8125,
            7.98828125, 88.1875,    58.6875,    63.59375,     55.875,
            52.8125,    72.9375,    46.96875,   22.109375,    87.125,
            89.625,     63.1875,    11.7578125, 70.6875,      57.71875,
            1.18359375, 25.265625,  95.3125,    1.9658203125, 53.375,
            21.40625,   55.875,     27.828125,  7.64453125,   82.3125,
            91.75,      62.78125,   28.109375,  28.71875,     29.890625,
            66.4375,    7.01171875, 34.34375,   73.1875,      7.8828125,
            17.828125,  33.8125,    65,         30.265625,    75.75,
            21.15625,   60.84375,   98.5625,    62.625,       42.4375,
            53.15625,   36.28125,   79.9375,    79.625,       16.0625,
            19.28125,   53.9375,    48.40625,   93,           62.09375,
            83.5,       61.09375,   75.5,       54.1875,      2.572265625,
            59.46875,   34.21875,   13.0703125, 12.421875,    55.8125,
            4.5546875,  63.46875,   62.34375,   56.96875,     34.78125,
            0.9609375,  35.3125,    98
          ],
          'descriptor': {shape: [1, 7, 7, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'layout': 'nhwc'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            42.9375, 55.25, 51.59375, 50.21875, 72.125, 41.53125, 48.90625,
            38.78125, 61.21875, 49.5, 57.71875, 38.6875, 50.1875, 29.15625,
            52.96875, 43.09375, 66.75, 55.28125
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float16'}
        }
      }
    }
  }
];

webnn_conformance_test(
    averagePool2dTests, buildAndExecuteGraph, getPrecisionTolerance);
