/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

const lazy = {};

ChromeUtils.defineESModuleGetters(lazy, {
  OpaqueDrag: "moz-src:///toolkit/modules/DragDropFilter.sys.mjs",
});

// This component is used for handling dragover and drop of urls.
//
// It checks to see whether a drop of a url is allowed. For instance, a url
// cannot be dropped if it is not a valid uri or the source of the drag cannot
// access the uri. This prevents, for example, a source document from tricking
// the user into dragging a chrome url.

export function ContentAreaDropListener() {}

ContentAreaDropListener.prototype = {
  classID: Components.ID("{1f34bc80-1bc7-11d6-a384-d705dd0746fc}"),
  QueryInterface: ChromeUtils.generateQI(["nsIDroppedLinkHandler"]),

  _addLink(links, url, name, type) {
    links.push({ url, name, type });
  },

  _addLinksFromItem(links, dt, i) {
    let types = dt.mozTypesAt(i);
    let type, data;

    type = "text/uri-list";
    if (types.contains(type)) {
      data = dt.mozGetDataAt(type, i);
      if (data) {
        let urls = data.split("\n");
        for (let url of urls) {
          // lines beginning with # are comments
          if (url.startsWith("#")) {
            continue;
          }
          url = url.replace(/^\s+|\s+$/g, "");
          this._addLink(links, url, url, type);
        }
        return;
      }
    }

    for (let type of ["text/x-moz-url", "application/x-torbrowser-opaque"]) {
      if (!types.contains(type)) {
        continue;
      }
      data = dt.mozGetDataAt(type, i);
      if (data) {
        if (type === "application/x-torbrowser-opaque") {
          ({ type, value: data = "" } = lazy.OpaqueDrag.retrieve(data));
        }
        let lines = data.split("\n");
        for (let i = 0, length = lines.length; i < length; i += 2) {
          this._addLink(links, lines[i], lines[i + 1], type);
        }
        return;
      }
    }

    for (let type of ["text/plain", "text/x-moz-text-internal"]) {
      if (types.contains(type)) {
        data = dt.mozGetDataAt(type, i);
        if (data) {
          let lines = data.replace(/^\s+|\s+$/gm, "").split("\n");
          if (!lines.length) {
            return;
          }

          // For plain text, there are 2 cases:
          //   * if there is at least one URI:
          //       Add all URIs, ignoring non-URI lines, so that all URIs
          //       are opened in tabs.
          //   * if there's no URI:
          //       Add the entire text as a single entry, so that the entire
          //       text is searched.
          let hasURI = false;
          // We don't care whether we are in a private context, because we are
          // only using fixedURI and thus there's no risk to use the wrong
          // search engine.
          let flags =
            Ci.nsIURIFixup.FIXUP_FLAG_FIX_SCHEME_TYPOS |
            Ci.nsIURIFixup.FIXUP_FLAG_ALLOW_KEYWORD_LOOKUP;
          for (let line of lines) {
            let info = Services.uriFixup.getFixupURIInfo(line, flags);
            if (info.fixedURI) {
              // Use the original line here, and let the caller decide
              // whether to perform fixup or not.
              hasURI = true;
              this._addLink(links, line, line, type);
            }
          }

          if (!hasURI) {
            this._addLink(links, data, data, type);
          }
          return;
        }
      }
    }

    // For shortcuts, we want to check for the file type last, so that the
    // url pointed to in one of the url types is found first before the file
    // type, which points to the actual file.
    let files = dt.files;
    if (files && i < files.length) {
      this._addLink(
        links,
        PathUtils.toFileURI(files[i].mozFullPath),
        files[i].name,
        "application/x-moz-file"
      );
    }
  },

  _getDropLinks(dt) {
    let links = [];
    for (let i = 0; i < dt.mozItemCount; i++) {
      this._addLinksFromItem(links, dt, i);
    }
    return links;
  },

  _validateURI(dataTransfer, uriString, disallowInherit, triggeringPrincipal) {
    if (!uriString) {
      return "";
    }

    // Strip leading and trailing whitespace, then try to create a
    // URI from the dropped string. If that succeeds, we're
    // dropping a URI and we need to do a security check to make
    // sure the source document can load the dropped URI.
    uriString = uriString.replace(/^\s*|\s*$/g, "");

    // Apply URI fixup so that this validation prevents bad URIs even if the
    // similar fixup is applied later, especialy fixing typos up will convert
    // non-URI to URI.
    // We don't know if the uri comes from a private context, but luckily we
    // are only using fixedURI, so there's no risk to use the wrong search
    // engine.
    let fixupFlags =
      Ci.nsIURIFixup.FIXUP_FLAG_FIX_SCHEME_TYPOS |
      Ci.nsIURIFixup.FIXUP_FLAG_ALLOW_KEYWORD_LOOKUP;
    let info = Services.uriFixup.getFixupURIInfo(uriString, fixupFlags);
    if (!info.fixedURI || info.keywordProviderName) {
      // Loading a keyword search should always be fine for all cases.
      return uriString;
    }
    let uri = info.fixedURI;

    let secMan = Services.scriptSecurityManager;
    let flags = secMan.STANDARD;
    if (disallowInherit) {
      flags |= secMan.DISALLOW_INHERIT_PRINCIPAL;
    }

    secMan.checkLoadURIWithPrincipal(triggeringPrincipal, uri, flags);

    // Once we validated, return the URI after fixup, instead of the original
    // uriString.
    return uri.spec;
  },

  _getTriggeringPrincipalFromDataTransfer(
    aDataTransfer,
    fallbackToSystemPrincipal
  ) {
    let sourceNode = aDataTransfer.mozSourceNode;
    if (
      sourceNode &&
      (sourceNode.localName !== "browser" ||
        sourceNode.namespaceURI !==
          "http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul")
    ) {
      // Use sourceNode's principal only if the sourceNode is not browser.
      //
      // If sourceNode is browser, the actual triggering principal may be
      // differ than sourceNode's principal, since sourceNode's principal is
      // top level document's one and the drag may be triggered from a frame
      // with different principal.
      if (sourceNode.nodePrincipal) {
        return sourceNode.nodePrincipal;
      }
    }

    // First, fallback to mozTriggeringPrincipalURISpec that is set when the
    // drop comes from another content process.
    let principalURISpec = aDataTransfer.mozTriggeringPrincipalURISpec;
    if (!principalURISpec) {
      // Fallback to either system principal or file principal, supposing
      // the drop comes from outside of the browser, so that drops of file
      // URIs are always allowed.
      //
      // TODO: Investigate and describe the difference between them,
      //       or use only one principal. (Bug 1367038)
      if (fallbackToSystemPrincipal) {
        return Services.scriptSecurityManager.getSystemPrincipal();
      }

      principalURISpec = "file:///";
    }
    return Services.scriptSecurityManager.createContentPrincipal(
      Services.io.newURI(principalURISpec),
      {}
    );
  },

  getTriggeringPrincipal(aEvent) {
    let dataTransfer = aEvent.dataTransfer;
    return this._getTriggeringPrincipalFromDataTransfer(dataTransfer, true);
  },

  getPolicyContainer(aEvent) {
    let sourceNode = aEvent.dataTransfer.mozSourceNode;
    if (aEvent.dataTransfer.policyContainer !== null) {
      return aEvent.dataTransfer.policyContainer;
    }

    if (
      sourceNode &&
      (sourceNode.localName !== "browser" ||
        sourceNode.namespaceURI !==
          "http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul")
    ) {
      // Use sourceNode's policyContainer only if the sourceNode is not browser.
      //
      // If sourceNode is browser, the actual triggering policyContainer may be differ than sourceNode's policyContainer,
      // since sourceNode's policyContainer is top level document's one and the drag may be triggered from a
      // frame with different policyContainer.
      return sourceNode.policyContainer;
    }
    return null;
  },

  canDropLink(aEvent, aAllowSameDocument) {
    if (this._eventTargetIsDisabled(aEvent)) {
      return false;
    }

    let dataTransfer = aEvent.dataTransfer;
    let types = dataTransfer.types;
    if (
      !types.includes("application/x-moz-file") &&
      !types.includes("text/x-moz-url") &&
      !types.includes("application/x-torbrowser-opaque") &&
      !types.includes("text/uri-list") &&
      !types.includes("text/x-moz-text-internal") &&
      !types.includes("text/plain")
    ) {
      return false;
    }

    if (aAllowSameDocument) {
      return true;
    }

    // If this is an external drag, allow drop.
    let sourceTopWC = dataTransfer.sourceTopWindowContext;
    if (!sourceTopWC) {
      return true;
    }

    // If drag source and drop target are in the same top window, don't allow.
    let eventWC =
      aEvent.originalTarget.ownerGlobal.browsingContext.currentWindowContext;
    if (eventWC && sourceTopWC == eventWC.topWindowContext) {
      return false;
    }

    return true;
  },

  dropLinks(aEvent, aDisallowInherit) {
    if (aEvent && this._eventTargetIsDisabled(aEvent)) {
      return [];
    }

    let dataTransfer = aEvent.dataTransfer;
    let links = this._getDropLinks(dataTransfer);
    let triggeringPrincipal = this._getTriggeringPrincipalFromDataTransfer(
      dataTransfer,
      false
    );

    for (let link of links) {
      try {
        link.url = this._validateURI(
          dataTransfer,
          link.url,
          aDisallowInherit,
          triggeringPrincipal
        );
      } catch (ex) {
        // Prevent the drop entirely if any of the links are invalid even if
        // one of them is valid.
        aEvent.stopPropagation();
        aEvent.preventDefault();
        throw ex;
      }
    }

    return links;
  },

  validateURIsForDrop(aEvent, aURIs, aDisallowInherit) {
    let dataTransfer = aEvent.dataTransfer;
    let triggeringPrincipal = this._getTriggeringPrincipalFromDataTransfer(
      dataTransfer,
      false
    );

    for (let uri of aURIs) {
      this._validateURI(
        dataTransfer,
        uri,
        aDisallowInherit,
        triggeringPrincipal
      );
    }
  },

  queryLinks(aDataTransfer) {
    return this._getDropLinks(aDataTransfer);
  },

  _eventTargetIsDisabled(aEvent) {
    let ownerDoc = aEvent.originalTarget.ownerDocument;
    if (!ownerDoc || !ownerDoc.defaultView) {
      return false;
    }

    return ownerDoc.defaultView.windowUtils.isNodeDisabledForEvents(
      aEvent.originalTarget
    );
  },
};
