//
//  MXSignpostMetric.h
//  MetricKit
//
//  Copyright (c) 2019 Apple Inc.
//  All rights reserved.

#import <Foundation/Foundation.h>
#import <MetricKit/MXMetric.h>
#import <MetricKit/MXHistogram.h>
#import <MetricKit/MXCPUMetric.h>
#import <MetricKit/MXMemoryMetric.h>
#import <MetricKit/MXDiskIOMetric.h>
#import <os/availability.h>

NS_ASSUME_NONNULL_BEGIN

/*!
 @class         MXSignpostIntervalData
 @abstract      A class that encapsulates metrics associated with app specific signpost intervals.
 @discussion    These metrics will be collected and aggregated if the associated signposts were emit using MXSignpost or MXSignpostAnimation APIs
 @discussion    To limit on-device overhead, the system will automatically limit the number of signposts (emitted using the MetricKit log handle) processed.
 @discussion    Avoid losing telemetry by limiting usage of signposts (emitted using the MetricKit log handle) to critical sections of code.
 */
API_AVAILABLE(ios(13.0), macos(10.15)) API_UNAVAILABLE(tvos, watchos)
@interface MXSignpostIntervalData : NSObject <NSSecureCoding>

/*!
 @property      histogrammedSignpostDuration
 @abstract      A histogram of signpost intervals durations associated with the given signposts with signpostName and signpostCategory.
 */
@property (readonly, strong, nonnull) MXHistogram<NSUnitDuration *> *histogrammedSignpostDuration;

/*!
 @property      cumulativeCPUTime
 @abstract      Cumulative CPU time aggregated over the MXSignpost intervals.
 @discussion    This property is null when signposts with the associated signpostName and signpostCategory contain no interval metric data.
 */
@property (readonly, strong, nullable) NSMeasurement<NSUnitDuration *> *cumulativeCPUTime;

/*!
 @property      averageMemory
 @abstract      Average value of memory snapshots taken at beginning and end of MXSignpost intervals
 @discussion    This property is null when signposts with the associated signpostName and signpostCategory contain no interval metric data.
 */
@property (readonly, strong, nullable) MXAverage<NSUnitInformationStorage *> *averageMemory;

/*!
 @property      cumulativeLogicalWrites
 @abstract      Cumulative logical writes aggregated over the MXSignpost intervals.
 @discussion    This property is null when signposts with the associated signpostName and signpostCategory contain no interval metric data.
 */
@property (readonly, strong, nullable)  NSMeasurement<NSUnitInformationStorage *> *cumulativeLogicalWrites;

/*!
 @property      cumulativeHitchTimeRatio
 @abstract      Cumulative hitch time ratio aggregated over the MXSignpostAnimation intervals.
 @discussion    This property is null when signposts with the associated signpostName and signpostCategory contain no interval metric data.
 */
@property (readonly, strong, nullable) NSMeasurement<NSUnit *> *cumulativeHitchTimeRatio API_AVAILABLE(ios(15.0), macos(12.0)) API_UNAVAILABLE(tvos, watchos);

@end

/*!
 @class         MXSignpostMetric
 @abstract      An MXMetric subclass that encapsulates signpost metrics.
 @discussion    Signposts emit using the os_log_t generated by makeLogHandleWithCategory: in MXMetricManger can be flagged for aggregation and reported back in MXMetricPayload.
 */
API_AVAILABLE(ios(13.0), macos(10.15)) API_UNAVAILABLE(tvos, watchos)
@interface MXSignpostMetric : MXMetric

/*!
 @property      signpostName
 @abstract      The name associated with this aggregated signpost.
 */
@property (readonly, strong, nonnull) NSString *signpostName;

/*!
 @property      signpostCategory
 @abstract      The category associated with this aggregated signpost.
 */
@property (readonly, strong, nonnull) NSString *signpostCategory;

/*!
 @property      signpostIntervalData
 @abstract      A class that encapsulates metrics associated with app specific signpost intervals.
 @discussion    This property is null when signposts with the associated signpostName and signpostCategory contain no intervals.
 */
@property (readonly, strong, nullable) MXSignpostIntervalData *signpostIntervalData;

/*!
 @property      totalCount
 @abstract      The total number of signposts emit with the given signpostName in the aggregation period of the parent payload.
 */
@property (readonly) NSUInteger totalCount;

@end

NS_ASSUME_NONNULL_END
