//
//  hv_gic_types.h
//  Hypervisor
//
//  Copyright © 2023-2024 Apple Inc. All rights reserved.
//

#pragma once

#ifdef __arm64__

#include <arm64/hv/hv_kern_types.h>
#include <os/object.h>

OS_ASSUME_NONNULL_BEGIN

__BEGIN_DECLS

/*!
 @abstract Configuration for hv_gic_create().
*/
#if OS_OBJECT_USE_OBJC
OS_OBJECT_DECL(hv_gic_config);
#else // OS_OBJECT_USE_OBJC
typedef struct hv_gic_config_s *hv_gic_config_t;
#endif // OS_OBJECT_USE_OBJC

/*!
 @abstract GIC state for hv_gic_state_get_data() and hv_gic_state_get_size().
*/
#if OS_OBJECT_USE_OBJC
OS_OBJECT_DECL(hv_gic_state);
#else // OS_OBJECT_USE_OBJC
typedef struct hv_gic_state_s *hv_gic_state_t;
#endif // OS_OBJECT_USE_OBJC

/*!
 @abstract Type of an ARM GIC interrupt id.
 @discussion
    Note that HV_GIC_INT_MAINTENANCE and HV_GIC_INT_EL2_PHYSICAL_TIMER are
    only present when EL2 (nested virtualization) is enabled.
*/
API_AVAILABLE(macos(15.0))
OS_ENUM(hv_gic_intid, uint16_t,
    HV_GIC_INT_PERFORMANCE_MONITOR = 23,
    HV_GIC_INT_MAINTENANCE = 25,
    HV_GIC_INT_EL2_PHYSICAL_TIMER = 26,
    HV_GIC_INT_EL1_VIRTUAL_TIMER = 27,
    HV_GIC_INT_EL1_PHYSICAL_TIMER = 30,
);

/*!
 @abstract Type of an ARM GIC distributor register.
*/
API_AVAILABLE(macos(15.0))
OS_ENUM(hv_gic_distributor_reg, uint16_t,
    HV_GIC_DISTRIBUTOR_REG_GICD_CTLR = 0x0000,
    HV_GIC_DISTRIBUTOR_REG_GICD_TYPER = 0x0004,

    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR0 = 0x0080,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR1 = 0x0084,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR2 = 0x0088,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR3 = 0x008c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR4 = 0x0090,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR5 = 0x0094,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR6 = 0x0098,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR7 = 0x009c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR8 = 0x00a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR9 = 0x00a4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR10 = 0x00a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR11 = 0x00ac,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR12 = 0x00b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR13 = 0x00b4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR14 = 0x00b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR15 = 0x00bc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR16 = 0x00c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR17 = 0x00c4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR18 = 0x00c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR19 = 0x00cc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR20 = 0x00d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR21 = 0x00d4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR22 = 0x00d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR23 = 0x00dc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR24 = 0x00e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR25 = 0x00e4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR26 = 0x00e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR27 = 0x00ec,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR28 = 0x00f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR29 = 0x00f4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR30 = 0x00f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IGROUPR31 = 0x00fc,

    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER0 = 0x0100,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER1 = 0x0104,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER2 = 0x0108,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER3 = 0x010c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER4 = 0x0110,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER5 = 0x0114,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER6 = 0x0118,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER7 = 0x011c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER8 = 0x0120,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER9 = 0x0124,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER10 = 0x0128,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER11 = 0x012c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER12 = 0x0130,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER13 = 0x0134,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER14 = 0x0138,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER15 = 0x013c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER16 = 0x0140,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER17 = 0x0144,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER18 = 0x0148,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER19 = 0x014c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER20 = 0x0150,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER21 = 0x0154,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER22 = 0x0158,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER23 = 0x015c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER24 = 0x0160,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER25 = 0x0164,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER26 = 0x0168,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER27 = 0x016c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER28 = 0x0170,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER29 = 0x0174,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER30 = 0x0178,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISENABLER31 = 0x017c,

    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER0 = 0x0180,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER1 = 0x0184,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER2 = 0x0188,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER3 = 0x018c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER4 = 0x0190,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER5 = 0x0194,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER6 = 0x0198,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER7 = 0x019c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER8 = 0x01a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER9 = 0x01a4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER10 = 0x01a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER11 = 0x01ac,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER12 = 0x01b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER13 = 0x01b4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER14 = 0x01b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER15 = 0x01bc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER16 = 0x01c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER17 = 0x01c4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER18 = 0x01c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER19 = 0x01cc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER20 = 0x01d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER21 = 0x01d4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER22 = 0x01d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER23 = 0x01dc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER24 = 0x01e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER25 = 0x01e4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER26 = 0x01e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER27 = 0x01ec,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER28 = 0x01f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER29 = 0x01f4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER30 = 0x01f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICENABLER31 = 0x01fc,

    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR0 = 0x0200,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR1 = 0x0204,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR2 = 0x0208,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR3 = 0x020c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR4 = 0x0210,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR5 = 0x0214,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR6 = 0x0218,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR7 = 0x021c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR8 = 0x0220,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR9 = 0x0224,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR10 = 0x0228,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR11 = 0x022c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR12 = 0x0230,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR13 = 0x0234,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR14 = 0x0238,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR15 = 0x023c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR16 = 0x0240,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR17 = 0x0244,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR18 = 0x0248,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR19 = 0x024c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR20 = 0x0250,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR21 = 0x0254,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR22 = 0x0258,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR23 = 0x025c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR24 = 0x0260,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR25 = 0x0264,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR26 = 0x0268,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR27 = 0x026c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR28 = 0x0270,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR29 = 0x0274,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR30 = 0x0278,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISPENDR31 = 0x027c,

    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR0 = 0x0280,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR1 = 0x0284,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR2 = 0x0288,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR3 = 0x028c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR4 = 0x0290,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR5 = 0x0294,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR6 = 0x0298,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR7 = 0x029c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR8 = 0x02a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR9 = 0x02a4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR10 = 0x02a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR11 = 0x02ac,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR12 = 0x02b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR13 = 0x02b4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR14 = 0x02b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR15 = 0x02bc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR16 = 0x02c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR17 = 0x02c4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR18 = 0x02c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR19 = 0x02cc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR20 = 0x02d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR21 = 0x02d4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR22 = 0x02d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR23 = 0x02dc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR24 = 0x02e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR25 = 0x02e4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR26 = 0x02e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR27 = 0x02ec,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR28 = 0x02f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR29 = 0x02f4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR30 = 0x02f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICPENDR31 = 0x02fc,

    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER0 = 0x0300,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER1 = 0x0304,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER2 = 0x0308,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER3 = 0x030c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER4 = 0x0310,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER5 = 0x0314,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER6 = 0x0318,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER7 = 0x031c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER8 = 0x0320,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER9 = 0x0324,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER10 = 0x0328,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER11 = 0x032c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER12 = 0x0330,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER13 = 0x0334,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER14 = 0x0338,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER15 = 0x033c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER16 = 0x0340,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER17 = 0x0344,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER18 = 0x0348,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER19 = 0x034c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER20 = 0x0350,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER21 = 0x0354,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER22 = 0x0358,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER23 = 0x035c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER24 = 0x0360,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER25 = 0x0364,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER26 = 0x0368,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER27 = 0x036c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER28 = 0x0370,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER29 = 0x0374,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER30 = 0x0378,
    HV_GIC_DISTRIBUTOR_REG_GICD_ISACTIVER31 = 0x037c,

    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER0 = 0x0380,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER1 = 0x0384,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER2 = 0x0388,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER3 = 0x038c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER4 = 0x0390,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER5 = 0x0394,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER6 = 0x0398,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER7 = 0x039c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER8 = 0x03a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER9 = 0x03a4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER10 = 0x03a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER11 = 0x03ac,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER12 = 0x03b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER13 = 0x03b4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER14 = 0x03b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER15 = 0x03bc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER16 = 0x03c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER17 = 0x03c4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER18 = 0x03c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER19 = 0x03cc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER20 = 0x03d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER21 = 0x03d4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER22 = 0x03d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER23 = 0x03dc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER24 = 0x03e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER25 = 0x03e4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER26 = 0x03e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER27 = 0x03ec,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER28 = 0x03f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER29 = 0x03f4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER30 = 0x03f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICACTIVER31 = 0x03fc,

    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR0 = 0x0400,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR1 = 0x0404,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR2 = 0x0408,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR3 = 0x040c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR4 = 0x0410,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR5 = 0x0414,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR6 = 0x0418,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR7 = 0x041c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR8 = 0x0420,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR9 = 0x0424,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR10 = 0x0428,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR11 = 0x042c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR12 = 0x0430,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR13 = 0x0434,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR14 = 0x0438,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR15 = 0x043c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR16 = 0x0440,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR17 = 0x0444,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR18 = 0x0448,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR19 = 0x044c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR20 = 0x0450,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR21 = 0x0454,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR22 = 0x0458,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR23 = 0x045c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR24 = 0x0460,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR25 = 0x0464,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR26 = 0x0468,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR27 = 0x046c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR28 = 0x0470,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR29 = 0x0474,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR30 = 0x0478,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR31 = 0x047c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR32 = 0x0480,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR33 = 0x0484,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR34 = 0x0488,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR35 = 0x048c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR36 = 0x0490,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR37 = 0x0494,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR38 = 0x0498,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR39 = 0x049c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR40 = 0x04a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR41 = 0x04a4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR42 = 0x04a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR43 = 0x04ac,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR44 = 0x04b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR45 = 0x04b4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR46 = 0x04b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR47 = 0x04bc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR48 = 0x04c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR49 = 0x04c4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR50 = 0x04c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR51 = 0x04cc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR52 = 0x04d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR53 = 0x04d4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR54 = 0x04d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR55 = 0x04dc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR56 = 0x04e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR57 = 0x04e4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR58 = 0x04e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR59 = 0x04ec,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR60 = 0x04f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR61 = 0x04f4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR62 = 0x04f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR63 = 0x04fc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR64 = 0x0500,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR65 = 0x0504,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR66 = 0x0508,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR67 = 0x050c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR68 = 0x0510,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR69 = 0x0514,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR70 = 0x0518,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR71 = 0x051c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR72 = 0x0520,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR73 = 0x0524,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR74 = 0x0528,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR75 = 0x052c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR76 = 0x0530,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR77 = 0x0534,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR78 = 0x0538,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR79 = 0x053c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR80 = 0x0540,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR81 = 0x0544,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR82 = 0x0548,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR83 = 0x054c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR84 = 0x0550,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR85 = 0x0554,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR86 = 0x0558,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR87 = 0x055c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR88 = 0x0560,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR89 = 0x0564,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR90 = 0x0568,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR91 = 0x056c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR92 = 0x0570,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR93 = 0x0574,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR94 = 0x0578,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR95 = 0x057c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR96 = 0x0580,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR97 = 0x0584,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR98 = 0x0588,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR99 = 0x058c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR100 = 0x0590,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR101 = 0x0594,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR102 = 0x0598,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR103 = 0x059c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR104 = 0x05a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR105 = 0x05a4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR106 = 0x05a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR107 = 0x05ac,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR108 = 0x05b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR109 = 0x05b4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR110 = 0x05b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR111 = 0x05bc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR112 = 0x05c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR113 = 0x05c4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR114 = 0x05c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR115 = 0x05cc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR116 = 0x05d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR117 = 0x05d4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR118 = 0x05d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR119 = 0x05dc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR120 = 0x05e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR121 = 0x05e4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR122 = 0x05e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR123 = 0x05ec,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR124 = 0x05f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR125 = 0x05f4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR126 = 0x05f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR127 = 0x05fc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR128 = 0x0600,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR129 = 0x0604,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR130 = 0x0608,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR131 = 0x060c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR132 = 0x0610,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR133 = 0x0614,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR134 = 0x0618,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR135 = 0x061c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR136 = 0x0620,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR137 = 0x0624,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR138 = 0x0628,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR139 = 0x062c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR140 = 0x0630,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR141 = 0x0634,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR142 = 0x0638,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR143 = 0x063c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR144 = 0x0640,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR145 = 0x0644,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR146 = 0x0648,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR147 = 0x064c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR148 = 0x0650,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR149 = 0x0654,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR150 = 0x0658,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR151 = 0x065c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR152 = 0x0660,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR153 = 0x0664,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR154 = 0x0668,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR155 = 0x066c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR156 = 0x0670,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR157 = 0x0674,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR158 = 0x0678,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR159 = 0x067c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR160 = 0x0680,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR161 = 0x0684,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR162 = 0x0688,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR163 = 0x068c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR164 = 0x0690,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR165 = 0x0694,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR166 = 0x0698,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR167 = 0x069c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR168 = 0x06a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR169 = 0x06a4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR170 = 0x06a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR171 = 0x06ac,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR172 = 0x06b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR173 = 0x06b4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR174 = 0x06b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR175 = 0x06bc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR176 = 0x06c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR177 = 0x06c4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR178 = 0x06c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR179 = 0x06cc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR180 = 0x06d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR181 = 0x06d4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR182 = 0x06d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR183 = 0x06dc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR184 = 0x06e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR185 = 0x06e4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR186 = 0x06e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR187 = 0x06ec,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR188 = 0x06f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR189 = 0x06f4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR190 = 0x06f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR191 = 0x06fc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR192 = 0x0700,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR193 = 0x0704,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR194 = 0x0708,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR195 = 0x070c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR196 = 0x0710,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR197 = 0x0714,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR198 = 0x0718,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR199 = 0x071c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR200 = 0x0720,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR201 = 0x0724,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR202 = 0x0728,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR203 = 0x072c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR204 = 0x0730,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR205 = 0x0734,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR206 = 0x0738,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR207 = 0x073c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR208 = 0x0740,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR209 = 0x0744,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR210 = 0x0748,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR211 = 0x074c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR212 = 0x0750,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR213 = 0x0754,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR214 = 0x0758,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR215 = 0x075c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR216 = 0x0760,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR217 = 0x0764,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR218 = 0x0768,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR219 = 0x076c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR220 = 0x0770,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR221 = 0x0774,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR222 = 0x0778,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR223 = 0x077c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR224 = 0x0780,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR225 = 0x0784,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR226 = 0x0788,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR227 = 0x078c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR228 = 0x0790,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR229 = 0x0794,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR230 = 0x0798,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR231 = 0x079c,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR232 = 0x07a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR233 = 0x07a4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR234 = 0x07a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR235 = 0x07ac,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR236 = 0x07b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR237 = 0x07b4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR238 = 0x07b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR239 = 0x07bc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR240 = 0x07c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR241 = 0x07c4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR242 = 0x07c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR243 = 0x07cc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR244 = 0x07d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR245 = 0x07d4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR246 = 0x07d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR247 = 0x07dc,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR248 = 0x07e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR249 = 0x07e4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR250 = 0x07e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR251 = 0x07ec,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR252 = 0x07f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR253 = 0x07f4,
    HV_GIC_DISTRIBUTOR_REG_GICD_IPRIORITYR254 = 0x07f8,

    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR0 = 0x0c00,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR1 = 0x0c04,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR2 = 0x0c08,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR3 = 0x0c0c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR4 = 0x0c10,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR5 = 0x0c14,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR6 = 0x0c18,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR7 = 0x0c1c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR8 = 0x0c20,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR9 = 0x0c24,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR10 = 0x0c28,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR11 = 0x0c2c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR12 = 0x0c30,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR13 = 0x0c34,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR14 = 0x0c38,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR15 = 0x0c3c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR16 = 0x0c40,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR17 = 0x0c44,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR18 = 0x0c48,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR19 = 0x0c4c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR20 = 0x0c50,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR21 = 0x0c54,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR22 = 0x0c58,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR23 = 0x0c5c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR24 = 0x0c60,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR25 = 0x0c64,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR26 = 0x0c68,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR27 = 0x0c6c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR28 = 0x0c70,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR29 = 0x0c74,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR30 = 0x0c78,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR31 = 0x0c7c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR32 = 0x0c80,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR33 = 0x0c84,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR34 = 0x0c88,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR35 = 0x0c8c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR36 = 0x0c90,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR37 = 0x0c94,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR38 = 0x0c98,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR39 = 0x0c9c,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR40 = 0x0ca0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR41 = 0x0ca4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR42 = 0x0ca8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR43 = 0x0cac,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR44 = 0x0cb0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR45 = 0x0cb4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR46 = 0x0cb8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR47 = 0x0cbc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR48 = 0x0cc0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR49 = 0x0cc4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR50 = 0x0cc8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR51 = 0x0ccc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR52 = 0x0cd0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR53 = 0x0cd4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR54 = 0x0cd8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR55 = 0x0cdc,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR56 = 0x0ce0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR57 = 0x0ce4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR58 = 0x0ce8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR59 = 0x0cec,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR60 = 0x0cf0,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR61 = 0x0cf4,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR62 = 0x0cf8,
    HV_GIC_DISTRIBUTOR_REG_GICD_ICFGR63 = 0x0cfc,

    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER32 = 0x6100,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER33 = 0x6108,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER34 = 0x6110,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER35 = 0x6118,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER36 = 0x6120,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER37 = 0x6128,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER38 = 0x6130,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER39 = 0x6138,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER40 = 0x6140,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER41 = 0x6148,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER42 = 0x6150,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER43 = 0x6158,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER44 = 0x6160,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER45 = 0x6168,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER46 = 0x6170,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER47 = 0x6178,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER48 = 0x6180,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER49 = 0x6188,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER50 = 0x6190,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER51 = 0x6198,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER52 = 0x61a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER53 = 0x61a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER54 = 0x61b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER55 = 0x61b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER56 = 0x61c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER57 = 0x61c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER58 = 0x61d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER59 = 0x61d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER60 = 0x61e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER61 = 0x61e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER62 = 0x61f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER63 = 0x61f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER64 = 0x6200,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER65 = 0x6208,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER66 = 0x6210,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER67 = 0x6218,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER68 = 0x6220,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER69 = 0x6228,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER70 = 0x6230,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER71 = 0x6238,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER72 = 0x6240,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER73 = 0x6248,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER74 = 0x6250,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER75 = 0x6258,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER76 = 0x6260,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER77 = 0x6268,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER78 = 0x6270,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER79 = 0x6278,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER80 = 0x6280,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER81 = 0x6288,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER82 = 0x6290,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER83 = 0x6298,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER84 = 0x62a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER85 = 0x62a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER86 = 0x62b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER87 = 0x62b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER88 = 0x62c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER89 = 0x62c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER90 = 0x62d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER91 = 0x62d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER92 = 0x62e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER93 = 0x62e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER94 = 0x62f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER95 = 0x62f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER96 = 0x6300,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER97 = 0x6308,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER98 = 0x6310,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER99 = 0x6318,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER100 = 0x6320,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER101 = 0x6328,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER102 = 0x6330,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER103 = 0x6338,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER104 = 0x6340,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER105 = 0x6348,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER106 = 0x6350,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER107 = 0x6358,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER108 = 0x6360,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER109 = 0x6368,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER110 = 0x6370,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER111 = 0x6378,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER112 = 0x6380,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER113 = 0x6388,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER114 = 0x6390,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER115 = 0x6398,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER116 = 0x63a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER117 = 0x63a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER118 = 0x63b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER119 = 0x63b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER120 = 0x63c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER121 = 0x63c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER122 = 0x63d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER123 = 0x63d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER124 = 0x63e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER125 = 0x63e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER126 = 0x63f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER127 = 0x63f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER128 = 0x6400,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER129 = 0x6408,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER130 = 0x6410,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER131 = 0x6418,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER132 = 0x6420,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER133 = 0x6428,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER134 = 0x6430,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER135 = 0x6438,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER136 = 0x6440,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER137 = 0x6448,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER138 = 0x6450,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER139 = 0x6458,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER140 = 0x6460,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER141 = 0x6468,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER142 = 0x6470,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER143 = 0x6478,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER144 = 0x6480,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER145 = 0x6488,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER146 = 0x6490,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER147 = 0x6498,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER148 = 0x64a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER149 = 0x64a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER150 = 0x64b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER151 = 0x64b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER152 = 0x64c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER153 = 0x64c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER154 = 0x64d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER155 = 0x64d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER156 = 0x64e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER157 = 0x64e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER158 = 0x64f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER159 = 0x64f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER160 = 0x6500,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER161 = 0x6508,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER162 = 0x6510,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER163 = 0x6518,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER164 = 0x6520,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER165 = 0x6528,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER166 = 0x6530,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER167 = 0x6538,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER168 = 0x6540,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER169 = 0x6548,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER170 = 0x6550,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER171 = 0x6558,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER172 = 0x6560,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER173 = 0x6568,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER174 = 0x6570,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER175 = 0x6578,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER176 = 0x6580,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER177 = 0x6588,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER178 = 0x6590,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER179 = 0x6598,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER180 = 0x65a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER181 = 0x65a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER182 = 0x65b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER183 = 0x65b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER184 = 0x65c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER185 = 0x65c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER186 = 0x65d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER187 = 0x65d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER188 = 0x65e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER189 = 0x65e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER190 = 0x65f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER191 = 0x65f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER192 = 0x6600,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER193 = 0x6608,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER194 = 0x6610,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER195 = 0x6618,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER196 = 0x6620,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER197 = 0x6628,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER198 = 0x6630,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER199 = 0x6638,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER200 = 0x6640,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER201 = 0x6648,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER202 = 0x6650,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER203 = 0x6658,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER204 = 0x6660,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER205 = 0x6668,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER206 = 0x6670,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER207 = 0x6678,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER208 = 0x6680,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER209 = 0x6688,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER210 = 0x6690,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER211 = 0x6698,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER212 = 0x66a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER213 = 0x66a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER214 = 0x66b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER215 = 0x66b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER216 = 0x66c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER217 = 0x66c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER218 = 0x66d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER219 = 0x66d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER220 = 0x66e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER221 = 0x66e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER222 = 0x66f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER223 = 0x66f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER224 = 0x6700,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER225 = 0x6708,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER226 = 0x6710,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER227 = 0x6718,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER228 = 0x6720,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER229 = 0x6728,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER230 = 0x6730,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER231 = 0x6738,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER232 = 0x6740,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER233 = 0x6748,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER234 = 0x6750,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER235 = 0x6758,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER236 = 0x6760,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER237 = 0x6768,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER238 = 0x6770,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER239 = 0x6778,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER240 = 0x6780,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER241 = 0x6788,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER242 = 0x6790,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER243 = 0x6798,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER244 = 0x67a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER245 = 0x67a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER246 = 0x67b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER247 = 0x67b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER248 = 0x67c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER249 = 0x67c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER250 = 0x67d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER251 = 0x67d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER252 = 0x67e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER253 = 0x67e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER254 = 0x67f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER255 = 0x67f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER256 = 0x6800,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER257 = 0x6808,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER258 = 0x6810,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER259 = 0x6818,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER260 = 0x6820,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER261 = 0x6828,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER262 = 0x6830,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER263 = 0x6838,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER264 = 0x6840,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER265 = 0x6848,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER266 = 0x6850,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER267 = 0x6858,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER268 = 0x6860,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER269 = 0x6868,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER270 = 0x6870,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER271 = 0x6878,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER272 = 0x6880,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER273 = 0x6888,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER274 = 0x6890,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER275 = 0x6898,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER276 = 0x68a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER277 = 0x68a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER278 = 0x68b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER279 = 0x68b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER280 = 0x68c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER281 = 0x68c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER282 = 0x68d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER283 = 0x68d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER284 = 0x68e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER285 = 0x68e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER286 = 0x68f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER287 = 0x68f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER288 = 0x6900,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER289 = 0x6908,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER290 = 0x6910,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER291 = 0x6918,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER292 = 0x6920,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER293 = 0x6928,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER294 = 0x6930,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER295 = 0x6938,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER296 = 0x6940,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER297 = 0x6948,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER298 = 0x6950,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER299 = 0x6958,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER300 = 0x6960,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER301 = 0x6968,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER302 = 0x6970,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER303 = 0x6978,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER304 = 0x6980,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER305 = 0x6988,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER306 = 0x6990,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER307 = 0x6998,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER308 = 0x69a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER309 = 0x69a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER310 = 0x69b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER311 = 0x69b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER312 = 0x69c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER313 = 0x69c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER314 = 0x69d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER315 = 0x69d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER316 = 0x69e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER317 = 0x69e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER318 = 0x69f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER319 = 0x69f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER320 = 0x6a00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER321 = 0x6a08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER322 = 0x6a10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER323 = 0x6a18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER324 = 0x6a20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER325 = 0x6a28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER326 = 0x6a30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER327 = 0x6a38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER328 = 0x6a40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER329 = 0x6a48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER330 = 0x6a50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER331 = 0x6a58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER332 = 0x6a60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER333 = 0x6a68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER334 = 0x6a70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER335 = 0x6a78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER336 = 0x6a80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER337 = 0x6a88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER338 = 0x6a90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER339 = 0x6a98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER340 = 0x6aa0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER341 = 0x6aa8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER342 = 0x6ab0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER343 = 0x6ab8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER344 = 0x6ac0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER345 = 0x6ac8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER346 = 0x6ad0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER347 = 0x6ad8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER348 = 0x6ae0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER349 = 0x6ae8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER350 = 0x6af0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER351 = 0x6af8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER352 = 0x6b00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER353 = 0x6b08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER354 = 0x6b10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER355 = 0x6b18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER356 = 0x6b20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER357 = 0x6b28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER358 = 0x6b30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER359 = 0x6b38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER360 = 0x6b40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER361 = 0x6b48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER362 = 0x6b50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER363 = 0x6b58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER364 = 0x6b60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER365 = 0x6b68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER366 = 0x6b70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER367 = 0x6b78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER368 = 0x6b80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER369 = 0x6b88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER370 = 0x6b90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER371 = 0x6b98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER372 = 0x6ba0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER373 = 0x6ba8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER374 = 0x6bb0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER375 = 0x6bb8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER376 = 0x6bc0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER377 = 0x6bc8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER378 = 0x6bd0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER379 = 0x6bd8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER380 = 0x6be0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER381 = 0x6be8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER382 = 0x6bf0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER383 = 0x6bf8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER384 = 0x6c00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER385 = 0x6c08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER386 = 0x6c10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER387 = 0x6c18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER388 = 0x6c20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER389 = 0x6c28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER390 = 0x6c30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER391 = 0x6c38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER392 = 0x6c40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER393 = 0x6c48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER394 = 0x6c50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER395 = 0x6c58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER396 = 0x6c60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER397 = 0x6c68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER398 = 0x6c70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER399 = 0x6c78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER400 = 0x6c80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER401 = 0x6c88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER402 = 0x6c90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER403 = 0x6c98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER404 = 0x6ca0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER405 = 0x6ca8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER406 = 0x6cb0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER407 = 0x6cb8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER408 = 0x6cc0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER409 = 0x6cc8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER410 = 0x6cd0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER411 = 0x6cd8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER412 = 0x6ce0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER413 = 0x6ce8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER414 = 0x6cf0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER415 = 0x6cf8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER416 = 0x6d00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER417 = 0x6d08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER418 = 0x6d10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER419 = 0x6d18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER420 = 0x6d20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER421 = 0x6d28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER422 = 0x6d30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER423 = 0x6d38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER424 = 0x6d40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER425 = 0x6d48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER426 = 0x6d50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER427 = 0x6d58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER428 = 0x6d60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER429 = 0x6d68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER430 = 0x6d70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER431 = 0x6d78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER432 = 0x6d80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER433 = 0x6d88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER434 = 0x6d90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER435 = 0x6d98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER436 = 0x6da0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER437 = 0x6da8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER438 = 0x6db0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER439 = 0x6db8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER440 = 0x6dc0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER441 = 0x6dc8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER442 = 0x6dd0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER443 = 0x6dd8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER444 = 0x6de0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER445 = 0x6de8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER446 = 0x6df0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER447 = 0x6df8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER448 = 0x6e00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER449 = 0x6e08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER450 = 0x6e10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER451 = 0x6e18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER452 = 0x6e20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER453 = 0x6e28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER454 = 0x6e30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER455 = 0x6e38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER456 = 0x6e40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER457 = 0x6e48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER458 = 0x6e50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER459 = 0x6e58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER460 = 0x6e60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER461 = 0x6e68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER462 = 0x6e70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER463 = 0x6e78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER464 = 0x6e80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER465 = 0x6e88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER466 = 0x6e90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER467 = 0x6e98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER468 = 0x6ea0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER469 = 0x6ea8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER470 = 0x6eb0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER471 = 0x6eb8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER472 = 0x6ec0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER473 = 0x6ec8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER474 = 0x6ed0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER475 = 0x6ed8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER476 = 0x6ee0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER477 = 0x6ee8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER478 = 0x6ef0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER479 = 0x6ef8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER480 = 0x6f00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER481 = 0x6f08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER482 = 0x6f10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER483 = 0x6f18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER484 = 0x6f20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER485 = 0x6f28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER486 = 0x6f30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER487 = 0x6f38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER488 = 0x6f40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER489 = 0x6f48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER490 = 0x6f50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER491 = 0x6f58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER492 = 0x6f60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER493 = 0x6f68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER494 = 0x6f70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER495 = 0x6f78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER496 = 0x6f80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER497 = 0x6f88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER498 = 0x6f90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER499 = 0x6f98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER500 = 0x6fa0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER501 = 0x6fa8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER502 = 0x6fb0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER503 = 0x6fb8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER504 = 0x6fc0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER505 = 0x6fc8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER506 = 0x6fd0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER507 = 0x6fd8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER508 = 0x6fe0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER509 = 0x6fe8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER510 = 0x6ff0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER511 = 0x6ff8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER512 = 0x7000,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER513 = 0x7008,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER514 = 0x7010,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER515 = 0x7018,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER516 = 0x7020,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER517 = 0x7028,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER518 = 0x7030,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER519 = 0x7038,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER520 = 0x7040,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER521 = 0x7048,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER522 = 0x7050,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER523 = 0x7058,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER524 = 0x7060,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER525 = 0x7068,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER526 = 0x7070,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER527 = 0x7078,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER528 = 0x7080,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER529 = 0x7088,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER530 = 0x7090,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER531 = 0x7098,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER532 = 0x70a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER533 = 0x70a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER534 = 0x70b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER535 = 0x70b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER536 = 0x70c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER537 = 0x70c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER538 = 0x70d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER539 = 0x70d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER540 = 0x70e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER541 = 0x70e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER542 = 0x70f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER543 = 0x70f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER544 = 0x7100,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER545 = 0x7108,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER546 = 0x7110,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER547 = 0x7118,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER548 = 0x7120,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER549 = 0x7128,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER550 = 0x7130,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER551 = 0x7138,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER552 = 0x7140,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER553 = 0x7148,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER554 = 0x7150,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER555 = 0x7158,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER556 = 0x7160,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER557 = 0x7168,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER558 = 0x7170,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER559 = 0x7178,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER560 = 0x7180,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER561 = 0x7188,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER562 = 0x7190,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER563 = 0x7198,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER564 = 0x71a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER565 = 0x71a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER566 = 0x71b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER567 = 0x71b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER568 = 0x71c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER569 = 0x71c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER570 = 0x71d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER571 = 0x71d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER572 = 0x71e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER573 = 0x71e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER574 = 0x71f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER575 = 0x71f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER576 = 0x7200,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER577 = 0x7208,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER578 = 0x7210,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER579 = 0x7218,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER580 = 0x7220,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER581 = 0x7228,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER582 = 0x7230,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER583 = 0x7238,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER584 = 0x7240,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER585 = 0x7248,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER586 = 0x7250,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER587 = 0x7258,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER588 = 0x7260,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER589 = 0x7268,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER590 = 0x7270,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER591 = 0x7278,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER592 = 0x7280,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER593 = 0x7288,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER594 = 0x7290,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER595 = 0x7298,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER596 = 0x72a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER597 = 0x72a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER598 = 0x72b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER599 = 0x72b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER600 = 0x72c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER601 = 0x72c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER602 = 0x72d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER603 = 0x72d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER604 = 0x72e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER605 = 0x72e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER606 = 0x72f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER607 = 0x72f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER608 = 0x7300,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER609 = 0x7308,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER610 = 0x7310,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER611 = 0x7318,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER612 = 0x7320,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER613 = 0x7328,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER614 = 0x7330,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER615 = 0x7338,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER616 = 0x7340,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER617 = 0x7348,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER618 = 0x7350,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER619 = 0x7358,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER620 = 0x7360,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER621 = 0x7368,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER622 = 0x7370,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER623 = 0x7378,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER624 = 0x7380,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER625 = 0x7388,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER626 = 0x7390,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER627 = 0x7398,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER628 = 0x73a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER629 = 0x73a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER630 = 0x73b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER631 = 0x73b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER632 = 0x73c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER633 = 0x73c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER634 = 0x73d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER635 = 0x73d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER636 = 0x73e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER637 = 0x73e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER638 = 0x73f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER639 = 0x73f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER640 = 0x7400,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER641 = 0x7408,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER642 = 0x7410,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER643 = 0x7418,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER644 = 0x7420,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER645 = 0x7428,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER646 = 0x7430,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER647 = 0x7438,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER648 = 0x7440,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER649 = 0x7448,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER650 = 0x7450,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER651 = 0x7458,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER652 = 0x7460,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER653 = 0x7468,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER654 = 0x7470,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER655 = 0x7478,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER656 = 0x7480,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER657 = 0x7488,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER658 = 0x7490,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER659 = 0x7498,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER660 = 0x74a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER661 = 0x74a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER662 = 0x74b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER663 = 0x74b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER664 = 0x74c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER665 = 0x74c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER666 = 0x74d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER667 = 0x74d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER668 = 0x74e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER669 = 0x74e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER670 = 0x74f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER671 = 0x74f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER672 = 0x7500,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER673 = 0x7508,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER674 = 0x7510,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER675 = 0x7518,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER676 = 0x7520,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER677 = 0x7528,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER678 = 0x7530,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER679 = 0x7538,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER680 = 0x7540,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER681 = 0x7548,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER682 = 0x7550,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER683 = 0x7558,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER684 = 0x7560,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER685 = 0x7568,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER686 = 0x7570,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER687 = 0x7578,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER688 = 0x7580,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER689 = 0x7588,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER690 = 0x7590,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER691 = 0x7598,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER692 = 0x75a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER693 = 0x75a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER694 = 0x75b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER695 = 0x75b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER696 = 0x75c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER697 = 0x75c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER698 = 0x75d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER699 = 0x75d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER700 = 0x75e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER701 = 0x75e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER702 = 0x75f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER703 = 0x75f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER704 = 0x7600,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER705 = 0x7608,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER706 = 0x7610,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER707 = 0x7618,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER708 = 0x7620,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER709 = 0x7628,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER710 = 0x7630,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER711 = 0x7638,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER712 = 0x7640,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER713 = 0x7648,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER714 = 0x7650,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER715 = 0x7658,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER716 = 0x7660,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER717 = 0x7668,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER718 = 0x7670,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER719 = 0x7678,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER720 = 0x7680,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER721 = 0x7688,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER722 = 0x7690,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER723 = 0x7698,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER724 = 0x76a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER725 = 0x76a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER726 = 0x76b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER727 = 0x76b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER728 = 0x76c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER729 = 0x76c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER730 = 0x76d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER731 = 0x76d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER732 = 0x76e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER733 = 0x76e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER734 = 0x76f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER735 = 0x76f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER736 = 0x7700,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER737 = 0x7708,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER738 = 0x7710,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER739 = 0x7718,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER740 = 0x7720,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER741 = 0x7728,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER742 = 0x7730,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER743 = 0x7738,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER744 = 0x7740,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER745 = 0x7748,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER746 = 0x7750,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER747 = 0x7758,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER748 = 0x7760,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER749 = 0x7768,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER750 = 0x7770,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER751 = 0x7778,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER752 = 0x7780,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER753 = 0x7788,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER754 = 0x7790,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER755 = 0x7798,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER756 = 0x77a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER757 = 0x77a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER758 = 0x77b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER759 = 0x77b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER760 = 0x77c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER761 = 0x77c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER762 = 0x77d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER763 = 0x77d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER764 = 0x77e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER765 = 0x77e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER766 = 0x77f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER767 = 0x77f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER768 = 0x7800,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER769 = 0x7808,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER770 = 0x7810,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER771 = 0x7818,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER772 = 0x7820,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER773 = 0x7828,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER774 = 0x7830,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER775 = 0x7838,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER776 = 0x7840,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER777 = 0x7848,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER778 = 0x7850,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER779 = 0x7858,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER780 = 0x7860,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER781 = 0x7868,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER782 = 0x7870,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER783 = 0x7878,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER784 = 0x7880,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER785 = 0x7888,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER786 = 0x7890,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER787 = 0x7898,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER788 = 0x78a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER789 = 0x78a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER790 = 0x78b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER791 = 0x78b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER792 = 0x78c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER793 = 0x78c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER794 = 0x78d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER795 = 0x78d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER796 = 0x78e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER797 = 0x78e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER798 = 0x78f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER799 = 0x78f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER800 = 0x7900,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER801 = 0x7908,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER802 = 0x7910,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER803 = 0x7918,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER804 = 0x7920,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER805 = 0x7928,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER806 = 0x7930,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER807 = 0x7938,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER808 = 0x7940,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER809 = 0x7948,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER810 = 0x7950,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER811 = 0x7958,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER812 = 0x7960,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER813 = 0x7968,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER814 = 0x7970,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER815 = 0x7978,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER816 = 0x7980,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER817 = 0x7988,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER818 = 0x7990,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER819 = 0x7998,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER820 = 0x79a0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER821 = 0x79a8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER822 = 0x79b0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER823 = 0x79b8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER824 = 0x79c0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER825 = 0x79c8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER826 = 0x79d0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER827 = 0x79d8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER828 = 0x79e0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER829 = 0x79e8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER830 = 0x79f0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER831 = 0x79f8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER832 = 0x7a00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER833 = 0x7a08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER834 = 0x7a10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER835 = 0x7a18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER836 = 0x7a20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER837 = 0x7a28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER838 = 0x7a30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER839 = 0x7a38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER840 = 0x7a40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER841 = 0x7a48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER842 = 0x7a50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER843 = 0x7a58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER844 = 0x7a60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER845 = 0x7a68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER846 = 0x7a70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER847 = 0x7a78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER848 = 0x7a80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER849 = 0x7a88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER850 = 0x7a90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER851 = 0x7a98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER852 = 0x7aa0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER853 = 0x7aa8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER854 = 0x7ab0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER855 = 0x7ab8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER856 = 0x7ac0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER857 = 0x7ac8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER858 = 0x7ad0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER859 = 0x7ad8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER860 = 0x7ae0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER861 = 0x7ae8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER862 = 0x7af0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER863 = 0x7af8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER864 = 0x7b00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER865 = 0x7b08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER866 = 0x7b10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER867 = 0x7b18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER868 = 0x7b20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER869 = 0x7b28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER870 = 0x7b30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER871 = 0x7b38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER872 = 0x7b40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER873 = 0x7b48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER874 = 0x7b50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER875 = 0x7b58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER876 = 0x7b60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER877 = 0x7b68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER878 = 0x7b70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER879 = 0x7b78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER880 = 0x7b80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER881 = 0x7b88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER882 = 0x7b90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER883 = 0x7b98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER884 = 0x7ba0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER885 = 0x7ba8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER886 = 0x7bb0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER887 = 0x7bb8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER888 = 0x7bc0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER889 = 0x7bc8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER890 = 0x7bd0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER891 = 0x7bd8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER892 = 0x7be0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER893 = 0x7be8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER894 = 0x7bf0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER895 = 0x7bf8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER896 = 0x7c00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER897 = 0x7c08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER898 = 0x7c10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER899 = 0x7c18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER900 = 0x7c20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER901 = 0x7c28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER902 = 0x7c30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER903 = 0x7c38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER904 = 0x7c40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER905 = 0x7c48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER906 = 0x7c50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER907 = 0x7c58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER908 = 0x7c60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER909 = 0x7c68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER910 = 0x7c70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER911 = 0x7c78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER912 = 0x7c80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER913 = 0x7c88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER914 = 0x7c90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER915 = 0x7c98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER916 = 0x7ca0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER917 = 0x7ca8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER918 = 0x7cb0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER919 = 0x7cb8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER920 = 0x7cc0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER921 = 0x7cc8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER922 = 0x7cd0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER923 = 0x7cd8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER924 = 0x7ce0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER925 = 0x7ce8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER926 = 0x7cf0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER927 = 0x7cf8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER928 = 0x7d00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER929 = 0x7d08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER930 = 0x7d10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER931 = 0x7d18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER932 = 0x7d20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER933 = 0x7d28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER934 = 0x7d30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER935 = 0x7d38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER936 = 0x7d40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER937 = 0x7d48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER938 = 0x7d50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER939 = 0x7d58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER940 = 0x7d60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER941 = 0x7d68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER942 = 0x7d70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER943 = 0x7d78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER944 = 0x7d80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER945 = 0x7d88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER946 = 0x7d90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER947 = 0x7d98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER948 = 0x7da0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER949 = 0x7da8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER950 = 0x7db0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER951 = 0x7db8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER952 = 0x7dc0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER953 = 0x7dc8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER954 = 0x7dd0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER955 = 0x7dd8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER956 = 0x7de0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER957 = 0x7de8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER958 = 0x7df0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER959 = 0x7df8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER960 = 0x7e00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER961 = 0x7e08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER962 = 0x7e10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER963 = 0x7e18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER964 = 0x7e20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER965 = 0x7e28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER966 = 0x7e30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER967 = 0x7e38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER968 = 0x7e40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER969 = 0x7e48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER970 = 0x7e50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER971 = 0x7e58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER972 = 0x7e60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER973 = 0x7e68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER974 = 0x7e70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER975 = 0x7e78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER976 = 0x7e80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER977 = 0x7e88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER978 = 0x7e90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER979 = 0x7e98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER980 = 0x7ea0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER981 = 0x7ea8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER982 = 0x7eb0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER983 = 0x7eb8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER984 = 0x7ec0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER985 = 0x7ec8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER986 = 0x7ed0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER987 = 0x7ed8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER988 = 0x7ee0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER989 = 0x7ee8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER990 = 0x7ef0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER991 = 0x7ef8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER992 = 0x7f00,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER993 = 0x7f08,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER994 = 0x7f10,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER995 = 0x7f18,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER996 = 0x7f20,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER997 = 0x7f28,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER998 = 0x7f30,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER999 = 0x7f38,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1000 = 0x7f40,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1001 = 0x7f48,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1002 = 0x7f50,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1003 = 0x7f58,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1004 = 0x7f60,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1005 = 0x7f68,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1006 = 0x7f70,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1007 = 0x7f78,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1008 = 0x7f80,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1009 = 0x7f88,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1010 = 0x7f90,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1011 = 0x7f98,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1012 = 0x7fa0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1013 = 0x7fa8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1014 = 0x7fb0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1015 = 0x7fb8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1016 = 0x7fc0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1017 = 0x7fc8,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1018 = 0x7fd0,
    HV_GIC_DISTRIBUTOR_REG_GICD_IROUTER1019 = 0x7fd8,

    HV_GIC_DISTRIBUTOR_REG_GICD_PIDR2 = 0xffe8,
);

/*!
 @abstract Type of an ARM GIC redistributor register.
*/
OS_ENUM(hv_gic_redistributor_reg, uint32_t,
    HV_GIC_REDISTRIBUTOR_REG_GICR_TYPER = 0x0008,
    HV_GIC_REDISTRIBUTOR_REG_GICR_PIDR2 = 0xffe8,

    HV_GIC_REDISTRIBUTOR_REG_GICR_IGROUPR0 = 0x10080,
    HV_GIC_REDISTRIBUTOR_REG_GICR_ISENABLER0 = 0x10100,
    HV_GIC_REDISTRIBUTOR_REG_GICR_ICENABLER0 = 0x10180,
    HV_GIC_REDISTRIBUTOR_REG_GICR_ISPENDR0 = 0x10200,
    HV_GIC_REDISTRIBUTOR_REG_GICR_ICPENDR0 = 0x10280,
    HV_GIC_REDISTRIBUTOR_REG_GICR_ISACTIVER0 = 0x10300,
    HV_GIC_REDISTRIBUTOR_REG_GICR_ICACTIVER0 = 0x10380,

    HV_GIC_REDISTRIBUTOR_REG_GICR_IPRIORITYR0 = 0x10400,
    HV_GIC_REDISTRIBUTOR_REG_GICR_IPRIORITYR1 = 0x10404,
    HV_GIC_REDISTRIBUTOR_REG_GICR_IPRIORITYR2 = 0x10408,
    HV_GIC_REDISTRIBUTOR_REG_GICR_IPRIORITYR3 = 0x1040c,
    HV_GIC_REDISTRIBUTOR_REG_GICR_IPRIORITYR4 = 0x10410,
    HV_GIC_REDISTRIBUTOR_REG_GICR_IPRIORITYR5 = 0x10414,
    HV_GIC_REDISTRIBUTOR_REG_GICR_IPRIORITYR6 = 0x10418,
    HV_GIC_REDISTRIBUTOR_REG_GICR_IPRIORITYR7 = 0x1041c,

    HV_GIC_REDISTRIBUTOR_REG_GICR_ICFGR0 = 0x10c00,
    HV_GIC_REDISTRIBUTOR_REG_GICR_ICFGR1 = 0x10c04,
);

/*!
 @abstract Type of an ARM GIC ICC system control register.
*/
API_AVAILABLE(macos(15.0))
OS_ENUM(hv_gic_icc_reg, uint16_t,
    HV_GIC_ICC_REG_PMR_EL1 = 0xc230,
    HV_GIC_ICC_REG_BPR0_EL1 = 0xc643,
    HV_GIC_ICC_REG_AP0R0_EL1 = 0xc644,
    HV_GIC_ICC_REG_AP1R0_EL1 = 0xc648,
    HV_GIC_ICC_REG_RPR_EL1 = 0xc65b,
    HV_GIC_ICC_REG_BPR1_EL1 = 0xc663,
    HV_GIC_ICC_REG_CTLR_EL1 = 0xc664,
    HV_GIC_ICC_REG_SRE_EL1 = 0xc665,
    HV_GIC_ICC_REG_IGRPEN0_EL1 = 0xc666,
    HV_GIC_ICC_REG_IGRPEN1_EL1 = 0xc667,
    HV_GIC_ICC_REG_SRE_EL2 = 0xe64d,
);

/*!
 @abstract Type of an ARM GIC virtualization control system register.
*/
API_AVAILABLE(macos(15.0))
OS_ENUM(hv_gic_ich_reg, uint16_t,
    HV_GIC_ICH_REG_AP0R0_EL2 = 0xe640,
    HV_GIC_ICH_REG_AP1R0_EL2 = 0xe648,
    HV_GIC_ICH_REG_HCR_EL2 = 0xe658,
    HV_GIC_ICH_REG_VTR_EL2 = 0xe659,
    HV_GIC_ICH_REG_MISR_EL2 = 0xe65a,
    HV_GIC_ICH_REG_EISR_EL2 = 0xe65b,
    HV_GIC_ICH_REG_ELRSR_EL2 = 0xe65d,
    HV_GIC_ICH_REG_VMCR_EL2 = 0xe65f,
    HV_GIC_ICH_REG_LR0_EL2 = 0xe660,
    HV_GIC_ICH_REG_LR1_EL2 = 0xe661,
    HV_GIC_ICH_REG_LR2_EL2 = 0xe662,
    HV_GIC_ICH_REG_LR3_EL2 = 0xe663,
    HV_GIC_ICH_REG_LR4_EL2 = 0xe664,
    HV_GIC_ICH_REG_LR5_EL2 = 0xe665,
    HV_GIC_ICH_REG_LR6_EL2 = 0xe666,
    HV_GIC_ICH_REG_LR7_EL2 = 0xe667,
    HV_GIC_ICH_REG_LR8_EL2 = 0xe668,
    HV_GIC_ICH_REG_LR9_EL2 = 0xe669,
    HV_GIC_ICH_REG_LR10_EL2 = 0xe66a,
    HV_GIC_ICH_REG_LR11_EL2 = 0xe66b,
    HV_GIC_ICH_REG_LR12_EL2 = 0xe66c,
    HV_GIC_ICH_REG_LR13_EL2 = 0xe66d,
    HV_GIC_ICH_REG_LR14_EL2 = 0xe66e,
    HV_GIC_ICH_REG_LR15_EL2 = 0xe66f,
);

/*!
 @abstract Type of an ARM GIC ICV system control register.
*/
API_AVAILABLE(macos(15.0))
OS_ENUM(hv_gic_icv_reg, uint16_t,
    HV_GIC_ICV_REG_PMR_EL1 = 0xc230,
    HV_GIC_ICV_REG_BPR0_EL1 = 0xc643,
    HV_GIC_ICV_REG_AP0R0_EL1 = 0xc644,
    HV_GIC_ICV_REG_AP1R0_EL1 = 0xc648,
    HV_GIC_ICV_REG_RPR_EL1 = 0xc65b,
    HV_GIC_ICV_REG_BPR1_EL1 = 0xc663,
    HV_GIC_ICV_REG_CTLR_EL1 = 0xc664,
    HV_GIC_ICV_REG_SRE_EL1 = 0xc665,
    HV_GIC_ICV_REG_IGRPEN0_EL1 = 0xc666,
    HV_GIC_ICV_REG_IGRPEN1_EL1 = 0xc667,
);

/*!
 @abstract Type of an ARM GIC Distributor message based interrupt register.
*/
OS_ENUM(hv_gic_msi_reg, uint16_t,
    HV_GIC_REG_GICM_TYPER = 0x0008,
    HV_GIC_REG_GICM_SET_SPI_NSR = 0x0040,
);

__END_DECLS

OS_ASSUME_NONNULL_END

#endif
