//
//  HKLiveWorkoutBuilder.h
//  HealthKit
//
//  Copyright © 2018-2025 Apple. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <HealthKit/HKDefines.h>
#import <HealthKit/HKLiveWorkoutDataSource.h>
#import <HealthKit/HKObjectType.h>
#import <HealthKit/HKWorkoutBuilder.h>

NS_ASSUME_NONNULL_BEGIN

@class HKLiveWorkoutBuilder;
@class HKWorkoutSession;

HK_EXTERN
API_AVAILABLE(ios(26.0), watchos(5.0)) API_UNAVAILABLE(visionos)
@protocol HKLiveWorkoutBuilderDelegate <NSObject>

/*!
 @method        workoutBuilder:didCollectDataOfTypes:
 @abstract      Called every time new samples are added to the workout builder.
 @discussion    With new samples added, statistics for the collectedTypes may have changed and should be read again
 
 @param         workoutBuilder    The workout builder to which samples were added.
 @param         collectedTypes    The sample types that were added.
 */
- (void)workoutBuilder:(HKLiveWorkoutBuilder *)workoutBuilder didCollectDataOfTypes:(NSSet<HKSampleType *> *)collectedTypes;

/*!
 @method        workoutBuilderDidCollectEvent:
 @abstract      Called every time a new event is added to the workout builder.
 
 @param         workoutBuilder    The workout builder to which an event was added.
 */
- (void)workoutBuilderDidCollectEvent:(HKLiveWorkoutBuilder *)workoutBuilder;

@optional

/*!
 @method        workoutBuilder:didBeginActivity:
 @abstract      Called every time new activity is started and added to the workout builder.
 
 @param         workoutBuilder      The workout builder to which the activity was added to.
 @param         workoutActivity     The activity that was added.
 */
- (void)workoutBuilder:(HKLiveWorkoutBuilder *)workoutBuilder didBeginActivity:(HKWorkoutActivity *)workoutActivity API_AVAILABLE(ios(26.0), watchos(9.0)) API_UNAVAILABLE(visionos);

/*!
 @method        workoutBuilder:didEndActivity:
 @abstract      Called every time the end date is set on an activity in the workout builder.
 
 @param         workoutBuilder      The workout builder to which the activity belongs to
 @param         workoutActivity     The activity that was ended.
 */
- (void)workoutBuilder:(HKLiveWorkoutBuilder *)workoutBuilder didEndActivity:(HKWorkoutActivity *)workoutActivity API_AVAILABLE(ios(26.0), watchos(9.0)) API_UNAVAILABLE(visionos);

@end

HK_EXTERN
NS_SWIFT_SENDABLE
API_AVAILABLE(ios(26.0), watchos(5.0)) API_UNAVAILABLE(visionos)
@interface HKLiveWorkoutBuilder : HKWorkoutBuilder

- (instancetype)initWithHealthStore:(HKHealthStore *)healthStore
                      configuration:(HKWorkoutConfiguration *)configuration
                             device:(nullable HKDevice *)device NS_UNAVAILABLE;

/*!
 @property      delegate
 @abstract      The delegate object which will be notified of changes to collected data and elapsed workout time.
 */
@property (weak) id<HKLiveWorkoutBuilderDelegate> delegate;

/*!
 @property      workoutSession
 @abstract      The workout session with which this builder is associated.
 */
@property (weak, readonly) HKWorkoutSession *workoutSession;

/*!
 @property      shouldCollectWorkoutEvents
 @abstract      Whether or not the builder should automatically incorporate events generated by the session.
 @discussion    YES if the builder should automatically incorporate events from the associated session. If NO, the client
                is expected to add relevant events directly to the builder and those generated by the session will be
                ignored. Defaults to YES. The result of changing this value after a call to
                beginCollectionWithStartDate:completion: is undefined.
 */
@property (assign) BOOL shouldCollectWorkoutEvents;

/*!
 @property      dataSource
 @abstract      The data source that will provide data to this workout builder.
 */
@property (strong, nullable) HKLiveWorkoutDataSource *dataSource;

/*!
 @property      elapsedTime
 @abstract      The elapsed duration of the workout as constructed thus far, considering paused time. When the builder
                is reading events from an active session, it is possible for this value to increase or decrease;
                when this occurs, the workoutBuilderDidCollectEvent: delegate method will be called.
 */
@property (assign, readonly) NSTimeInterval elapsedTime;

/*!
 @property      currentWorkoutActivity
 @abstract      The latest activity that has been added to this builder.
 @discussion    When an activity is in progress it will be returned by this property. The end date of this activity will always
                be nil. When the activity is ended, the property would be set to nil until a new activity begins.
 */
@property (copy, readonly, nullable) HKWorkoutActivity *currentWorkoutActivity API_AVAILABLE(ios(26.0), watchos(9.0)) API_UNAVAILABLE(visionos);

@end

NS_ASSUME_NONNULL_END
