//
//  ASAccountAuthenticationModificationViewController.h
//  AuthenticationServices Framework
//
//  Copyright © 2020 Apple Inc. All rights reserved.
//

#import <AuthenticationServices/ASAccountAuthenticationModificationExtensionContext.h>
#import <AuthenticationServices/ASCredentialServiceIdentifier.h>
#import <AuthenticationServices/ASFoundation.h>
#import <AuthenticationServices/ASPasswordCredential.h>
#import <Foundation/Foundation.h>

NS_ASSUME_NONNULL_BEGIN

AS_EXTERN API_AVAILABLE(ios(14.0)) API_UNAVAILABLE(macos, watchos, tvos)
@interface ASAccountAuthenticationModificationViewController : ASViewController

@property (nonatomic, readonly, strong) ASAccountAuthenticationModificationExtensionContext *extensionContext;

/*! @abstract This method will be called when a user initiates a Sign in with Apple upgrade request.
 @param serviceIdentifier The service identifier of the credential the user wishes to upgrade.
 @param existingCredential The current password-based credential of the account for the upgrade.
 @param userInfo For upgrades invoked within the extension's containing app, any
 information the app wants to pass to the extension, most likely to help authorize the upgrade.
 @discussion This method will be called to begin a Sign in with Apple upgrade request and will not
 present any extension UI. If extension UI needs to be shown, this initial request should be canceled
 with the ASExtensionErrorCodeUserCanceled error code. The existingCredential parameter will have
 an empty password for in-app upgrades. The extension should check if a user is already logged in by
 checking a data container shared with its containing app.
*/
- (void)convertAccountToSignInWithAppleWithoutUserInteractionForServiceIdentifier:(ASCredentialServiceIdentifier *)serviceIdentifier existingCredential:(ASPasswordCredential *)existingCredential userInfo:(nullable NSDictionary *)userInfo NS_SWIFT_NAME(convertAccountToSignInWithAppleWithoutUserInteraction(for:existingCredential:userInfo:));

/*! @abstract This method will be called when a Sign in with Apple upgrade request requires user interaction.
 @param serviceIdentifier The service identifier of the credential the user wishes to upgrade.
 @param existingCredential The current password-based credential of the account for the upgrade.
 @param userInfo For upgrades invoked within the extension's containing app, any
 information the app wants to pass to the extension, most likely to help authorize the upgrade.
 @discussion This method will be called when the initial request is canceled with the
 ASExtensionErrorCodeUserCanceled error code. The implementation of this method should do any
 work necessary to prepare to present the extension UI. Once it finishes, the system will present the
 extension UI. The existingCredential parameter will have an empty password for in-app upgrades. The
 extension should check if a user is already logged in by checking state in a data container shared
 with its containing app.
*/
- (void)prepareInterfaceToConvertAccountToSignInWithAppleForServiceIdentifier:(ASCredentialServiceIdentifier *)serviceIdentifier existingCredential:(ASPasswordCredential *)existingCredential userInfo:(nullable NSDictionary *)userInfo NS_SWIFT_NAME(prepareInterfaceToConvertAccountToSignInWithApple(for:existingCredential:userInfo:));

/*! @abstract This method will be called when a user initiates a strong password upgrade request.
 @param serviceIdentifier The service identifier of the credential the user wishes to upgrade.
 @param existingCredential The current password-based credential of the account for the upgrade.
 @param newPassword A strong password generated by the system to be used for the upgrade.
 @param userInfo For upgrades invoked within the extension's containing app, any
 information the app wants to pass to the extension, most likely to help authorize the upgrade.
 @discussion Password rules for generating the newPassword parameter can be specified
 with the ASAccountAuthenticationModificationSupportsUpgradeToSignInWithApple key in the
 extension's Info.plist. This method will be called to begin a strong password upgrade request and
 will not present any extension UI. If extension UI needs to be shown, this initial request should be
 canceled with the ASExtensionErrorCodeUserCanceled error code. The existingCredential parameter
 will have an empty password for in-app upgrades. The extension should check if a user is already
 logged in by checking a data container shared with its containing app.
*/
- (void)changePasswordWithoutUserInteractionForServiceIdentifier:(ASCredentialServiceIdentifier *)serviceIdentifier existingCredential:(ASPasswordCredential *)existingCredential newPassword:(NSString *)newPassword userInfo:(nullable NSDictionary *)userInfo NS_SWIFT_NAME(changePasswordWithoutUserInteraction(for:existingCredential:newPassword:userInfo:));

/*! @abstract This method will be called when a strong password upgrade request requires user interaction.
 @param serviceIdentifier The service identifier of the credential the user wishes to upgrade.
 @param existingCredential The current password-based credential of the account for the upgrade.
 @param newPassword A strong password generated by the system to be used for the upgrade.
 @param userInfo For upgrades invoked within the extension's containing app, any
 information the app wants to pass to the extension, most likely to help authorize the upgrade.
 @discussion This method will be called when the initial request is canceled with the
 ASExtensionErrorCodeUserCanceled error code. The implementation of this method should do
 any work necessary to prepare to present the extension UI. Once it finishes, the system will present
 the extension UI. Password rules for generating the newPassword parameter can be specified
 with the ASAccountAuthenticationModificationSupportsUpgradeToSignInWithApple key in the
 extension's Info.plist. The existingCredential parameter will have an empty password for in-app upgrades.
 The extension should check if a user is already logged in by checking state in a data container shared
 with its containing app.
*/
- (void)prepareInterfaceToChangePasswordForServiceIdentifier:(ASCredentialServiceIdentifier *)serviceIdentifier existingCredential:(ASPasswordCredential *)existingCredential newPassword:(NSString *)newPassword userInfo:(nullable NSDictionary *)userInfo NS_SWIFT_NAME(prepareInterfaceToChangePassword(for:existingCredential:newPassword:userInfo:));

/*! @abstract This method will be called when the user taps the system-provided "Cancel" button shown above extension UI.
 @discussion Extension UI will be dismissed as soon as this method is called. The default
 implementation simply cancels the request with the ASExtensionErrorCodeUserCanceled error code.
 Override this method to do any cleanup work before the request is canceled. The implementation of this
 method should still cancel the request with the ASExtensionErrorCodeUserCanceled error code once the
 cleanup work is completed. If the request is not canceled in 10 seconds, the system will cancel the request.
*/
- (void)cancelRequest;

@end

NS_ASSUME_NONNULL_END
